#ifndef CASM_irreps_SimpleOrbit
#define CASM_irreps_SimpleOrbit

#include "casm/container/multivector.hh"

namespace CASM {
namespace irreps {

/// Generate an orbit of unique Element generated by a group
///
/// Use of SimpleOrbit requires implementation of a SymCompareType that
/// describes how to apply group elements to the orbit elements and how to
/// compare orbit elements. The struct `VectorSymCompare` in `irreps/
/// Symmetrizer.hh` provides an example implementation.
///
/// \ingroup OrbitGeneration
///
template <typename _SymCompareType>
class SimpleOrbit {
 public:
  using size_type = unsigned int;
  using SymCompareType = _SymCompareType;
  using Element = typename _SymCompareType::Element;
  using InvariantsType = typename _SymCompareType::InvariantsType;
  using const_iterator = typename std::vector<Element>::const_iterator;

  /// Construct a SimpleOrbit using a range of symmetry representations
  template <typename GroupIterator>
  SimpleOrbit(typename SymCompareType::Element const &_generating_element,
              GroupIterator _group_begin, GroupIterator _group_end,
              SymCompareType const &_sym_compare);

  const_iterator begin() const { return m_element.cbegin(); }

  const_iterator end() const { return m_element.cend(); }

  const_iterator cbegin() const { return m_element.cbegin(); }

  const_iterator cend() const { return m_element.cend(); }

  size_type size() const { return m_element.size(); }

  /// \brief Identical to element(0)
  Element const &prototype() const { return m_element[0]; }

  /// \brief Return Element at index, without bounds checking
  ///
  /// - May not be prepared
  Element const &operator[](size_type index) const { return element(index); }

  /// \brief Equivalent to operator[](size_type index) const
  ///
  /// - May not be prepared
  Element const &element(size_type index) const { return m_element[index]; }

  /// \brief const Access vector of Element
  ///
  /// - May not be prepared
  std::vector<Element> const &elements() const { return m_element; }

  /// \brief Return the SymCompare functor reference
  ///
  /// - implements symmetry properties of this orbit
  SymCompareType const &sym_compare() const { return m_sym_compare; }

  InvariantsType const &invariants() const { return m_invariants; }

  /// Compare orbits, using SymCompareType::inter_orbit_compare
  bool operator<(const SimpleOrbit &B) const;

 private:
  SymCompareType m_sym_compare;
  InvariantsType m_invariants;
  std::vector<Element> m_element;
};

}  // namespace irreps
}  // namespace CASM

#endif
