#ifndef CASM_group_subgroups
#define CASM_group_subgroups

#include "casm/configuration/group/Group.hh"
#include "casm/configuration/group/definitions.hh"

namespace CASM {
namespace group {

typedef std::set<Index> SubgroupIndices;
typedef std::set<SubgroupIndices> SubgroupOrbit;

/// \brief Return all cyclic subgroups
template <typename ElementType>
std::set<SubgroupOrbit> make_cyclic_subgroups(Group<ElementType> const &group);

/// \brief Return all subgroups
template <typename ElementType>
std::set<SubgroupOrbit> make_all_subgroups(Group<ElementType> const &group);

/// \brief Make the invariant subgroup for each orbit element, as
///     indices of group elements
template <typename GroupElementType>
std::vector<SubgroupIndices> make_invariant_subgroups(
    std::vector<std::vector<Index>> const &equivalence_map,
    Group<GroupElementType> const &group);

}  // namespace group
}  // namespace CASM

// --- Implementation ---

#include <numeric>

namespace CASM {
namespace group {

namespace subgroups_impl {

typedef std::set<Index> CosetIndices;

/// \brief Return the unique left cosets of a subgroup
///
/// - If subgroup B of group G contains elements: (E, B1, B2, …, Bg),
/// the "left coset” of X is (X*E, X*B1, X*B2, …, X*Bg),
/// where X is an element of G.
/// - A coset need not be a subgroup.
/// - If X is an element of B, then the coset will be a subgroup of B.
/// - Two left cosets of a given subgroup either contain exactly the same
/// elements, or have no elements in common.
template <typename ElementType>
std::set<CosetIndices> _make_left_cosets(
    Group<ElementType> const &group, SubgroupIndices const &subgroup_indices) {
  std::set<CosetIndices> left_cosets;

  // each group element is only included in one coset
  std::vector<bool> check(group.element.size(), false);
  Index product_index;
  Index group_element_index = 0;
  while (group_element_index < group.element.size()) {
    if (check[group_element_index]) {
      ++group_element_index;
      continue;
    }
    CosetIndices left_coset;
    for (auto subgroup_element_index : subgroup_indices) {
      product_index = group.mult(group_element_index, subgroup_element_index);
      left_coset.insert(product_index);
      check[product_index] = true;
    }
    left_cosets.insert(left_coset);
    ++group_element_index;
  }
  return left_cosets;
}

template <typename ElementType>
SubgroupOrbit _make_subgroup_orbit(Group<ElementType> const &group,
                                   SubgroupIndices const &subgroup) {
  SubgroupOrbit orbit;
  std::set<subgroups_impl::CosetIndices> left_cosets =
      subgroups_impl::_make_left_cosets(group, subgroup);
  for (auto const &coset : left_cosets) {
    Index X_index = *coset.begin();
    Index X_inv_index = group.inv(X_index);
    SubgroupIndices equiv_subgroup;
    for (auto const &A_index : subgroup) {
      equiv_subgroup.insert(
          group.mult(X_index, group.mult(A_index, X_inv_index)));
    }
    orbit.insert(equiv_subgroup);
  }
  return orbit;
}

inline std::function<bool(SubgroupIndices const &)> _make_subgroup_count(
    std::set<SubgroupOrbit> const &subgroups) {
  return [&](SubgroupIndices const &subgroup) {
    for (auto const &orbit : subgroups) {
      if (orbit.count(subgroup)) {
        return true;
      }
    }
    return false;
  };
}

template <typename ElementType>
std::function<void(SubgroupIndices &)> _make_close_subgroup(
    Group<ElementType> const &group) {
  return [&](SubgroupIndices &subgroup) {
    std::vector<Index> vgroup(subgroup.begin(), subgroup.end());
    for (Index i = 0; i < vgroup.size(); i++) {
      for (Index j = 0; j < vgroup.size(); j++) {
        Index product_index = group.mult(vgroup[i], vgroup[j]);
        if (subgroup.insert(product_index).second) {
          vgroup.push_back(product_index);
        }
      }
    }
  };
}

}  // namespace subgroups_impl

/// \brief Return all cyclic subgroups
///
/// - A cyclic subgroup, A, is the subgroup generated by repeated multiplication
/// of a single element, i.e. A={a, a^2, a^3, ..., a^k}, where a^k=E, the
/// identity element
/// - An equivalent subgroup is {X*a*X^-1, X*(a^2)*X^-1, X*(a^3)*X^-1, ..., E},
/// where X is an element in a coset of A
/// - The orbit of a cyclic subgroup is all distinct equivalent subgroups
///
/// \param group The group to find cyclic subgroups of
/// \returns A set of orbits of cyclic subgroups
///
template <typename ElementType>
std::set<SubgroupOrbit> make_cyclic_subgroups(Group<ElementType> const &group) {
  using namespace subgroups_impl;

  std::set<SubgroupOrbit> cyclic_subgroups;
  Index group_element_index = 0;
  while (group_element_index < group.element.size()) {
    // Make cyclic subgroup of element `group_element_index`
    SubgroupIndices cyclic_subgroup;
    cyclic_subgroup.insert(group_element_index);
    Index product_index = group_element_index;
    while (product_index != 0) {
      product_index = group.mult(group_element_index, product_index);
      cyclic_subgroup.insert(product_index);
    }

    // Make orbit of subgroups equivalent to `cyclic_subgroup` && Insert orbit
    cyclic_subgroups.insert(_make_subgroup_orbit(group, cyclic_subgroup));

    ++group_element_index;
  }
  return cyclic_subgroups;
}

/// \brief Return all subgroups
///
/// Method:
/// - Start with m_subgroups = m_small_subgroups, then add new subgroups by
/// finding the closure of a union of a large_group and a small_group.
/// - If the the new large_group is unique, add it as a large_group.
/// - Repeat for all (large_group, small_group) pairs, until no new
/// m_subgroups are found.
///
/// Note:
/// - This is probably not the fastest algorithm, but it is complete
///
/// \param group The group to find subgroups of
/// \returns A set of orbits of all subgroups
///
template <typename ElementType>
std::set<SubgroupOrbit> make_all_subgroups(Group<ElementType> const &group) {
  using namespace subgroups_impl;
  std::set<SubgroupOrbit> small_subgroups = make_cyclic_subgroups(group);
  std::set<SubgroupOrbit> all_subgroups = small_subgroups;

  // functor to close incomplete subgroup
  auto _close_subgroup = _make_close_subgroup(group);

  // functor to find if any orbit contains a particular subgroup
  auto _subgroup_count = _make_subgroup_count(all_subgroups);

  auto all_subgroups_it = all_subgroups.begin();
  while (all_subgroups_it != all_subgroups.end()) {
    for (auto const &small_subgroups_orbit : small_subgroups) {
      for (auto const &small_subgroups_equiv : small_subgroups_orbit) {
        // Combine an existing subgroup and a small (cyclic) subgroup
        SubgroupIndices subgroup = *(all_subgroups_it->begin());
        Index init_size = subgroup.size();
        subgroup.insert(small_subgroups_equiv.begin(),
                        small_subgroups_equiv.end());
        if (subgroup.size() == init_size) continue;

        // Find group closure
        _close_subgroup(subgroup);

        // If subgroup already exists in all_subgroups, continue
        if (_subgroup_count(subgroup)) continue;

        // Else, make orbit and insert
        all_subgroups.insert(_make_subgroup_orbit(group, subgroup));
      }
    }
    ++all_subgroups_it;
  }
  return all_subgroups;
}

/// \brief Make the invariant subgroup for each orbit element, as
///     indices of group elements
///
/// \param equivalence_map The indices equivalence_map[i] are the
///     indices of the group elements which map orbit element 0 onto
///     orbit element i
/// \param group The group used to generate the orbit
///
/// \returns invariant_subgroups, The indices invariant_subgroups[i] are
///     the indices of the group elements which leave orbit element i
///     invariant.
template <typename GroupElementType>
std::vector<SubgroupIndices> make_invariant_subgroups(
    std::vector<std::vector<Index>> const &equivalence_map,
    Group<GroupElementType> const &group) {
  /// The first row of equivalence_map is the invariant subgroup
  /// of the first element in the orbit
  ///
  /// Invariant subgroups of subsequent orbit elements are constructed
  /// using:
  ///   inv_subgrp(i) = eq_map(i,0) * eq_map(0,j) * inverse(eq_map(i,0)), for
  ///   all j
  ///
  /// inverse(eq_map(i,0)): transforms orbit element i back to orbit element 0
  /// eq_map(0,j): invariant transformation of orbit element 0
  /// eq_map(i,0): transforms orbit element 0 to orbit element i
  ///
  std::vector<SubgroupIndices> invariant_subgroups;
  if (!equivalence_map.size()) {
    return invariant_subgroups;
  }

  // first row of equivalence map is an invariant subgroup
  {
    SubgroupIndices subgroup;
    for (Index e_0j : equivalence_map[0]) {
      subgroup.insert(e_0j);
    }
    invariant_subgroups.emplace_back(std::move(subgroup));
  }

  // first column can be used to generate others
  for (Index i = 1; i < equivalence_map.size(); ++i) {
    if (!equivalence_map[i].size()) {
      throw std::runtime_error(
          "Error in make_invariant_subgroups: failed due to empty row in "
          "equivalence_map");
    }
    SubgroupIndices subgroup;
    Index e_i0 = equivalence_map[i][0];
    for (Index e_0j : equivalence_map[0]) {
      subgroup.insert(group.mult(e_i0, group.mult(e_0j, group.inv(e_i0))));
    }
    invariant_subgroups.emplace_back(std::move(subgroup));
  }
  return invariant_subgroups;
}

}  // namespace group
}  // namespace CASM

#endif
