#ifndef CASM_config_copy_configuration
#define CASM_config_copy_configuration

#include "casm/configuration/definitions.hh"
#include "casm/crystallography/UnitCellCoord.hh"

namespace CASM {
namespace config {

struct Configuration;
struct ConfigurationWithProperties;
struct Supercell;

/// \brief Copy configuration DoF values into a supercell
Configuration copy_configuration(
    Configuration const &motif,
    std::shared_ptr<Supercell const> const &supercell,
    UnitCell const &origin = UnitCell(0, 0, 0));

/// \brief Copy transformed configuration DoF values into a supercell
Configuration copy_configuration(
    Index prim_factor_group_index, UnitCell translation,
    Configuration const &motif,
    std::shared_ptr<Supercell const> const &supercell,
    UnitCell const &origin = UnitCell(0, 0, 0));

/// \brief Copy configuration DoF values and properties into a supercell
ConfigurationWithProperties copy_configuration_with_properties(
    ConfigurationWithProperties const &motif_with_properties,
    std::shared_ptr<Supercell const> const &supercell,
    UnitCell const &origin = UnitCell(0, 0, 0));

/// \brief Copy transformed configuration DoF values and properties into a
///     supercell
ConfigurationWithProperties copy_configuration_with_properties(
    Index prim_factor_group_index, UnitCell translation,
    ConfigurationWithProperties const &motif_with_properties,
    std::shared_ptr<Supercell const> const &supercell,
    UnitCell const &origin = UnitCell(0, 0, 0));

/// \brief Return prim factor group indices that create tilings of motif
///     into supercell that are not equivalent under supercell factor group
///     operations
std::set<Index> unique_generating_prim_factor_group_indices(
    Configuration const &prim_motif, Configuration const &motif,
    std::shared_ptr<Supercell const> const &supercell);

/// \brief Make all equivalent configurations with respect to the prim factor
/// group that fill a supercell
std::vector<Configuration> make_all_super_configurations(
    Configuration const &motif,
    std::shared_ptr<Supercell const> const &supercell);

/// \brief Make all equivalent configurations with respect to the prim factor
/// group that fill a supercell, sorted by subsets of equivalents generated by
/// SupercellSymOp
std::vector<std::vector<Configuration>>
make_all_super_configurations_by_subsets(
    Configuration const &motif,
    std::shared_ptr<Supercell const> const &supercell);

/// \brief Make all equivalent configurations with respect to the prim factor
/// group that fill a supercell
std::vector<ConfigurationWithProperties> make_all_super_configurations(
    ConfigurationWithProperties const &motif_with_properties,
    std::shared_ptr<Supercell const> const &supercell);

/// \brief Make all equivalent configurations with respect to the prim factor
/// group that fill a supercell, sorted by subsets of equivalents generated by
/// SupercellSymOp
std::vector<std::vector<ConfigurationWithProperties>>
make_all_super_configurations_by_subsets(
    ConfigurationWithProperties const &motif_with_properties,
    std::shared_ptr<Supercell const> const &supercell);

/// \brief Return true if no translations within the supercell result in the
///     same configuration
bool is_primitive(Configuration const &configuration);

/// \brief Return the primitive configuration
Configuration make_primitive(Configuration const &configuration);

/// \brief Transform a configuration with properties so it has the primitive
///     configuration
ConfigurationWithProperties make_primitive(
    ConfigurationWithProperties const &configuration_with_properties);

/// \brief Return a prim factor group index that transforms a supercell to an
///     a particular equivalent supercell
Index prim_factor_group_index_to_supercell(
    std::shared_ptr<Supercell const> current_supercell,
    std::shared_ptr<Supercell const> new_supercell);

/// \brief Return the canonical configuration in the canonical supercell
Configuration make_in_canonical_supercell(Configuration const &configuration);

/// \brief Transform a configuration with properties so that it has the
///     canonical configuration in the canonical supercell
ConfigurationWithProperties make_in_canonical_supercell(
    ConfigurationWithProperties const &configuration_with_properties);

}  // namespace config
}  // namespace CASM

#endif
