import os
import f90nml

from .sfe_cfg import SFEconfig
from .fun3d import SimulationFun3dSFE


class SimulationFun3dSFEAdjoint(SimulationFun3dSFE):
    def __init__(self, project_name, pbs=None, fwd_omp_threads=None, adj_omp_threads=None, external_wall_distance=True):
        """
        Runs a steady FUN3D SFE forward and adjoint analysis.
        The preconditioners in the SFE adjoint benefit from larger partition sizes,
        so use of the `adj_omp_threads` argument is encouraged.

        Parameters
        ----------
        project_name: str
            The root name of the project (without any mesh numbers)
        pbs: :class:PBS
            PBS queue helper for forward and adjoint solves
        fwd_omp_threads: int
            Number of OpenMP threads per mpi process to run in the forward analysis
            if doing hybrid parallelism. If not doing hybrid parallelism, this
            optional argument should be left as None.
        adj_omp_threads: int
            Number of OpenMP threads per mpi process to run in the adjoint analysis
            if doing hybrid parallelism. If not doing hybrid parallelism, this
            optional argument should be left as None.
        external_wall_distance: bool
            Whether FUN3D needs to read the wall distance from a file generated by
            an external distance calculator
        """
        super().__init__(project_name, pbs, external_wall_distance, fwd_omp_threads)

        self.adj_omp_threads = adj_omp_threads

        #: str: Name of the namelist file in the adaptation root directory to use for
        #:      the adjoint portion of the simulation.
        self.fun3d_nml_adjoint = 'fun3d.nml_adjoint'

        #: str: Name of the sfe.cfg file in the adaptation root directory to use for the
        #:      adjoint portion of the simulation.
        self.sfe_cfg_adjoint = 'sfe.cfg_adjoint'

        # These are from the parent class but redo doc to clarify forward

        #: str: Name of the namelist file in the adaptation root director to use for
        #       the forward portion of the simulation
        self.fun3d_nml = 'fun3d.nml_forward'

        #: str: Name of the sfe.cfg file in the adaptation root directory to use for the
        #:      forward portion of the simulation.
        self.sfe_cfg = 'sfe.cfg_forward'

        #: bool: Whether the simulation should read the refine-interpolated
        #: adjoint, i.e., whether or not to use the adjoint initialization cfg.
        self.import_adjoint_from_previous_mesh = False

    def get_expected_file_list(self):
        expected_files = super().get_expected_file_list()
        expected_files.extend([self.fun3d_nml_adjoint, self.sfe_cfg_adjoint])
        return expected_files

    def run(self, istep):
        """
        Perform a forward SFE solve and then the adjoint
        """
        print('Running the flow forward solver')
        self._run_fun3d_simulation(istep, 'forward')

        print('Running the flow adjoint solver')
        self._run_fun3d_simulation(istep, 'adjoint', skip_external_distance=True)

    def _check_for_output_files(self, istep, job_name):
        if job_name == 'forward':
            super()._check_for_output_files(istep, job_name)
        else:
            self._check_for_prim_dual_solb()

    def _check_for_prim_dual_solb(self):
        expected_file = 'prim_dual.solb'
        if not os.path.exists(expected_file):
            raise FileNotFoundError(
                f'Expected file: {expected_file} was not found. Something failed with the adjoint solver.')

    def _update_fun3d_nml_fields(self, istep: int, job_name: str, nml: f90nml.Namelist, import_from=True):
        if job_name == 'adjoint':
            self._set_nml_to_not_write_restart_file(nml)
            import_from = False
        super()._update_fun3d_nml_fields(istep, job_name, nml, import_from)

    def _prepare_sfe_cfg(self, istep: int, job_name: str):
        if job_name == 'adjoint':
            self._prepare_sfe_cfg_for_adjoint(istep)
        else:
            super()._prepare_sfe_cfg(istep, job_name)

    def _prepare_sfe_cfg_for_adjoint(self, istep):
        sfe_cfg = SFEconfig(f'../{self.sfe_cfg_adjoint}')
        self._update_sfe_cfg_fields_for_adjoint(istep, sfe_cfg)
        sfe_cfg.write('sfe.cfg', force=True)

    def _set_nml_to_not_write_restart_file(self, nml: f90nml.Namelist):
        nml['code_run_control']['no_restart'] = True

    def _update_sfe_cfg_fields_for_adjoint(self, istep: int, sfe_cfg: SFEconfig):
        if istep > 1 and self.import_adjoint_from_previous_mesh:
            sfe_cfg['import_adjoint_from'] = f'{self._create_project_rootname(istep)}_volume_init.solb'

    def _get_template_fun3d_nml_filename(self, job_name):
        if job_name == 'forward':
            return f'../{self.fun3d_nml}'
        else:
            return f'../{self.fun3d_nml_adjoint}'
