"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.WaiterStateMachine = void 0;
const jsiiDeprecationWarnings = require("../../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const providers_1 = require("./providers");
/**
 * A very simple StateMachine construct highly customized to the provider framework.
 * This is so that this package does not need to depend on aws-stepfunctions module.
 *
 * The state machine continuously calls the isCompleteHandler, until it succeeds or times out.
 * The handler is called `maxAttempts` times with an `interval` duration and a `backoffRate` rate.
 *
 * For example with:
 * - maxAttempts = 360 (30 minutes)
 * - interval = 5
 * - backoffRate = 1 (no backoff)
 *
 * it will make the API Call every 5 seconds and fail after 360 failures.
 *
 * If the backoff rate is changed to 2 (for example), it will
 * - make the first call
 * - wait 5 seconds
 * - make the second call
 * - wait 15 seconds
 * - etc.
 */
class WaiterStateMachine extends constructs_1.Construct {
    constructor(scope, id, props = {}) {
        super(scope, id);
        try {
            jsiiDeprecationWarnings._aws_cdk_integ_tests_alpha_WaiterStateMachineProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, WaiterStateMachine);
            }
            throw error;
        }
        const interval = props.interval || aws_cdk_lib_1.Duration.seconds(5);
        const totalTimeout = props.totalTimeout || aws_cdk_lib_1.Duration.minutes(30);
        const maxAttempts = calculateMaxRetries(totalTimeout.toSeconds(), interval.toSeconds(), props.backoffRate ?? 1);
        if (Math.round(maxAttempts) !== maxAttempts) {
            throw new Error(`Cannot determine retry count since totalTimeout=${totalTimeout.toSeconds()}s is not integrally dividable by queryInterval=${interval.toSeconds()}s`);
        }
        this.isCompleteProvider = new providers_1.AssertionsProvider(this, 'IsCompleteProvider', {
            handler: 'index.isComplete',
            uuid: '76b3e830-a873-425f-8453-eddd85c86925',
        });
        const timeoutProvider = new providers_1.AssertionsProvider(this, 'TimeoutProvider', {
            handler: 'index.onTimeout',
            uuid: '5c1898e0-96fb-4e3e-95d5-f6c67f3ce41a',
        });
        const role = new aws_cdk_lib_1.CfnResource(this, 'Role', {
            type: 'AWS::IAM::Role',
            properties: {
                AssumeRolePolicyDocument: {
                    Version: '2012-10-17',
                    Statement: [{ Action: 'sts:AssumeRole', Effect: 'Allow', Principal: { Service: 'states.amazonaws.com' } }],
                },
                Policies: [
                    {
                        PolicyName: 'InlineInvokeFunctions',
                        PolicyDocument: {
                            Version: '2012-10-17',
                            Statement: [{
                                    Action: 'lambda:InvokeFunction',
                                    Effect: 'Allow',
                                    Resource: [
                                        this.isCompleteProvider.serviceToken,
                                        timeoutProvider.serviceToken,
                                    ],
                                }],
                        },
                    },
                ],
            },
        });
        const definition = aws_cdk_lib_1.Stack.of(this).toJsonString({
            StartAt: 'framework-isComplete-task',
            States: {
                'framework-isComplete-task': {
                    End: true,
                    Retry: [{
                            ErrorEquals: ['States.ALL'],
                            IntervalSeconds: interval.toSeconds(),
                            MaxAttempts: maxAttempts,
                            BackoffRate: props.backoffRate ?? 1,
                        }],
                    Catch: [{
                            ErrorEquals: ['States.ALL'],
                            Next: 'framework-onTimeout-task',
                        }],
                    Type: 'Task',
                    Resource: this.isCompleteProvider.serviceToken,
                },
                'framework-onTimeout-task': {
                    End: true,
                    Type: 'Task',
                    Resource: timeoutProvider.serviceToken,
                },
            },
        });
        const resource = new aws_cdk_lib_1.CfnResource(this, 'Resource', {
            type: 'AWS::StepFunctions::StateMachine',
            properties: {
                DefinitionString: definition,
                RoleArn: role.getAtt('Arn'),
            },
        });
        resource.node.addDependency(role);
        this.stateMachineArn = resource.ref;
        this.roleArn = role.getAtt('Arn').toString();
        this.isCompleteProvider.grantInvoke(this.roleArn);
        timeoutProvider.grantInvoke(this.roleArn);
    }
}
exports.WaiterStateMachine = WaiterStateMachine;
_a = JSII_RTTI_SYMBOL_1;
WaiterStateMachine[_a] = { fqn: "@aws-cdk/integ-tests-alpha.WaiterStateMachine", version: "2.65.0-alpha.0" };
/**
 * Calculate the max number of retries
 */
function calculateMaxRetries(maxSeconds, intervalSeconds, backoff) {
    // if backoff === 1 then we aren't really using backoff
    if (backoff === 1) {
        return Math.floor(maxSeconds / intervalSeconds);
    }
    let retries = 1;
    let nextInterval = intervalSeconds;
    let i = 0;
    while (i < maxSeconds) {
        nextInterval = nextInterval + nextInterval * backoff;
        i += nextInterval;
        if (i >= maxSeconds)
            break;
        retries++;
    }
    return retries;
}
//# sourceMappingURL=data:application/json;base64,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