"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const graphql_transformer_core_1 = require("graphql-transformer-core");
const graphqlTypeStatements = ['Query', 'Mutation', 'Subscription'];
class CdkTransformer extends graphql_transformer_core_1.Transformer {
    constructor() {
        super('MyTransformer', 'directive @nullable on FIELD_DEFINITION');
        this.after = (ctx) => {
            this.printWithoutFilePath(ctx);
            Object.keys(this.tables).forEach(tableName => {
                let table = this.tables[tableName];
                if (!table.Resolvers)
                    table.Resolvers = [];
                if (!table.GSIResolvers)
                    table.GSIResolvers = [];
                Object.keys(this.resolverTableMap).forEach(resolverName => {
                    if (this.resolverTableMap[resolverName] === tableName)
                        table.Resolvers.push(resolverName);
                });
                Object.keys(this.gsiResolverTableMap).forEach(resolverName => {
                    if (this.gsiResolverTableMap[resolverName] === tableName)
                        table.GSIResolvers.push(resolverName);
                });
            });
            ctx.setOutput('CDK_TABLES', this.tables);
            ctx.setOutput('NONE', this.noneDataSources);
            // @ts-ignore
            ctx.setOutput('FUNCTION_RESOLVERS', this.functionResolvers);
            const query = ctx.getQuery();
            if (query) {
                const queryFields = graphql_transformer_core_1.getFieldArguments(query);
                ctx.setOutput('QUERIES', queryFields);
            }
            const mutation = ctx.getMutation();
            if (mutation) {
                const mutationFields = graphql_transformer_core_1.getFieldArguments(mutation);
                ctx.setOutput('MUTATIONS', mutationFields);
            }
            const subscription = ctx.getSubscription();
            if (subscription) {
                const subscriptionFields = graphql_transformer_core_1.getFieldArguments(subscription);
                ctx.setOutput('SUBSCRIPTIONS', subscriptionFields);
            }
        };
        this.tables = {};
        this.noneDataSources = {};
        this.functionResolvers = [];
        this.resolverTableMap = {};
        this.gsiResolverTableMap = {};
    }
    printWithoutFilePath(ctx) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        const templateResources = ctx.template.Resources;
        if (!templateResources)
            return;
        for (const resourceName of Object.keys(templateResources)) {
            const resource = templateResources[resourceName];
            if (resource.Type === 'AWS::DynamoDB::Table') {
                this.buildTablesFromResource(resourceName, ctx);
            }
            else if (resource.Type === 'AWS::AppSync::Resolver') {
                if (((_a = resource.Properties) === null || _a === void 0 ? void 0 : _a.DataSourceName) === 'NONE') {
                    this.noneDataSources[resource.Properties.FieldName] = {
                        typeName: resource.Properties.TypeName,
                        fieldName: resource.Properties.FieldName
                    };
                }
                else if (((_b = resource.Properties) === null || _b === void 0 ? void 0 : _b.Kind) === 'PIPELINE') { // TODO: This may not be accurate but works for now
                    let fieldName = (_c = resource.Properties) === null || _c === void 0 ? void 0 : _c.FieldName;
                    let typeName = (_d = resource.Properties) === null || _d === void 0 ? void 0 : _d.TypeName;
                    this.functionResolvers.push({
                        typeName: typeName,
                        fieldName: fieldName
                    });
                }
                else {
                    let typeName = (_e = resource.Properties) === null || _e === void 0 ? void 0 : _e.TypeName;
                    let fieldName = (_f = resource.Properties) === null || _f === void 0 ? void 0 : _f.FieldName;
                    let tableName = (_h = (_g = resource.Properties) === null || _g === void 0 ? void 0 : _g.DataSourceName) === null || _h === void 0 ? void 0 : _h.payload[0];
                    tableName = tableName.replace('DataSource', 'Table');
                    if (graphqlTypeStatements.indexOf(typeName) >= 0) {
                        this.resolverTableMap[fieldName] = tableName;
                    }
                    else { // this is a GSI
                        this.gsiResolverTableMap[`${typeName}${fieldName}`] = tableName;
                    }
                }
            }
        }
    }
    buildTablesFromResource(resourceName, ctx) {
        var _a, _b, _c, _d;
        const tableResource = ctx.template.Resources ? ctx.template.Resources[resourceName] : undefined;
        const attributeDefinitions = (_a = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _a === void 0 ? void 0 : _a.AttributeDefinitions;
        const keySchema = (_b = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _b === void 0 ? void 0 : _b.KeySchema;
        let keys = this.parseKeySchema(keySchema, attributeDefinitions);
        let table = {
            TableName: resourceName,
            PartitionKey: keys.partitionKey,
            SortKey: keys.sortKey,
            TTL: (_c = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _c === void 0 ? void 0 : _c.TimeToLiveSpecification,
            GlobalSecondaryIndexes: []
        };
        const gsis = (_d = tableResource === null || tableResource === void 0 ? void 0 : tableResource.Properties) === null || _d === void 0 ? void 0 : _d.GlobalSecondaryIndexes;
        if (gsis) {
            gsis.forEach((gsi) => {
                let gsiKeys = this.parseKeySchema(gsi.KeySchema, attributeDefinitions);
                let gsiDefinition = {
                    IndexName: gsi.IndexName,
                    Projection: gsi.Projection,
                    PartitionKey: gsiKeys.partitionKey,
                    SortKey: gsiKeys.sortKey,
                };
                table.GlobalSecondaryIndexes.push(gsiDefinition);
            });
        }
        this.tables[resourceName] = table;
    }
    parseKeySchema(keySchema, attributeDefinitions) {
        let partitionKey = {};
        let sortKey = {};
        keySchema.forEach((key) => {
            let keyType = key.KeyType;
            let attributeName = key.AttributeName;
            let attribute = attributeDefinitions.find((attribute) => {
                return attribute.AttributeName === attributeName;
            });
            if (keyType === 'HASH') {
                partitionKey = {
                    name: attribute.AttributeName,
                    type: attribute.AttributeType
                };
            }
            else if (keyType === 'RANGE') {
                sortKey = {
                    name: attribute.AttributeName,
                    type: attribute.AttributeType
                };
            }
        });
        return {
            partitionKey,
            sortKey
        };
    }
}
exports.default = CdkTransformer;
//# sourceMappingURL=data:application/json;base64,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