# Copyright 2019-2021 Cambridge Quantum Computing
#
# You may not use this file except in compliance with the Licence.
# You may obtain a copy of the Licence in the LICENCE file accompanying
# these documents or at:
#
#     https://cqcl.github.io/pytket/build/html/licence.html

from typing import Optional, List, TYPE_CHECKING

from qiskit.providers.aer import QasmSimulator  # type: ignore

from .aer import AerBackend
from .ibm import IBMQBackend

if TYPE_CHECKING:
    from pytket.device import Device  # type: ignore
    from pytket.predicates import Predicate  # type: ignore
    from pytket.passes import BasePass  # type: ignore


class IBMQEmulatorBackend(AerBackend):
    """A backend which uses the AerBackend to emulate the behaviour of IBMQBackend.
    Attempts to perform the same compilation and predicate checks as IBMQBackend.
    Requires a valid IBMQ account.

    """

    _supports_shots = True
    _supports_counts = True
    _persistent_handles = False

    def __init__(
        self,
        backend_name: str,
        hub: Optional[str] = None,
        group: Optional[str] = None,
        project: Optional[str] = None,
    ):
        """Construct an IBMQEmulatorBackend. Identical to :py:class:`IBMQBackend` constructor,
        except there is no `monitor` parameter. See :py:class:`IBMQBackend` docs for more details.
        """

        self._ibmq = IBMQBackend(backend_name, hub, group, project)
        super().__init__()
        self._backend = QasmSimulator.from_backend(self._ibmq._backend)

    @property
    def device(self) -> Optional["Device"]:
        return self._ibmq._device

    @property
    def required_predicates(self) -> List["Predicate"]:
        return list(self._ibmq.required_predicates)

    def default_compilation_pass(self, optimisation_level: int = 1) -> "BasePass":
        return self._ibmq.default_compilation_pass(optimisation_level)
