"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.RESERVED_TUNNEL_INSIDE_CIDR = exports.VpnConnection = exports.VpnGateway = exports.VpnConnectionType = void 0;
const net = require("net");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * The VPN connection type.
 *
 * @stability stable
 */
var VpnConnectionType;
(function (VpnConnectionType) {
    VpnConnectionType["IPSEC_1"] = "ipsec.1";
    VpnConnectionType["DUMMY"] = "dummy";
})(VpnConnectionType = exports.VpnConnectionType || (exports.VpnConnectionType = {}));
/**
 * The VPN Gateway that shall be added to the VPC.
 *
 * @stability stable
 * @resource AWS::EC2::VPNGateway
 */
class VpnGateway extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        // This is 'Default' instead of 'Resource', because using 'Default' will generate
        // a logical ID for a VpnGateway which is exactly the same as the logical ID that used
        // to be created for the CfnVPNGateway (and 'Resource' would not do that).
        const vpnGW = new ec2_generated_1.CfnVPNGateway(this, 'Default', props);
        this.gatewayId = vpnGW.ref;
    }
}
exports.VpnGateway = VpnGateway;
/**
 * Define a VPN Connection.
 *
 * @stability stable
 * @resource AWS::EC2::VPNConnection
 */
class VpnConnection extends core_1.Resource {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.vpc.vpnGatewayId) {
            props.vpc.enableVpnGateway({
                type: 'ipsec.1',
                amazonSideAsn: props.asn,
            });
        }
        if (!core_1.Token.isUnresolved(props.ip) && !net.isIPv4(props.ip)) {
            throw new Error(`The \`ip\` ${props.ip} is not a valid IPv4 address.`);
        }
        const type = VpnConnectionType.IPSEC_1;
        const bgpAsn = props.asn || 65000;
        const customerGateway = new ec2_generated_1.CfnCustomerGateway(this, 'CustomerGateway', {
            bgpAsn,
            ipAddress: props.ip,
            type,
        });
        this.customerGatewayId = customerGateway.ref;
        this.customerGatewayAsn = bgpAsn;
        this.customerGatewayIp = props.ip;
        // Validate tunnel options
        if (props.tunnelOptions) {
            if (props.tunnelOptions.length > 2) {
                throw new Error('Cannot specify more than two `tunnelOptions`');
            }
            if (props.tunnelOptions.length === 2 && props.tunnelOptions[0].tunnelInsideCidr === props.tunnelOptions[1].tunnelInsideCidr) {
                throw new Error(`Same ${props.tunnelOptions[0].tunnelInsideCidr} \`tunnelInsideCidr\` cannot be used for both tunnels.`);
            }
            props.tunnelOptions.forEach((options, index) => {
                if (options.preSharedKey && !core_1.Token.isUnresolved(options.preSharedKey) && !/^[a-zA-Z1-9._][a-zA-Z\d._]{7,63}$/.test(options.preSharedKey)) {
                    /* eslint-disable max-len */
                    throw new Error(`The \`preSharedKey\` ${options.preSharedKey} for tunnel ${index + 1} is invalid. Allowed characters are alphanumeric characters and ._. Must be between 8 and 64 characters in length and cannot start with zero (0).`);
                    /* eslint-enable max-len */
                }
                if (options.tunnelInsideCidr) {
                    if (exports.RESERVED_TUNNEL_INSIDE_CIDR.includes(options.tunnelInsideCidr)) {
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is a reserved inside CIDR.`);
                    }
                    if (!/^169\.254\.\d{1,3}\.\d{1,3}\/30$/.test(options.tunnelInsideCidr)) {
                        /* eslint-disable-next-line max-len */
                        throw new Error(`The \`tunnelInsideCidr\` ${options.tunnelInsideCidr} for tunnel ${index + 1} is not a size /30 CIDR block from the 169.254.0.0/16 range.`);
                    }
                }
            });
        }
        const vpnConnection = new ec2_generated_1.CfnVPNConnection(this, 'Resource', {
            type,
            customerGatewayId: customerGateway.ref,
            staticRoutesOnly: props.staticRoutes ? true : false,
            vpnGatewayId: props.vpc.vpnGatewayId,
            vpnTunnelOptionsSpecifications: props.tunnelOptions,
        });
        this.vpnId = vpnConnection.ref;
        if (props.staticRoutes) {
            props.staticRoutes.forEach(route => {
                new ec2_generated_1.CfnVPNConnectionRoute(this, `Route${route.replace(/[^\d]/g, '')}`, {
                    destinationCidrBlock: route,
                    vpnConnectionId: this.vpnId,
                });
            });
        }
    }
    /**
     * Return the given named metric for all VPN connections in the account/region.
     *
     * @stability stable
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/VPN',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the tunnel state of all VPN connections in the account/region.
     *
     * @default average over 5 minutes
     * @stability stable
     */
    static metricAllTunnelState(props) {
        return this.metricAll('TunnelState', { statistic: 'avg', ...props });
    }
    /**
     * Metric for the tunnel data in of all VPN connections in the account/region.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllTunnelDataIn(props) {
        return this.metricAll('TunnelDataIn', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the tunnel data out of all VPN connections.
     *
     * @default sum over 5 minutes
     * @stability stable
     */
    static metricAllTunnelDataOut(props) {
        return this.metricAll('TunnelDataOut', { statistic: 'sum', ...props });
    }
}
exports.VpnConnection = VpnConnection;
exports.RESERVED_TUNNEL_INSIDE_CIDR = [
    '169.254.0.0/30',
    '169.254.1.0/30',
    '169.254.2.0/30',
    '169.254.3.0/30',
    '169.254.4.0/30',
    '169.254.5.0/30',
    '169.254.169.252/30',
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidnBuLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidnBuLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDJCQUEyQjtBQUMzQixzREFBc0Q7QUFDdEQsd0NBQTJEO0FBRTNELG1EQUt5Qjs7Ozs7O0FBMkh6QixJQUFZLGlCQVdYO0FBWEQsV0FBWSxpQkFBaUI7SUFJM0Isd0NBQW1CLENBQUE7SUFNbkIsb0NBQWUsQ0FBQTtBQUNqQixDQUFDLEVBWFcsaUJBQWlCLEdBQWpCLHlCQUFpQixLQUFqQix5QkFBaUIsUUFXNUI7Ozs7Ozs7QUFPRCxNQUFhLFVBQVcsU0FBUSxlQUFROzs7O0lBT3RDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0I7UUFDOUQsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixpRkFBaUY7UUFDakYsc0ZBQXNGO1FBQ3RGLDBFQUEwRTtRQUMxRSxNQUFNLEtBQUssR0FBRyxJQUFJLDZCQUFhLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4RCxJQUFJLENBQUMsU0FBUyxHQUFHLEtBQUssQ0FBQyxHQUFHLENBQUM7SUFDN0IsQ0FBQztDQUNGO0FBaEJELGdDQWdCQzs7Ozs7OztBQU1ELE1BQWEsYUFBYyxTQUFRLGVBQVE7Ozs7SUE0Q3pDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxZQUFZLEVBQUU7WUFDM0IsS0FBSyxDQUFDLEdBQUcsQ0FBQyxnQkFBZ0IsQ0FBQztnQkFDekIsSUFBSSxFQUFFLFNBQVM7Z0JBQ2YsYUFBYSxFQUFFLEtBQUssQ0FBQyxHQUFHO2FBQ3pCLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLEVBQUU7WUFDMUQsTUFBTSxJQUFJLEtBQUssQ0FBQyxjQUFjLEtBQUssQ0FBQyxFQUFFLCtCQUErQixDQUFDLENBQUM7U0FDeEU7UUFFRCxNQUFNLElBQUksR0FBRyxpQkFBaUIsQ0FBQyxPQUFPLENBQUM7UUFDdkMsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLEdBQUcsSUFBSSxLQUFLLENBQUM7UUFFbEMsTUFBTSxlQUFlLEdBQUcsSUFBSSxrQ0FBa0IsQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLEVBQUU7WUFDdEUsTUFBTTtZQUNOLFNBQVMsRUFBRSxLQUFLLENBQUMsRUFBRTtZQUNuQixJQUFJO1NBQ0wsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLGlCQUFpQixHQUFHLGVBQWUsQ0FBQyxHQUFHLENBQUM7UUFDN0MsSUFBSSxDQUFDLGtCQUFrQixHQUFHLE1BQU0sQ0FBQztRQUNqQyxJQUFJLENBQUMsaUJBQWlCLEdBQUcsS0FBSyxDQUFDLEVBQUUsQ0FBQztRQUVsQywwQkFBMEI7UUFDMUIsSUFBSSxLQUFLLENBQUMsYUFBYSxFQUFFO1lBQ3ZCLElBQUksS0FBSyxDQUFDLGFBQWEsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO2dCQUNsQyxNQUFNLElBQUksS0FBSyxDQUFDLDhDQUE4QyxDQUFDLENBQUM7YUFDakU7WUFFRCxJQUFJLEtBQUssQ0FBQyxhQUFhLENBQUMsTUFBTSxLQUFLLENBQUMsSUFBSSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixLQUFLLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsZ0JBQWdCLEVBQUU7Z0JBQzNILE1BQU0sSUFBSSxLQUFLLENBQUMsUUFBUSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLGdCQUFnQix3REFBd0QsQ0FBQyxDQUFDO2FBQzFIO1lBRUQsS0FBSyxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsS0FBSyxFQUFFLEVBQUU7Z0JBQzdDLElBQUksT0FBTyxDQUFDLFlBQVksSUFBSSxDQUFDLFlBQUssQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsbUNBQW1DLENBQUMsSUFBSSxDQUFDLE9BQU8sQ0FBQyxZQUFZLENBQUMsRUFBRTtvQkFDeEksNEJBQTRCO29CQUM1QixNQUFNLElBQUksS0FBSyxDQUFDLHdCQUF3QixPQUFPLENBQUMsWUFBWSxlQUFlLEtBQUssR0FBRyxDQUFDLG1KQUFtSixDQUFDLENBQUM7b0JBQ3pPLDJCQUEyQjtpQkFDNUI7Z0JBRUQsSUFBSSxPQUFPLENBQUMsZ0JBQWdCLEVBQUU7b0JBQzVCLElBQUksbUNBQTJCLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFO3dCQUNsRSxNQUFNLElBQUksS0FBSyxDQUFDLDRCQUE0QixPQUFPLENBQUMsZ0JBQWdCLGVBQWUsS0FBSyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztxQkFDNUg7b0JBRUQsSUFBSSxDQUFDLGtDQUFrQyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLENBQUMsRUFBRTt3QkFDdEUsc0NBQXNDO3dCQUN0QyxNQUFNLElBQUksS0FBSyxDQUFDLDRCQUE0QixPQUFPLENBQUMsZ0JBQWdCLGVBQWUsS0FBSyxHQUFHLENBQUMsOERBQThELENBQUMsQ0FBQztxQkFDN0o7aUJBQ0Y7WUFDSCxDQUFDLENBQUMsQ0FBQztTQUNKO1FBRUQsTUFBTSxhQUFhLEdBQUcsSUFBSSxnQ0FBZ0IsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQzNELElBQUk7WUFDSixpQkFBaUIsRUFBRSxlQUFlLENBQUMsR0FBRztZQUN0QyxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsWUFBWSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUs7WUFDbkQsWUFBWSxFQUFFLEtBQUssQ0FBQyxHQUFHLENBQUMsWUFBWTtZQUNwQyw4QkFBOEIsRUFBRSxLQUFLLENBQUMsYUFBYTtTQUNwRCxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsS0FBSyxHQUFHLGFBQWEsQ0FBQyxHQUFHLENBQUM7UUFFL0IsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3RCLEtBQUssQ0FBQyxZQUFZLENBQUMsT0FBTyxDQUFDLEtBQUssQ0FBQyxFQUFFO2dCQUNqQyxJQUFJLHFDQUFxQixDQUFDLElBQUksRUFBRSxRQUFRLEtBQUssQ0FBQyxPQUFPLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxFQUFFLEVBQUU7b0JBQ3JFLG9CQUFvQixFQUFFLEtBQUs7b0JBQzNCLGVBQWUsRUFBRSxJQUFJLENBQUMsS0FBSztpQkFDNUIsQ0FBQyxDQUFDO1lBQ0wsQ0FBQyxDQUFDLENBQUM7U0FDSjtJQUNILENBQUM7Ozs7OztJQW5ITSxNQUFNLENBQUMsU0FBUyxDQUFDLFVBQWtCLEVBQUUsS0FBZ0M7UUFDMUUsT0FBTyxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUM7WUFDM0IsU0FBUyxFQUFFLFNBQVM7WUFDcEIsVUFBVTtZQUNWLEdBQUcsS0FBSztTQUNULENBQUMsQ0FBQztJQUNMLENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsb0JBQW9CLENBQUMsS0FBZ0M7UUFDakUsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLGFBQWEsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ3ZFLENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBZ0M7UUFDbEUsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLGNBQWMsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ3hFLENBQUM7Ozs7Ozs7SUFPTSxNQUFNLENBQUMsc0JBQXNCLENBQUMsS0FBZ0M7UUFDbkUsT0FBTyxJQUFJLENBQUMsU0FBUyxDQUFDLGVBQWUsRUFBRSxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsR0FBRyxLQUFLLEVBQUUsQ0FBQyxDQUFDO0lBQ3pFLENBQUM7Q0FtRkY7QUF4SEQsc0NBd0hDO0FBRVksUUFBQSwyQkFBMkIsR0FBRztJQUN6QyxnQkFBZ0I7SUFDaEIsZ0JBQWdCO0lBQ2hCLGdCQUFnQjtJQUNoQixnQkFBZ0I7SUFDaEIsZ0JBQWdCO0lBQ2hCLGdCQUFnQjtJQUNoQixvQkFBb0I7Q0FDckIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIG5ldCBmcm9tICduZXQnO1xuaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgeyBJUmVzb3VyY2UsIFJlc291cmNlLCBUb2tlbiB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnY29uc3RydWN0cyc7XG5pbXBvcnQge1xuICBDZm5DdXN0b21lckdhdGV3YXksXG4gIENmblZQTkNvbm5lY3Rpb24sXG4gIENmblZQTkNvbm5lY3Rpb25Sb3V0ZSxcbiAgQ2ZuVlBOR2F0ZXdheSxcbn0gZnJvbSAnLi9lYzIuZ2VuZXJhdGVkJztcbmltcG9ydCB7IElWcGMsIFN1Ym5ldFNlbGVjdGlvbiB9IGZyb20gJy4vdnBjJztcblxuZXhwb3J0IGludGVyZmFjZSBJVnBuQ29ubmVjdGlvbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwbklkOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjdXN0b21lckdhdGV3YXlJZDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBjdXN0b21lckdhdGV3YXlJcDogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IGN1c3RvbWVyR2F0ZXdheUFzbjogbnVtYmVyO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBpbnRlcmZhY2UgSVZwbkdhdGV3YXkgZXh0ZW5kcyBJUmVzb3VyY2Uge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBnYXRld2F5SWQ6IHN0cmluZ1xufVxuXG5leHBvcnQgaW50ZXJmYWNlIFZwblR1bm5lbE9wdGlvbiB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBwcmVTaGFyZWRLZXk/OiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHR1bm5lbEluc2lkZUNpZHI/OiBzdHJpbmc7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVnBuQ29ubmVjdGlvbk9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgaXA6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgYXNuPzogbnVtYmVyO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBzdGF0aWNSb3V0ZXM/OiBzdHJpbmdbXTtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSB0dW5uZWxPcHRpb25zPzogVnBuVHVubmVsT3B0aW9uW107XG59XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBWcG5HYXRld2F5UHJvcHMge1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHR5cGU6IHN0cmluZztcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBhbWF6b25TaWRlQXNuPzogbnVtYmVyO1xufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBFbmFibGVWcG5HYXRld2F5T3B0aW9ucyBleHRlbmRzIFZwbkdhdGV3YXlQcm9wcyB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdnBuUm91dGVQcm9wYWdhdGlvbj86IFN1Ym5ldFNlbGVjdGlvbltdXG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgVnBuQ29ubmVjdGlvblByb3BzIGV4dGVuZHMgVnBuQ29ubmVjdGlvbk9wdGlvbnMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwYzogSVZwYztcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBlbnVtIFZwbkNvbm5lY3Rpb25UeXBlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgSVBTRUNfMSA9ICdpcHNlYy4xJyxcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICBEVU1NWSA9ICdkdW1teSdcbn1cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGNsYXNzIFZwbkdhdGV3YXkgZXh0ZW5kcyBSZXNvdXJjZSBpbXBsZW1lbnRzIElWcG5HYXRld2F5IHtcblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHJlYWRvbmx5IGdhdGV3YXlJZDogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBWcG5HYXRld2F5UHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgLy8gVGhpcyBpcyAnRGVmYXVsdCcgaW5zdGVhZCBvZiAnUmVzb3VyY2UnLCBiZWNhdXNlIHVzaW5nICdEZWZhdWx0JyB3aWxsIGdlbmVyYXRlXG4gICAgLy8gYSBsb2dpY2FsIElEIGZvciBhIFZwbkdhdGV3YXkgd2hpY2ggaXMgZXhhY3RseSB0aGUgc2FtZSBhcyB0aGUgbG9naWNhbCBJRCB0aGF0IHVzZWRcbiAgICAvLyB0byBiZSBjcmVhdGVkIGZvciB0aGUgQ2ZuVlBOR2F0ZXdheSAoYW5kICdSZXNvdXJjZScgd291bGQgbm90IGRvIHRoYXQpLlxuICAgIGNvbnN0IHZwbkdXID0gbmV3IENmblZQTkdhdGV3YXkodGhpcywgJ0RlZmF1bHQnLCBwcm9wcyk7XG4gICAgdGhpcy5nYXRld2F5SWQgPSB2cG5HVy5yZWY7XG4gIH1cbn1cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG5leHBvcnQgY2xhc3MgVnBuQ29ubmVjdGlvbiBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSVZwbkNvbm5lY3Rpb24ge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbWV0cmljQWxsKG1ldHJpY05hbWU6IHN0cmluZywgcHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBjbG91ZHdhdGNoLk1ldHJpYyh7XG4gICAgICBuYW1lc3BhY2U6ICdBV1MvVlBOJyxcbiAgICAgIG1ldHJpY05hbWUsXG4gICAgICAuLi5wcm9wcyxcbiAgICB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcHVibGljIHN0YXRpYyBtZXRyaWNBbGxUdW5uZWxTdGF0ZShwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWNBbGwoJ1R1bm5lbFN0YXRlJywgeyBzdGF0aXN0aWM6ICdhdmcnLCAuLi5wcm9wcyB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbWV0cmljQWxsVHVubmVsRGF0YUluKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWMge1xuICAgIHJldHVybiB0aGlzLm1ldHJpY0FsbCgnVHVubmVsRGF0YUluJywgeyBzdGF0aXN0aWM6ICdzdW0nLCAuLi5wcm9wcyB9KTtcbiAgfVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHB1YmxpYyBzdGF0aWMgbWV0cmljQWxsVHVubmVsRGF0YU91dChwcm9wcz86IGNsb3Vkd2F0Y2guTWV0cmljT3B0aW9ucyk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICByZXR1cm4gdGhpcy5tZXRyaWNBbGwoJ1R1bm5lbERhdGFPdXQnLCB7IHN0YXRpc3RpYzogJ3N1bScsIC4uLnByb3BzIH0pO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IHZwbklkOiBzdHJpbmc7XG4gIHB1YmxpYyByZWFkb25seSBjdXN0b21lckdhdGV3YXlJZDogc3RyaW5nO1xuICBwdWJsaWMgcmVhZG9ubHkgY3VzdG9tZXJHYXRld2F5SXA6IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGN1c3RvbWVyR2F0ZXdheUFzbjogbnVtYmVyO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByb3BzOiBWcG5Db25uZWN0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgaWYgKCFwcm9wcy52cGMudnBuR2F0ZXdheUlkKSB7XG4gICAgICBwcm9wcy52cGMuZW5hYmxlVnBuR2F0ZXdheSh7XG4gICAgICAgIHR5cGU6ICdpcHNlYy4xJyxcbiAgICAgICAgYW1hem9uU2lkZUFzbjogcHJvcHMuYXNuLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgaWYgKCFUb2tlbi5pc1VucmVzb2x2ZWQocHJvcHMuaXApICYmICFuZXQuaXNJUHY0KHByb3BzLmlwKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBUaGUgXFxgaXBcXGAgJHtwcm9wcy5pcH0gaXMgbm90IGEgdmFsaWQgSVB2NCBhZGRyZXNzLmApO1xuICAgIH1cblxuICAgIGNvbnN0IHR5cGUgPSBWcG5Db25uZWN0aW9uVHlwZS5JUFNFQ18xO1xuICAgIGNvbnN0IGJncEFzbiA9IHByb3BzLmFzbiB8fCA2NTAwMDtcblxuICAgIGNvbnN0IGN1c3RvbWVyR2F0ZXdheSA9IG5ldyBDZm5DdXN0b21lckdhdGV3YXkodGhpcywgJ0N1c3RvbWVyR2F0ZXdheScsIHtcbiAgICAgIGJncEFzbixcbiAgICAgIGlwQWRkcmVzczogcHJvcHMuaXAsXG4gICAgICB0eXBlLFxuICAgIH0pO1xuXG4gICAgdGhpcy5jdXN0b21lckdhdGV3YXlJZCA9IGN1c3RvbWVyR2F0ZXdheS5yZWY7XG4gICAgdGhpcy5jdXN0b21lckdhdGV3YXlBc24gPSBiZ3BBc247XG4gICAgdGhpcy5jdXN0b21lckdhdGV3YXlJcCA9IHByb3BzLmlwO1xuXG4gICAgLy8gVmFsaWRhdGUgdHVubmVsIG9wdGlvbnNcbiAgICBpZiAocHJvcHMudHVubmVsT3B0aW9ucykge1xuICAgICAgaWYgKHByb3BzLnR1bm5lbE9wdGlvbnMubGVuZ3RoID4gMikge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0Nhbm5vdCBzcGVjaWZ5IG1vcmUgdGhhbiB0d28gYHR1bm5lbE9wdGlvbnNgJyk7XG4gICAgICB9XG5cbiAgICAgIGlmIChwcm9wcy50dW5uZWxPcHRpb25zLmxlbmd0aCA9PT0gMiAmJiBwcm9wcy50dW5uZWxPcHRpb25zWzBdLnR1bm5lbEluc2lkZUNpZHIgPT09IHByb3BzLnR1bm5lbE9wdGlvbnNbMV0udHVubmVsSW5zaWRlQ2lkcikge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFNhbWUgJHtwcm9wcy50dW5uZWxPcHRpb25zWzBdLnR1bm5lbEluc2lkZUNpZHJ9IFxcYHR1bm5lbEluc2lkZUNpZHJcXGAgY2Fubm90IGJlIHVzZWQgZm9yIGJvdGggdHVubmVscy5gKTtcbiAgICAgIH1cblxuICAgICAgcHJvcHMudHVubmVsT3B0aW9ucy5mb3JFYWNoKChvcHRpb25zLCBpbmRleCkgPT4ge1xuICAgICAgICBpZiAob3B0aW9ucy5wcmVTaGFyZWRLZXkgJiYgIVRva2VuLmlzVW5yZXNvbHZlZChvcHRpb25zLnByZVNoYXJlZEtleSkgJiYgIS9eW2EtekEtWjEtOS5fXVthLXpBLVpcXGQuX117Nyw2M30kLy50ZXN0KG9wdGlvbnMucHJlU2hhcmVkS2V5KSkge1xuICAgICAgICAgIC8qIGVzbGludC1kaXNhYmxlIG1heC1sZW4gKi9cbiAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBcXGBwcmVTaGFyZWRLZXlcXGAgJHtvcHRpb25zLnByZVNoYXJlZEtleX0gZm9yIHR1bm5lbCAke2luZGV4ICsgMX0gaXMgaW52YWxpZC4gQWxsb3dlZCBjaGFyYWN0ZXJzIGFyZSBhbHBoYW51bWVyaWMgY2hhcmFjdGVycyBhbmQgLl8uIE11c3QgYmUgYmV0d2VlbiA4IGFuZCA2NCBjaGFyYWN0ZXJzIGluIGxlbmd0aCBhbmQgY2Fubm90IHN0YXJ0IHdpdGggemVybyAoMCkuYCk7XG4gICAgICAgICAgLyogZXNsaW50LWVuYWJsZSBtYXgtbGVuICovXG4gICAgICAgIH1cblxuICAgICAgICBpZiAob3B0aW9ucy50dW5uZWxJbnNpZGVDaWRyKSB7XG4gICAgICAgICAgaWYgKFJFU0VSVkVEX1RVTk5FTF9JTlNJREVfQ0lEUi5pbmNsdWRlcyhvcHRpb25zLnR1bm5lbEluc2lkZUNpZHIpKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBcXGB0dW5uZWxJbnNpZGVDaWRyXFxgICR7b3B0aW9ucy50dW5uZWxJbnNpZGVDaWRyfSBmb3IgdHVubmVsICR7aW5kZXggKyAxfSBpcyBhIHJlc2VydmVkIGluc2lkZSBDSURSLmApO1xuICAgICAgICAgIH1cblxuICAgICAgICAgIGlmICghL14xNjlcXC4yNTRcXC5cXGR7MSwzfVxcLlxcZHsxLDN9XFwvMzAkLy50ZXN0KG9wdGlvbnMudHVubmVsSW5zaWRlQ2lkcikpIHtcbiAgICAgICAgICAgIC8qIGVzbGludC1kaXNhYmxlLW5leHQtbGluZSBtYXgtbGVuICovXG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFRoZSBcXGB0dW5uZWxJbnNpZGVDaWRyXFxgICR7b3B0aW9ucy50dW5uZWxJbnNpZGVDaWRyfSBmb3IgdHVubmVsICR7aW5kZXggKyAxfSBpcyBub3QgYSBzaXplIC8zMCBDSURSIGJsb2NrIGZyb20gdGhlIDE2OS4yNTQuMC4wLzE2IHJhbmdlLmApO1xuICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgY29uc3QgdnBuQ29ubmVjdGlvbiA9IG5ldyBDZm5WUE5Db25uZWN0aW9uKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIHR5cGUsXG4gICAgICBjdXN0b21lckdhdGV3YXlJZDogY3VzdG9tZXJHYXRld2F5LnJlZixcbiAgICAgIHN0YXRpY1JvdXRlc09ubHk6IHByb3BzLnN0YXRpY1JvdXRlcyA/IHRydWUgOiBmYWxzZSxcbiAgICAgIHZwbkdhdGV3YXlJZDogcHJvcHMudnBjLnZwbkdhdGV3YXlJZCxcbiAgICAgIHZwblR1bm5lbE9wdGlvbnNTcGVjaWZpY2F0aW9uczogcHJvcHMudHVubmVsT3B0aW9ucyxcbiAgICB9KTtcblxuICAgIHRoaXMudnBuSWQgPSB2cG5Db25uZWN0aW9uLnJlZjtcblxuICAgIGlmIChwcm9wcy5zdGF0aWNSb3V0ZXMpIHtcbiAgICAgIHByb3BzLnN0YXRpY1JvdXRlcy5mb3JFYWNoKHJvdXRlID0+IHtcbiAgICAgICAgbmV3IENmblZQTkNvbm5lY3Rpb25Sb3V0ZSh0aGlzLCBgUm91dGUke3JvdXRlLnJlcGxhY2UoL1teXFxkXS9nLCAnJyl9YCwge1xuICAgICAgICAgIGRlc3RpbmF0aW9uQ2lkckJsb2NrOiByb3V0ZSxcbiAgICAgICAgICB2cG5Db25uZWN0aW9uSWQ6IHRoaXMudnBuSWQsXG4gICAgICAgIH0pO1xuICAgICAgfSk7XG4gICAgfVxuICB9XG59XG5cbmV4cG9ydCBjb25zdCBSRVNFUlZFRF9UVU5ORUxfSU5TSURFX0NJRFIgPSBbXG4gICcxNjkuMjU0LjAuMC8zMCcsXG4gICcxNjkuMjU0LjEuMC8zMCcsXG4gICcxNjkuMjU0LjIuMC8zMCcsXG4gICcxNjkuMjU0LjMuMC8zMCcsXG4gICcxNjkuMjU0LjQuMC8zMCcsXG4gICcxNjkuMjU0LjUuMC8zMCcsXG4gICcxNjkuMjU0LjE2OS4yNTIvMzAnLFxuXTtcbiJdfQ==