"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UserData = void 0;
const core_1 = require("@aws-cdk/core");
const machine_image_1 = require("./machine-image");
/**
 * Instance User Data.
 *
 * @stability stable
 */
class UserData {
    /**
     * Create a userdata object for Linux hosts.
     *
     * @stability stable
     */
    static forLinux(options = {}) {
        return new LinuxUserData(options);
    }
    /**
     * Create a userdata object for Windows hosts.
     *
     * @stability stable
     */
    static forWindows() {
        return new WindowsUserData();
    }
    /**
     * Create a userdata object with custom content.
     *
     * @stability stable
     */
    static custom(content) {
        const userData = new CustomUserData();
        userData.addCommands(content);
        return userData;
    }
    /**
     * @stability stable
     */
    static forOperatingSystem(os) {
        switch (os) {
            case machine_image_1.OperatingSystemType.LINUX: return UserData.forLinux();
            case machine_image_1.OperatingSystemType.WINDOWS: return UserData.forWindows();
            case machine_image_1.OperatingSystemType.UNKNOWN: throw new Error('Cannot determine UserData for unknown operating system type');
        }
    }
}
exports.UserData = UserData;
/**
 * Linux Instance User Data
 */
class LinuxUserData extends UserData {
    constructor(props = {}) {
        super();
        this.props = props;
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        var _a;
        const shebang = (_a = this.props.shebang) !== null && _a !== void 0 ? _a : '#!/bin/bash';
        return [shebang, ...(this.renderOnExitLines()), ...this.lines].join('\n');
    }
    addS3DownloadCommand(params) {
        const s3Path = `s3://${params.bucket.bucketName}/${params.bucketKey}`;
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `/tmp/${params.bucketKey}`;
        this.addCommands(`mkdir -p $(dirname '${localPath}')`, `aws s3 cp '${s3Path}' '${localPath}'`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _a;
        this.addCommands('set -e', `chmod +x '${params.filePath}'`, `'${params.filePath}' ${(_a = params.arguments) !== null && _a !== void 0 ? _a : ''}`.trim());
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`/opt/aws/bin/cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} -e $exitCode || echo 'Failed to send Cloudformation Signal'`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['function exitTrap(){', 'exitCode=$?', ...this.onExitLines, '}', 'trap exitTrap EXIT'];
        }
        return [];
    }
}
/**
 * Windows Instance User Data
 */
class WindowsUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
        this.onExitLines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands(...commands) {
        this.onExitLines.push(...commands);
    }
    render() {
        return `<powershell>${[...(this.renderOnExitLines()),
            ...this.lines,
            ...(this.onExitLines.length > 0 ? ['throw "Success"'] : [])].join('\n')}</powershell>`;
    }
    addS3DownloadCommand(params) {
        const localPath = (params.localFile && params.localFile.length !== 0) ? params.localFile : `C:/temp/${params.bucketKey}`;
        this.addCommands(`mkdir (Split-Path -Path '${localPath}' ) -ea 0`, `Read-S3Object -BucketName '${params.bucket.bucketName}' -key '${params.bucketKey}' -file '${localPath}' -ErrorAction Stop`);
        return localPath;
    }
    addExecuteFileCommand(params) {
        var _a;
        this.addCommands(`&'${params.filePath}' ${(_a = params.arguments) !== null && _a !== void 0 ? _a : ''}`.trim(), `if (!$?) { Write-Error 'Failed to execute the file "${params.filePath}"' -ErrorAction Stop }`);
    }
    addSignalOnExitCommand(resource) {
        const stack = core_1.Stack.of(resource);
        const resourceID = stack.getLogicalId(resource.node.defaultChild);
        this.addOnExitCommands(`cfn-signal --stack ${stack.stackName} --resource ${resourceID} --region ${stack.region} --success ($success.ToString().ToLower())`);
    }
    renderOnExitLines() {
        if (this.onExitLines.length > 0) {
            return ['trap {', '$success=($PSItem.Exception.Message -eq "Success")', ...this.onExitLines, 'break', '}'];
        }
        return [];
    }
}
/**
 * Custom Instance User Data
 */
class CustomUserData extends UserData {
    constructor() {
        super();
        this.lines = [];
    }
    addCommands(...commands) {
        this.lines.push(...commands);
    }
    addOnExitCommands() {
        throw new Error('CustomUserData does not support addOnExitCommands, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    render() {
        return this.lines.join('\n');
    }
    addS3DownloadCommand() {
        throw new Error('CustomUserData does not support addS3DownloadCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addExecuteFileCommand() {
        throw new Error('CustomUserData does not support addExecuteFileCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
    addSignalOnExitCommand() {
        throw new Error('CustomUserData does not support addSignalOnExitCommand, use UserData.forLinux() or UserData.forWindows() instead.');
    }
}
//# sourceMappingURL=data:application/json;base64,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