"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SecurityGroup = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const SECURITY_GROUP_SYMBOL = Symbol.for('@aws-cdk/iam.SecurityGroup');
/**
 * A SecurityGroup that is not created in this template
 */
class SecurityGroupBase extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, props);
        /**
         * Whether the rule can be inlined into a SecurityGroup or not.
         *
         * @stability stable
         */
        this.canInlineRule = false;
        /**
         * @stability stable
         */
        this.connections = new connections_1.Connections({ securityGroups: [this] });
        Object.defineProperty(this, SECURITY_GROUP_SYMBOL, { value: true });
    }
    /**
     * Return whether the indicated object is a security group.
     *
     * @stability stable
     */
    static isSecurityGroup(x) {
        return SECURITY_GROUP_SYMBOL in x;
    }
    /**
     * A unique identifier for this connection peer.
     *
     * @stability stable
     */
    get uniqueId() {
        return core_1.Names.nodeUniqueId(this.node);
    }
    addIngressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'from', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupIngress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toIngressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    addEgressRule(peer, connection, description, remoteRule) {
        if (description === undefined) {
            description = `to ${peer.uniqueId}:${connection}`;
        }
        const [scope, id] = determineRuleScope(this, peer, connection, 'to', remoteRule);
        // Skip duplicates
        if (scope.node.tryFindChild(id) === undefined) {
            new ec2_generated_1.CfnSecurityGroupEgress(scope, id, {
                groupId: this.securityGroupId,
                ...peer.toEgressRuleConfig(),
                ...connection.toRuleJson(),
                description,
            });
        }
    }
    /**
     * Produce the ingress rule JSON for the given connection.
     *
     * @stability stable
     */
    toIngressRuleConfig() {
        return { sourceSecurityGroupId: this.securityGroupId };
    }
    /**
     * Produce the egress rule JSON for the given connection.
     *
     * @stability stable
     */
    toEgressRuleConfig() {
        return { destinationSecurityGroupId: this.securityGroupId };
    }
}
/**
 * Determine where to parent a new ingress/egress rule
 *
 * A SecurityGroup rule is parented under the group it's related to, UNLESS
 * we're in a cross-stack scenario with another Security Group. In that case,
 * we respect the 'remoteRule' flag and will parent under the other security
 * group.
 *
 * This is necessary to avoid cyclic dependencies between stacks, since both
 * ingress and egress rules will reference both security groups, and a naive
 * parenting will lead to the following situation:
 *
 *   ╔════════════════════╗         ╔════════════════════╗
 *   ║  ┌───────────┐     ║         ║    ┌───────────┐   ║
 *   ║  │  GroupA   │◀────╬─┐   ┌───╬───▶│  GroupB   │   ║
 *   ║  └───────────┘     ║ │   │   ║    └───────────┘   ║
 *   ║        ▲           ║ │   │   ║          ▲         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║        │           ║ │   │   ║          │         ║
 *   ║  ┌───────────┐     ║ └───┼───╬────┌───────────┐   ║
 *   ║  │  EgressA  │─────╬─────┘   ║    │ IngressB  │   ║
 *   ║  └───────────┘     ║         ║    └───────────┘   ║
 *   ║                    ║         ║                    ║
 *   ╚════════════════════╝         ╚════════════════════╝
 *
 * By having the ability to switch the parent, we avoid the cyclic reference by
 * keeping all rules in a single stack.
 *
 * If this happens, we also have to change the construct ID, because
 * otherwise we might have two objects with the same ID if we have
 * multiple reversed security group relationships.
 *
 *   ╔═══════════════════════════════════╗
 *   ║┌───────────┐                      ║
 *   ║│  GroupB   │                      ║
 *   ║└───────────┘                      ║
 *   ║      ▲                            ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├────"from A"──│ IngressB  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      ├─────"to B"───│  EgressA  │ ║
 *   ║      │              └───────────┘ ║
 *   ║      │              ┌───────────┐ ║
 *   ║      └─────"to B"───│  EgressC  │ ║  <-- oops
 *   ║                     └───────────┘ ║
 *   ╚═══════════════════════════════════╝
 */
function determineRuleScope(group, peer, connection, fromTo, remoteRule) {
    if (remoteRule && SecurityGroupBase.isSecurityGroup(peer) && differentStacks(group, peer)) {
        // Reversed
        const reversedFromTo = fromTo === 'from' ? 'to' : 'from';
        return [peer, `${group.uniqueId}:${connection} ${reversedFromTo}`];
    }
    else {
        // Regular (do old ID escaping to in order to not disturb existing deployments)
        return [group, `${fromTo} ${renderPeer(peer)}:${connection}`.replace('/', '_')];
    }
}
function renderPeer(peer) {
    return core_1.Token.isUnresolved(peer.uniqueId) ? '{IndirectPeer}' : peer.uniqueId;
}
function differentStacks(group1, group2) {
    return core_1.Stack.of(group1) !== core_1.Stack.of(group2);
}
/**
 * Creates an Amazon EC2 security group within a VPC.
 *
 * Security Groups act like a firewall with a set of rules, and are associated
 * with any AWS resource that has or creates Elastic Network Interfaces (ENIs).
 * A typical example of a resource that has a security group is an Instance (or
 * Auto Scaling Group of instances)
 *
 * If you are defining new infrastructure in CDK, there is a good chance you
 * won't have to interact with this class at all. Like IAM Roles, Security
 * Groups need to exist to control access between AWS resources, but CDK will
 * automatically generate and populate them with least-privilege permissions
 * for you so you can concentrate on your business logic.
 *
 * All Constructs that require Security Groups will create one for you if you
 * don't specify one at construction. After construction, you can selectively
 * allow connections to and between constructs via--for example-- the `instance.connections`
 * object. Think of it as "allowing connections to your instance", rather than
 * "adding ingress rules a security group". See the [Allowing
 * Connections](https://docs.aws.amazon.com/cdk/api/latest/docs/aws-ec2-readme.html#allowing-connections)
 * section in the library documentation for examples.
 *
 * Direct manipulation of the Security Group through `addIngressRule` and
 * `addEgressRule` is possible, but mutation through the `.connections` object
 * is recommended. If you peer two constructs with security groups this way,
 * appropriate rules will be created in both.
 *
 * If you have an existing security group you want to use in your CDK application,
 * you would import it like this:
 *
 * ```ts
 * const securityGroup = SecurityGroup.fromSecurityGroupId(this, 'SG', 'sg-12345', {
 *    mutable: false
 * });
 * ```
 *
 * @stability stable
 */
class SecurityGroup extends SecurityGroupBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.securityGroupName,
        });
        this.directIngressRules = [];
        this.directEgressRules = [];
        const groupDescription = props.description || this.node.path;
        this.allowAllOutbound = props.allowAllOutbound !== false;
        this.securityGroup = new ec2_generated_1.CfnSecurityGroup(this, 'Resource', {
            groupName: this.physicalName,
            groupDescription,
            securityGroupIngress: core_1.Lazy.any({ produce: () => this.directIngressRules }, { omitEmptyArray: true }),
            securityGroupEgress: core_1.Lazy.any({ produce: () => this.directEgressRules }, { omitEmptyArray: true }),
            vpcId: props.vpc.vpcId,
        });
        this.securityGroupId = this.securityGroup.attrGroupId;
        this.securityGroupVpcId = this.securityGroup.attrVpcId;
        this.securityGroupName = this.securityGroup.ref;
        this.addDefaultEgressRule();
    }
    /**
     * Look up a security group by id.
     *
     * @stability stable
     */
    static fromLookup(scope, id, securityGroupId) {
        if (core_1.Token.isUnresolved(securityGroupId)) {
            throw new Error('All arguments to look up a security group must be concrete (no Tokens)');
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.SECURITY_GROUP_PROVIDER,
            props: { securityGroupId },
            dummyValue: {
                securityGroupId: 'sg-12345',
                allowAllOutbound: true,
            },
        }).value;
        return SecurityGroup.fromSecurityGroupId(scope, id, attributes.securityGroupId, {
            allowAllOutbound: attributes.allowAllOutbound,
            mutable: true,
        });
    }
    /**
     * Import an existing security group into this app.
     *
     * This method will assume that the Security Group has a rule in it which allows
     * all outbound traffic, and so will not add egress rules to the imported Security
     * Group (only ingress rules).
     *
     * If your existing Security Group needs to have egress rules added, pass the
     * `allowAllOutbound: false` option on import.
     *
     * @stability stable
     */
    static fromSecurityGroupId(scope, id, securityGroupId, options = {}) {
        class MutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(peer, connection, description, remoteRule) {
                // Only if allowAllOutbound has been disabled
                if (options.allowAllOutbound === false) {
                    super.addEgressRule(peer, connection, description, remoteRule);
                }
            }
        }
        class ImmutableImport extends SecurityGroupBase {
            constructor() {
                var _a;
                super(...arguments);
                this.securityGroupId = securityGroupId;
                this.allowAllOutbound = (_a = options.allowAllOutbound) !== null && _a !== void 0 ? _a : true;
            }
            addEgressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
            addIngressRule(_peer, _connection, _description, _remoteRule) {
                // do nothing
            }
        }
        return options.mutable !== false
            ? new MutableImport(scope, id)
            : new ImmutableImport(scope, id);
    }
    /**
     * Add an ingress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addIngressRule(peer, connection, description, remoteRule) {
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addIngressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        this.addDirectIngressRule({
            ...peer.toIngressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        });
    }
    /**
     * Add an egress rule for the current security group.
     *
     * `remoteRule` controls where the Rule object is created if the peer is also a
     * securityGroup and they are in different stack. If false (default) the
     * rule object is created under the current SecurityGroup object. If true and the
     * peer is also a SecurityGroup, the rule object is created under the remote
     * SecurityGroup object.
     *
     * @stability stable
     */
    addEgressRule(peer, connection, description, remoteRule) {
        if (this.allowAllOutbound) {
            // In the case of "allowAllOutbound", we don't add any more rules. There
            // is only one rule which allows all traffic and that subsumes any other
            // rule.
            core_1.Annotations.of(this).addWarning('Ignoring Egress rule since \'allowAllOutbound\' is set to true; To add customize rules, set allowAllOutbound=false on the SecurityGroup');
            return;
        }
        else {
            // Otherwise, if the bogus rule exists we can now remove it because the
            // presence of any other rule will get rid of EC2's implicit "all
            // outbound" rule anyway.
            this.removeNoTrafficRule();
        }
        if (!peer.canInlineRule || !connection.canInlineRule) {
            super.addEgressRule(peer, connection, description, remoteRule);
            return;
        }
        if (description === undefined) {
            description = `from ${peer.uniqueId}:${connection}`;
        }
        const rule = {
            ...peer.toEgressRuleConfig(),
            ...connection.toRuleJson(),
            description,
        };
        if (isAllTrafficRule(rule)) {
            // We cannot allow this; if someone adds the rule in this way, it will be
            // removed again if they add other rules. We also can't automatically switch
            // to "allOutbound=true" mode, because we might have already emitted
            // EgressRule objects (which count as rules added later) and there's no way
            // to recall those. Better to prevent this for now.
            throw new Error('Cannot add an "all traffic" egress rule in this way; set allowAllOutbound=true on the SecurityGroup instead.');
        }
        this.addDirectEgressRule(rule);
    }
    /**
     * Add a direct ingress rule
     */
    addDirectIngressRule(rule) {
        if (!this.hasIngressRule(rule)) {
            this.directIngressRules.push(rule);
        }
    }
    /**
     * Return whether the given ingress rule exists on the group
     */
    hasIngressRule(rule) {
        return this.directIngressRules.findIndex(r => ingressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add a direct egress rule
     */
    addDirectEgressRule(rule) {
        if (!this.hasEgressRule(rule)) {
            this.directEgressRules.push(rule);
        }
    }
    /**
     * Return whether the given egress rule exists on the group
     */
    hasEgressRule(rule) {
        return this.directEgressRules.findIndex(r => egressRulesEqual(r, rule)) > -1;
    }
    /**
     * Add the default egress rule to the securityGroup
     *
     * This depends on allowAllOutbound:
     *
     * - If allowAllOutbound is true, we *TECHNICALLY* don't need to do anything, because
     *   EC2 is going to create this default rule anyway. But, for maximum readability
     *   of the template, we will add one anyway.
     * - If allowAllOutbound is false, we add a bogus rule that matches no traffic in
     *   order to get rid of the default "all outbound" rule that EC2 creates by default.
     *   If other rules happen to get added later, we remove the bogus rule again so
     *   that it doesn't clutter up the template too much (even though that's not
     *   strictly necessary).
     */
    addDefaultEgressRule() {
        if (this.allowAllOutbound) {
            this.directEgressRules.push(ALLOW_ALL_RULE);
        }
        else {
            this.directEgressRules.push(MATCH_NO_TRAFFIC);
        }
    }
    /**
     * Remove the bogus rule if it exists
     */
    removeNoTrafficRule() {
        const i = this.directEgressRules.findIndex(r => egressRulesEqual(r, MATCH_NO_TRAFFIC));
        if (i > -1) {
            this.directEgressRules.splice(i, 1);
        }
    }
}
exports.SecurityGroup = SecurityGroup;
/**
 * Egress rule that purposely matches no traffic
 *
 * This is used in order to disable the "all traffic" default of Security Groups.
 *
 * No machine can ever actually have the 255.255.255.255 IP address, but
 * in order to lock it down even more we'll restrict to a nonexistent
 * ICMP traffic type.
 */
const MATCH_NO_TRAFFIC = {
    cidrIp: '255.255.255.255/32',
    description: 'Disallow all traffic',
    ipProtocol: 'icmp',
    fromPort: 252,
    toPort: 86,
};
/**
 * Egress rule that matches all traffic
 */
const ALLOW_ALL_RULE = {
    cidrIp: '0.0.0.0/0',
    description: 'Allow all outbound traffic by default',
    ipProtocol: '-1',
};
/**
 * Compare two ingress rules for equality the same way CloudFormation would (discarding description)
 */
function ingressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.sourceSecurityGroupId === b.sourceSecurityGroupId
        && a.sourceSecurityGroupName === b.sourceSecurityGroupName
        && a.sourceSecurityGroupOwnerId === b.sourceSecurityGroupOwnerId;
}
/**
 * Compare two egress rules for equality the same way CloudFormation would (discarding description)
 */
function egressRulesEqual(a, b) {
    return a.cidrIp === b.cidrIp
        && a.cidrIpv6 === b.cidrIpv6
        && a.fromPort === b.fromPort
        && a.toPort === b.toPort
        && a.ipProtocol === b.ipProtocol
        && a.destinationPrefixListId === b.destinationPrefixListId
        && a.destinationSecurityGroupId === b.destinationSecurityGroupId;
}
/**
 * Whether this rule refers to all traffic
 */
function isAllTrafficRule(rule) {
    return rule.cidrIp === '0.0.0.0/0' && rule.ipProtocol === '-1';
}
//# sourceMappingURL=data:application/json;base64,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