/**
 * Protocol for use in Connection Rules.
 *
 * @stability stable
 */
export declare enum Protocol {
    /**
     * @stability stable
     */
    ALL = "-1",
    /**
     * @stability stable
     */
    TCP = "tcp",
    /**
     * @stability stable
     */
    UDP = "udp",
    /**
     * @stability stable
     */
    ICMP = "icmp",
    /**
     * @stability stable
     */
    ICMPV6 = "58"
}
/**
 * Properties to create a port range.
 *
 * @stability stable
 */
export interface PortProps {
    /**
     * The protocol for the range.
     *
     * @stability stable
     */
    readonly protocol: Protocol;
    /**
     * The starting port for the range.
     *
     * @default - Not included in the rule
     * @stability stable
     */
    readonly fromPort?: number;
    /**
     * The ending port for the range.
     *
     * @default - Not included in the rule
     * @stability stable
     */
    readonly toPort?: number;
    /**
     * String representation for this object.
     *
     * @stability stable
     */
    readonly stringRepresentation: string;
}
/**
 * Interface for classes that provide the connection-specification parts of a security group rule.
 *
 * @stability stable
 */
export declare class Port {
    private readonly props;
    /**
     * A single TCP port.
     *
     * @stability stable
     */
    static tcp(port: number): Port;
    /**
     * A TCP port range.
     *
     * @stability stable
     */
    static tcpRange(startPort: number, endPort: number): Port;
    /**
     * Any TCP traffic.
     *
     * @stability stable
     */
    static allTcp(): Port;
    /**
     * A single UDP port.
     *
     * @stability stable
     */
    static udp(port: number): Port;
    /**
     * A UDP port range.
     *
     * @stability stable
     */
    static udpRange(startPort: number, endPort: number): Port;
    /**
     * Any UDP traffic.
     *
     * @stability stable
     */
    static allUdp(): Port;
    /**
     * A specific combination of ICMP type and code.
     *
     * @see https://www.iana.org/assignments/icmp-parameters/icmp-parameters.xhtml
     * @stability stable
     */
    static icmpTypeAndCode(type: number, code: number): Port;
    /**
     * All codes for a single ICMP type.
     *
     * @stability stable
     */
    static icmpType(type: number): Port;
    /**
     * ICMP ping (echo) traffic.
     *
     * @stability stable
     */
    static icmpPing(): Port;
    /**
     * All ICMP traffic.
     *
     * @stability stable
     */
    static allIcmp(): Port;
    /**
     * All traffic.
     *
     * @stability stable
     */
    static allTraffic(): Port;
    /**
     * Whether the rule containing this port range can be inlined into a securitygroup or not.
     *
     * @stability stable
     */
    readonly canInlineRule: boolean;
    /**
     * @stability stable
     */
    constructor(props: PortProps);
    /**
     * Produce the ingress/egress rule JSON for the given connection.
     *
     * @stability stable
     */
    toRuleJson(): any;
    /**
     * @stability stable
     */
    toString(): string;
}
