"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubnetNetworkAclAssociation = exports.NetworkAclEntry = exports.TrafficDirection = exports.Action = exports.NetworkAcl = void 0;
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * (experimental) Add a new entry to the ACL.
     *
     * @experimental
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, {
            networkAcl: this,
            ...options,
        });
    }
}
/**
 * (experimental) Define a new custom network ACL.
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName,
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * (experimental) Import an existing NetworkAcl into this app.
     *
     * @experimental
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * (experimental) Associate the ACL with a given set of subnets.
     *
     * @experimental
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * (experimental) What action to apply to traffic matching the ACL.
 *
 * @experimental
 */
var Action;
(function (Action) {
    Action["ALLOW"] = "allow";
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * (experimental) Direction of traffic the AclEntry applies to.
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * (experimental) Define an entry in a Network ACL table.
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.networkAclEntryName,
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', {
            networkAclId: this.networkAcl.networkAclId,
            ruleNumber: props.ruleNumber,
            ruleAction: (_a = props.ruleAction) !== null && _a !== void 0 ? _a : Action.ALLOW,
            egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined,
            ...props.traffic.toTrafficConfig(),
            ...props.cidr.toCidrConfig(),
        });
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
/**
 * @stability stable
 */
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName,
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId,
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    /**
     * @stability stable
     */
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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