"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LaunchTemplate = exports.LaunchTemplateSpecialVersions = exports.SpotRequestType = exports.SpotInstanceInterruption = exports.InstanceInitiatedShutdownBehavior = exports.CpuCredits = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const ebs_util_1 = require("./private/ebs-util");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
// dev-note: This could be used in the Instance L2
/**
 * Provides the options for specifying the CPU credit type for burstable EC2 instance types (T2, T3, T3a, etc).
 *
 * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/burstable-performance-instances-how-to.html
 * @stability stable
 */
var CpuCredits;
(function (CpuCredits) {
    CpuCredits["STANDARD"] = "standard";
    CpuCredits["UNLIMITED"] = "unlimited";
})(CpuCredits = exports.CpuCredits || (exports.CpuCredits = {}));
;
// dev-note: This could be used in the Instance L2
/**
 * Provides the options for specifying the instance initiated shutdown behavior.
 *
 * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/terminating-instances.html#Using_ChangingInstanceInitiatedShutdownBehavior
 * @stability stable
 */
var InstanceInitiatedShutdownBehavior;
(function (InstanceInitiatedShutdownBehavior) {
    InstanceInitiatedShutdownBehavior["STOP"] = "stop";
    InstanceInitiatedShutdownBehavior["TERMINATE"] = "terminate";
})(InstanceInitiatedShutdownBehavior = exports.InstanceInitiatedShutdownBehavior || (exports.InstanceInitiatedShutdownBehavior = {}));
;
// dev-note: This could be used in a SpotFleet L2 if one gets developed.
/**
 * Provides the options for the types of interruption for spot instances.
 *
 * @stability stable
 */
var SpotInstanceInterruption;
(function (SpotInstanceInterruption) {
    SpotInstanceInterruption["STOP"] = "stop";
    SpotInstanceInterruption["TERMINATE"] = "terminate";
    SpotInstanceInterruption["HIBERNATE"] = "hibernate";
})(SpotInstanceInterruption = exports.SpotInstanceInterruption || (exports.SpotInstanceInterruption = {}));
/**
 * The Spot Instance request type.
 *
 * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-requests.html
 * @stability stable
 */
var SpotRequestType;
(function (SpotRequestType) {
    SpotRequestType["ONE_TIME"] = "one-time";
    SpotRequestType["PERSISTENT"] = "persistent";
})(SpotRequestType = exports.SpotRequestType || (exports.SpotRequestType = {}));
;
/**
 * A class that provides convenient access to special version tokens for LaunchTemplate versions.
 *
 * @stability stable
 */
class LaunchTemplateSpecialVersions {
}
exports.LaunchTemplateSpecialVersions = LaunchTemplateSpecialVersions;
/**
 * The special value that denotes that users of a Launch Template should reference the LATEST version of the template.
 *
 * @stability stable
 */
LaunchTemplateSpecialVersions.LATEST_VERSION = '$Latest';
/**
 * The special value that denotes that users of a Launch Template should reference the DEFAULT version of the template.
 *
 * @stability stable
 */
LaunchTemplateSpecialVersions.DEFAULT_VERSION = '$Default';
/**
 * This represents an EC2 LaunchTemplate.
 *
 * @see https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/ec2-launch-templates.html
 * @stability stable
 */
class LaunchTemplate extends core_1.Resource {
    // =============================================
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        // Basic validation of the provided spot block duration
        const spotDuration = (_b = (_a = props === null || props === void 0 ? void 0 : props.spotOptions) === null || _a === void 0 ? void 0 : _a.blockDuration) === null || _b === void 0 ? void 0 : _b.toHours({ integral: true });
        if (spotDuration !== undefined && (spotDuration < 1 || spotDuration > 6)) {
            // See: https://docs.aws.amazon.com/AWSEC2/latest/UserGuide/spot-requests.html#fixed-duration-spot-instances
            core_1.Annotations.of(this).addError('Spot block duration must be exactly 1, 2, 3, 4, 5, or 6 hours.');
        }
        this.role = props.role;
        this._grantPrincipal = this.role;
        const iamProfile = this.role ? new iam.CfnInstanceProfile(this, 'Profile', {
            roles: [this.role.roleName],
        }) : undefined;
        if (props.securityGroup) {
            this._connections = new connections_1.Connections({ securityGroups: [props.securityGroup] });
        }
        const securityGroupsToken = core_1.Lazy.list({
            produce: () => {
                if (this._connections && this._connections.securityGroups.length > 0) {
                    return this._connections.securityGroups.map(sg => sg.securityGroupId);
                }
                return undefined;
            },
        });
        if (props.userData) {
            this.userData = props.userData;
        }
        const userDataToken = core_1.Lazy.string({
            produce: () => {
                if (this.userData) {
                    return core_1.Fn.base64(this.userData.render());
                }
                return undefined;
            },
        });
        const imageConfig = (_c = props.machineImage) === null || _c === void 0 ? void 0 : _c.getImage(this);
        if (imageConfig) {
            this.osType = imageConfig.osType;
        }
        let marketOptions = undefined;
        if (props === null || props === void 0 ? void 0 : props.spotOptions) {
            marketOptions = {
                marketType: 'spot',
                spotOptions: {
                    blockDurationMinutes: spotDuration !== undefined ? spotDuration * 60 : undefined,
                    instanceInterruptionBehavior: props.spotOptions.interruptionBehavior,
                    maxPrice: (_d = props.spotOptions.maxPrice) === null || _d === void 0 ? void 0 : _d.toString(),
                    spotInstanceType: props.spotOptions.requestType,
                    validUntil: (_e = props.spotOptions.validUntil) === null || _e === void 0 ? void 0 : _e.date.toUTCString(),
                },
            };
            // Remove SpotOptions if there are none.
            if (Object.keys(marketOptions.spotOptions).filter(k => marketOptions.spotOptions[k]).length == 0) {
                marketOptions.spotOptions = undefined;
            }
        }
        this.tags = new core_1.TagManager(core_1.TagType.KEY_VALUE, 'AWS::EC2::LaunchTemplate');
        const tagsToken = core_1.Lazy.any({
            produce: () => {
                if (this.tags.hasTags()) {
                    const renderedTags = this.tags.renderTags();
                    const lowerCaseRenderedTags = renderedTags.map((tag) => {
                        return {
                            key: tag.Key,
                            value: tag.Value,
                        };
                    });
                    return [
                        {
                            resourceType: 'instance',
                            tags: lowerCaseRenderedTags,
                        },
                        {
                            resourceType: 'volume',
                            tags: lowerCaseRenderedTags,
                        },
                    ];
                }
                return undefined;
            },
        });
        const resource = new ec2_generated_1.CfnLaunchTemplate(this, 'Resource', {
            launchTemplateName: props === null || props === void 0 ? void 0 : props.launchTemplateName,
            launchTemplateData: {
                blockDeviceMappings: (props === null || props === void 0 ? void 0 : props.blockDevices) !== undefined ? ebs_util_1.launchTemplateBlockDeviceMappings(this, props.blockDevices) : undefined,
                creditSpecification: (props === null || props === void 0 ? void 0 : props.cpuCredits) !== undefined ? {
                    cpuCredits: props.cpuCredits,
                } : undefined,
                disableApiTermination: props === null || props === void 0 ? void 0 : props.disableApiTermination,
                ebsOptimized: props === null || props === void 0 ? void 0 : props.ebsOptimized,
                enclaveOptions: (props === null || props === void 0 ? void 0 : props.nitroEnclaveEnabled) !== undefined ? {
                    enabled: props.nitroEnclaveEnabled,
                } : undefined,
                hibernationOptions: (props === null || props === void 0 ? void 0 : props.hibernationConfigured) !== undefined ? {
                    configured: props.hibernationConfigured,
                } : undefined,
                iamInstanceProfile: iamProfile !== undefined ? {
                    arn: iamProfile.getAtt('Arn').toString(),
                } : undefined,
                imageId: imageConfig === null || imageConfig === void 0 ? void 0 : imageConfig.imageId,
                instanceType: (_f = props === null || props === void 0 ? void 0 : props.instanceType) === null || _f === void 0 ? void 0 : _f.toString(),
                instanceInitiatedShutdownBehavior: props === null || props === void 0 ? void 0 : props.instanceInitiatedShutdownBehavior,
                instanceMarketOptions: marketOptions,
                keyName: props === null || props === void 0 ? void 0 : props.keyName,
                monitoring: (props === null || props === void 0 ? void 0 : props.detailedMonitoring) !== undefined ? {
                    enabled: props.detailedMonitoring,
                } : undefined,
                securityGroupIds: securityGroupsToken,
                tagSpecifications: tagsToken,
                userData: userDataToken,
            },
        });
        core_1.Tags.of(this).add(NAME_TAG, this.node.path);
        this.defaultVersionNumber = resource.attrDefaultVersionNumber;
        this.latestVersionNumber = resource.attrLatestVersionNumber;
        this.launchTemplateId = resource.ref;
        this.versionNumber = core_1.Token.asString(resource.getAtt('LatestVersionNumber'));
    }
    /**
     * Import an existing LaunchTemplate.
     *
     * @stability stable
     */
    static fromLaunchTemplateAttributes(scope, id, attrs) {
        const haveId = Boolean(attrs.launchTemplateId);
        const haveName = Boolean(attrs.launchTemplateName);
        if (haveId == haveName) {
            throw new Error('LaunchTemplate.fromLaunchTemplateAttributes() requires exactly one of launchTemplateId or launchTemplateName be provided.');
        }
        class Import extends core_1.Resource {
            constructor() {
                var _a;
                super(...arguments);
                this.versionNumber = (_a = attrs.versionNumber) !== null && _a !== void 0 ? _a : LaunchTemplateSpecialVersions.DEFAULT_VERSION;
                this.launchTemplateId = attrs.launchTemplateId;
                this.launchTemplateName = attrs.launchTemplateName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Allows specifying security group connections for the instance.
     *
     * @stability stable
     * @note Only available if you provide a securityGroup when constructing the LaunchTemplate.
     */
    get connections() {
        if (!this._connections) {
            throw new Error('LaunchTemplate can only be used as IConnectable if a securityGroup is provided when contructing it.');
        }
        return this._connections;
    }
    /**
     * Principal to grant permissions to.
     *
     * @stability stable
     * @note Only available if you provide a role when constructing the LaunchTemplate.
     */
    get grantPrincipal() {
        if (!this._grantPrincipal) {
            throw new Error('LaunchTemplate can only be used as IGrantable if a role is provided when constructing it.');
        }
        return this._grantPrincipal;
    }
}
exports.LaunchTemplate = LaunchTemplate;
//# sourceMappingURL=data:application/json;base64,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