"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.InstanceType = exports.InstanceSize = exports.InstanceArchitecture = exports.InstanceClass = void 0;
/**
 * What class and generation of instance to use.
 *
 * We have both symbolic and concrete enums for every type.
 *
 * The first are for people that want to specify by purpose,
 * the second one are for people who already know exactly what
 * 'R4' means.
 *
 * @stability stable
 */
var InstanceClass;
(function (InstanceClass) {
    InstanceClass["STANDARD3"] = "m3";
    /**
     * Standard instances, 3rd generation
     */
    InstanceClass["M3"] = "m3";
    InstanceClass["STANDARD4"] = "m4";
    /**
     * Standard instances, 4th generation
     */
    InstanceClass["M4"] = "m4";
    InstanceClass["STANDARD5"] = "m5";
    /**
     * Standard instances, 5th generation
     */
    InstanceClass["M5"] = "m5";
    InstanceClass["STANDARD5_NVME_DRIVE"] = "m5d";
    /**
     * Standard instances with local NVME drive, 5th generation
     */
    InstanceClass["M5D"] = "m5d";
    InstanceClass["STANDARD5_AMD"] = "m5a";
    /**
     * Standard instances based on AMD EPYC, 5th generation
     */
    InstanceClass["M5A"] = "m5a";
    InstanceClass["STANDARD5_AMD_NVME_DRIVE"] = "m5ad";
    /**
     * Standard instances based on AMD EPYC with local NVME drive, 5th generation
     */
    InstanceClass["M5AD"] = "m5ad";
    InstanceClass["MEMORY3"] = "r3";
    /**
     * Memory optimized instances, 3rd generation
     */
    InstanceClass["R3"] = "r3";
    InstanceClass["MEMORY4"] = "r4";
    /**
     * Memory optimized instances, 4th generation
     */
    InstanceClass["R4"] = "r4";
    InstanceClass["MEMORY5"] = "r5";
    /**
     * Memory optimized instances, 5th generation
     */
    InstanceClass["R5"] = "r5";
    InstanceClass["MEMORY5_HIGH_PERFORMANCE"] = "r5n";
    /**
     * Memory optimized instances for high performance computing, 5th generation
     */
    InstanceClass["R5N"] = "r5n";
    InstanceClass["MEMORY5_NVME_DRIVE"] = "r5d";
    /**
     * Memory optimized instances with local NVME drive, 5th generation
     */
    InstanceClass["R5D"] = "r5d";
    InstanceClass["MEMORY5_NVME_DRIVE_HIGH_PERFORMANCE"] = "r5dn";
    /**
     * Memory optimized instances with local NVME drive for high performance computing, 5th generation
     */
    InstanceClass["R5DN"] = "r5dn";
    InstanceClass["MEMORY5_AMD"] = "r5a";
    /**
     * Memory optimized instances based on AMD EPYC, 5th generation
     */
    InstanceClass["R5A"] = "r5a";
    /**
     * Memory optimized instances based on AMD EPYC with local NVME drive, 5th generation
     */
    InstanceClass["MEMORY5_AMD_NVME_DRIVE"] = "r5a";
    /**
     * Memory optimized instances based on AMD EPYC with local NVME drive, 5th generation
     */
    InstanceClass["R5AD"] = "r5a";
    InstanceClass["MEMORY5_EBS_OPTIMIZED"] = "r5b";
    /**
     * Memory optimized instances that are also EBS-optimized, 5th generation
     */
    InstanceClass["R5B"] = "r5b";
    InstanceClass["MEMORY6_GRAVITON"] = "r6g";
    /**
     * Memory optimized instances, 6th generation with Graviton2 processors
     */
    InstanceClass["R6G"] = "r6g";
    InstanceClass["MEMORY6_GRAVITON2_NVME_DRIVE"] = "r6gd";
    /**
     * Memory optimized instances, 6th generation with Graviton2 processors and local NVME drive
     */
    InstanceClass["R6GD"] = "r6gd";
    InstanceClass["COMPUTE3"] = "c3";
    /**
     * Compute optimized instances, 3rd generation
     */
    InstanceClass["C3"] = "c3";
    InstanceClass["COMPUTE4"] = "c4";
    /**
     * Compute optimized instances, 4th generation
     */
    InstanceClass["C4"] = "c4";
    InstanceClass["COMPUTE5"] = "c5";
    /**
     * Compute optimized instances, 5th generation
     */
    InstanceClass["C5"] = "c5";
    InstanceClass["COMPUTE5_NVME_DRIVE"] = "c5d";
    InstanceClass["COMPUTE5_AMD"] = "c5a";
    /**
     * Compute optimized instances based on AMD EPYC, 5th generation
     */
    InstanceClass["C5A"] = "c5a";
    /**
     * Compute optimized instances with local NVME drive, 5th generation
     */
    InstanceClass["C5D"] = "c5d";
    InstanceClass["COMPUTE5_HIGH_PERFORMANCE"] = "c5n";
    /**
     * Compute optimized instances for high performance computing, 5th generation
     */
    InstanceClass["C5N"] = "c5n";
    InstanceClass["COMPUTE6_GRAVITON2"] = "c6g";
    /**
     * Compute optimized instances for high performance computing, 6th generation with Graviton2 processors
     */
    InstanceClass["C6G"] = "c6g";
    InstanceClass["COMPUTE6_GRAVITON2_NVME_DRIVE"] = "c6gd";
    /**
     * Compute optimized instances for high performance computing, 6th generation with Graviton2 processors
     * and local NVME drive
     */
    InstanceClass["C6GD"] = "c6gd";
    InstanceClass["COMPUTE6_GRAVITON2_HIGH_NETWORK_BANDWITH"] = "c6gn";
    /**
     * Compute optimized instances for high performance computing, 6th generation with Graviton2 processors
     * and high network bandwidth capabilities
     */
    InstanceClass["C6GN"] = "c6gn";
    InstanceClass["STORAGE2"] = "d2";
    /**
     * Storage-optimized instances, 2nd generation
     */
    InstanceClass["D2"] = "d2";
    InstanceClass["STORAGE_COMPUTE_1"] = "h1";
    /**
     * Storage/compute balanced instances, 1st generation
     */
    InstanceClass["H1"] = "h1";
    InstanceClass["IO3"] = "i3";
    /**
     * I/O-optimized instances, 3rd generation
     */
    InstanceClass["I3"] = "i3";
    InstanceClass["IO3_DENSE_NVME_DRIVE"] = "i3en";
    /**
     * I/O-optimized instances with local NVME drive, 3rd generation
     */
    InstanceClass["I3EN"] = "i3en";
    InstanceClass["BURSTABLE2"] = "t2";
    /**
     * Burstable instances, 2nd generation
     */
    InstanceClass["T2"] = "t2";
    InstanceClass["BURSTABLE3"] = "t3";
    /**
     * Burstable instances, 3rd generation
     */
    InstanceClass["T3"] = "t3";
    InstanceClass["BURSTABLE3_AMD"] = "t3a";
    /**
     * Burstable instances based on AMD EPYC, 3rd generation
     */
    InstanceClass["T3A"] = "t3a";
    InstanceClass["BURSTABLE4_GRAVITON"] = "t4g";
    /**
     * Burstable instances, 4th generation with Graviton2 processors
     */
    InstanceClass["T4G"] = "t4g";
    InstanceClass["MEMORY_INTENSIVE_1"] = "x1";
    /**
     * Memory-intensive instances, 1st generation
     */
    InstanceClass["X1"] = "x1";
    InstanceClass["MEMORY_INTENSIVE_1_EXTENDED"] = "x1e";
    /**
     * Memory-intensive instances, 1st generation
     */
    InstanceClass["X1E"] = "x1e";
    InstanceClass["FPGA1"] = "f1";
    /**
     * Instances with customizable hardware acceleration, 1st generation
     */
    InstanceClass["F1"] = "f1";
    InstanceClass["GRAPHICS3"] = "g3";
    /**
     * Graphics-optimized instances, 3rd generation
     */
    InstanceClass["G3"] = "g3";
    InstanceClass["GRAPHICS4_NVME_DRIVE_HIGH_PERFORMANCE"] = "g4dn";
    /**
     * Graphics-optimized instances with NVME drive for high performance computing, 4th generation
     */
    InstanceClass["G4DN"] = "g4dn";
    InstanceClass["PARALLEL2"] = "p2";
    /**
     * Parallel-processing optimized instances, 2nd generation
     */
    InstanceClass["P2"] = "p2";
    InstanceClass["PARALLEL3"] = "p3";
    /**
     * Parallel-processing optimized instances, 3nd generation
     */
    InstanceClass["P3"] = "p3";
    InstanceClass["ARM1"] = "a1";
    /**
     * Arm processor based instances, 1st generation
     */
    InstanceClass["A1"] = "a1";
    InstanceClass["STANDARD6_GRAVITON"] = "m6g";
    /**
     * Arm processor based instances, 2nd generation
     */
    InstanceClass["M6G"] = "m6g";
    InstanceClass["STANDARD6_GRAVITON2_NVME_DRIVE"] = "m6gd";
    /**
     * Standard instances, 6th generation with Graviton2 processors and local NVME drive
     */
    InstanceClass["M6GD"] = "m6gd";
    InstanceClass["HIGH_COMPUTE_MEMORY1"] = "z1d";
    /**
     * High memory and compute capacity instances, 1st generation
     */
    InstanceClass["Z1D"] = "z1d";
    InstanceClass["INFERENCE1"] = "inf1";
    /**
     * Inferentia Chips based instances for machine learning inference applications, 1st generation
     */
    InstanceClass["INF1"] = "inf1";
})(InstanceClass = exports.InstanceClass || (exports.InstanceClass = {}));
/**
 * Identifies an instance's CPU architecture.
 *
 * @stability stable
 */
var InstanceArchitecture;
(function (InstanceArchitecture) {
    InstanceArchitecture["ARM_64"] = "arm64";
    InstanceArchitecture["X86_64"] = "x86_64";
})(InstanceArchitecture = exports.InstanceArchitecture || (exports.InstanceArchitecture = {}));
/**
 * What size of instance to use.
 *
 * @stability stable
 */
var InstanceSize;
(function (InstanceSize) {
    InstanceSize["NANO"] = "nano";
    InstanceSize["MICRO"] = "micro";
    InstanceSize["SMALL"] = "small";
    InstanceSize["MEDIUM"] = "medium";
    InstanceSize["LARGE"] = "large";
    InstanceSize["XLARGE"] = "xlarge";
    InstanceSize["XLARGE2"] = "2xlarge";
    InstanceSize["XLARGE4"] = "4xlarge";
    InstanceSize["XLARGE6"] = "6xlarge";
    InstanceSize["XLARGE8"] = "8xlarge";
    InstanceSize["XLARGE9"] = "9xlarge";
    InstanceSize["XLARGE10"] = "10xlarge";
    InstanceSize["XLARGE12"] = "12xlarge";
    InstanceSize["XLARGE16"] = "16xlarge";
    InstanceSize["XLARGE18"] = "18xlarge";
    InstanceSize["XLARGE24"] = "24xlarge";
    InstanceSize["XLARGE32"] = "32xlarge";
    InstanceSize["METAL"] = "metal";
})(InstanceSize = exports.InstanceSize || (exports.InstanceSize = {}));
/**
 * Instance type for EC2 instances.
 *
 * This class takes a literal string, good if you already
 * know the identifier of the type you want.
 *
 * @stability stable
 */
class InstanceType {
    /**
     * @stability stable
     */
    constructor(instanceTypeIdentifier) {
        this.instanceTypeIdentifier = instanceTypeIdentifier;
    }
    /**
     * Instance type for EC2 instances.
     *
     * This class takes a combination of a class and size.
     *
     * Be aware that not all combinations of class and size are available, and not all
     * classes are available in all regions.
     *
     * @stability stable
     */
    static of(instanceClass, instanceSize) {
        return new InstanceType(`${instanceClass}.${instanceSize}`);
    }
    /**
     * Return the instance type as a dotted string.
     *
     * @stability stable
     */
    toString() {
        return this.instanceTypeIdentifier;
    }
    /**
     * The instance's CPU architecture.
     *
     * @stability stable
     */
    get architecture() {
        // capture the family, generation, capabilities, and size portions of the instance type id
        const instanceTypeComponents = this.instanceTypeIdentifier.match(/^([a-z]+)(\d{1,2})([a-z]*)\.([a-z0-9]+)$/);
        if (instanceTypeComponents == null) {
            throw new Error('Malformed instance type identifier');
        }
        const family = instanceTypeComponents[1];
        const capabilities = instanceTypeComponents[3];
        // Instance family `a` are first-gen Graviton instances
        // Capability `g` indicates the instance is Graviton2 powered
        if (family === 'a' || capabilities.includes('g')) {
            return InstanceArchitecture.ARM_64;
        }
        return InstanceArchitecture.X86_64;
    }
}
exports.InstanceType = InstanceType;
//# sourceMappingURL=data:application/json;base64,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