"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BastionHostLinux = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const _1 = require(".");
const instance_1 = require("./instance");
const machine_image_1 = require("./machine-image");
const port_1 = require("./port");
/**
 * (experimental) This creates a linux bastion host you can use to connect to other instances or services in your VPC.
 *
 * The recommended way to connect to the bastion host is by using AWS Systems Manager Session Manager.
 *
 * The operating system is Amazon Linux 2 with the latest SSM agent installed
 *
 * You can also configure this bastion host to allow connections via SSH
 *
 * @experimental
 * @resource AWS::EC2::Instance
 */
class BastionHostLinux extends core_1.Resource {
    /**
     * @experimental
     */
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        const instanceType = (_a = props.instanceType) !== null && _a !== void 0 ? _a : _1.InstanceType.of(_1.InstanceClass.T3, _1.InstanceSize.NANO);
        this.instance = new instance_1.Instance(this, 'Resource', {
            vpc: props.vpc,
            availabilityZone: props.availabilityZone,
            securityGroup: props.securityGroup,
            instanceName: (_b = props.instanceName) !== null && _b !== void 0 ? _b : 'BastionHost',
            instanceType,
            machineImage: (_c = props.machineImage) !== null && _c !== void 0 ? _c : machine_image_1.MachineImage.latestAmazonLinux({
                generation: _1.AmazonLinuxGeneration.AMAZON_LINUX_2,
                cpuType: this.toAmazonLinuxCpuType(instanceType.architecture),
            }),
            vpcSubnets: (_d = props.subnetSelection) !== null && _d !== void 0 ? _d : {},
            blockDevices: (_e = props.blockDevices) !== null && _e !== void 0 ? _e : undefined,
        });
        this.instance.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ssmmessages:*',
                'ssm:UpdateInstanceInformation',
                'ec2messages:*',
            ],
            resources: ['*'],
        }));
        this.connections = this.instance.connections;
        this.role = this.instance.role;
        this.grantPrincipal = this.instance.role;
        this.instanceId = this.instance.instanceId;
        this.instancePrivateIp = this.instance.instancePrivateIp;
        this.instanceAvailabilityZone = this.instance.instanceAvailabilityZone;
        this.instancePrivateDnsName = this.instance.instancePrivateDnsName;
        this.instancePublicIp = this.instance.instancePublicIp;
        this.instancePublicDnsName = this.instance.instancePublicDnsName;
        new core_1.CfnOutput(this, 'BastionHostId', {
            description: 'Instance ID of the bastion host. Use this to connect via SSM Session Manager',
            value: this.instanceId,
        });
    }
    /**
     * Returns the AmazonLinuxCpuType corresponding to the given instance architecture
     * @param architecture the instance architecture value to convert
     */
    toAmazonLinuxCpuType(architecture) {
        if (architecture === _1.InstanceArchitecture.ARM_64) {
            return machine_image_1.AmazonLinuxCpuType.ARM_64;
        }
        else if (architecture === _1.InstanceArchitecture.X86_64) {
            return machine_image_1.AmazonLinuxCpuType.X86_64;
        }
        throw new Error(`Unsupported instance architecture '${architecture}'`);
    }
    /**
     * (experimental) Allow SSH access from the given peer or peers.
     *
     * Necessary if you want to connect to the instance using ssh. If not
     * called, you should use SSM Session Manager to connect to the instance.
     *
     * @experimental
     */
    allowSshAccessFrom(...peer) {
        peer.forEach(p => {
            this.connections.allowFrom(p, port_1.Port.tcp(22), 'SSH access');
        });
    }
}
exports.BastionHostLinux = BastionHostLinux;
//# sourceMappingURL=data:application/json;base64,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