"""
The main interface to work with YaraRule objects
"""

import logging
import collections

from yarabuilder.yararule import YaraRule


class YaraBuilder:
    """
    Main class to interface with the YaraRule object
    """

    def __init__(self, ws="    "):
        """
        Initialise YaraBuilder
        :param ws: whitespace to use when building the rules (defaults to 4 spaces)
        """
        self.ws = ws
        self.yara_rules = collections.OrderedDict()

    def _no_rule_name_exception_handler(self, rule_name):
        """
        Handler for if a rule_name is not present in the YaraBuilder object
        :param rule_name: the rule_name to check
        """
        if rule_name not in self.yara_rules:
            raise KeyError('Rule  "{0}" doesn\'t exist'.format(rule_name))

    def create_rule(self, rule_name):
        """
        Create a new YaraRule object in the YaraBuilder
        :param rule_name: the name of the rule as a string
        """
        if rule_name in self.yara_rules:
            raise KeyError('Rule with name "{0}" already exists'.format(rule_name))

        self.yara_rules[rule_name] = YaraRule(rule_name, ws=self.ws)

    def add_tag(self, rule_name, tag):
        """
        Add a tag to a specified rule (i.e. appears after the rule_name when built)
        :param rule_name: the rule_name to add the tag to
        :param tag: the tag to be added
        """
        self._no_rule_name_exception_handler(rule_name)
        self.yara_rules[rule_name].tags.add_tag(tag)

    def add_import(self, rule_name, import_str):
        """
        Add an import to a specified rule (i.e. appears before the rule_name when built)
        :param rule_name: the rule_name to add the import to
        :param import_str: the import to be added
        """
        self._no_rule_name_exception_handler(rule_name)
        self.yara_rules[rule_name].imports.add_import(import_str)

    def add_meta(self, rule_name, name, value, meta_type=None):
        """
        Add a meta key/value pair to the specified rule_name
        :param rule_name: the rule_name to add the meta to
        :param name: the name of the meta key
        :param value: the value to go in the metadata (can be a str, int or bool)
        :param meta_type: the type of the meta data,
                          which will be determined by the function if nothing supplied
        """
        self._no_rule_name_exception_handler(rule_name)

        if meta_type:
            self.yara_rules[rule_name].meta.add_meta(name, value, meta_type=meta_type)

        elif value is True or value is False:
            self.yara_rules[rule_name].meta.add_meta(name, value, meta_type="bool")

        elif isinstance(value, int):
            self.yara_rules[rule_name].meta.add_meta(name, value, meta_type="int")

        elif isinstance(value, str):
            self.yara_rules[rule_name].meta.add_meta(name, value, meta_type="text")

    def add_text_string(self, rule_name, value, name=None, modifiers=None):
        """
        Add a text string (e.g. $ = "test") to the specified rule_name
        :param rule_name: the rule_name to add the string to
        :param value: the text string
        :param name: the optional name of the string (if not provided will add as anonymous string)
        :param modifiers: any modifiers to add to the string
        """
        if modifiers is None:
            modifiers = []
        self._no_rule_name_exception_handler(rule_name)

        if name:
            self.yara_rules[rule_name].strings.add_string(name, value, str_type="text")

        else:
            name = self.yara_rules[rule_name].strings.add_anonymous_string(
                value, str_type="text"
            )

        self._modifier_handler(rule_name, name, modifiers)

    def add_hex_string(self, rule_name, value, name=None, modifiers=None):
        """
        Add a hex string (e.g. $ = {DE AD BE EF}) to the specified rule_name
        :param rule_name: the rule_name to add the string to
        :param value: the hex string
        :param name: the optional name of the string (if not provided will add as anonymous string)
        :param modifiers: any modifiers to add to the string
        """
        if modifiers is None:
            modifiers = []
        self._no_rule_name_exception_handler(rule_name)

        if name:
            self.yara_rules[rule_name].strings.add_string(name, value, str_type="hex")

        else:
            name = self.yara_rules[rule_name].strings.add_anonymous_string(
                value, str_type="hex"
            )

        self._modifier_handler(rule_name, name, modifiers)

    def add_regex_string(self, rule_name, value, name=None, modifiers=None):
        """
        Add a regex string (e.g. $ = /test[0-9]{2}/) to the specified rule_name
        :param rule_name: the rule_name to add the string to
        :param value: the regex string
        :param name: the optional name of the string (if not provided will add as anonymous string)
        :param modifiers: any modifiers to add to the string
        """
        if modifiers is None:
            modifiers = []
        self._no_rule_name_exception_handler(rule_name)

        if name:
            self.yara_rules[rule_name].strings.add_string(name, value, str_type="regex")

        else:
            self.yara_rules[rule_name].strings.add_anonymous_string(
                value, str_type="regex"
            )

        self._modifier_handler(rule_name, name, modifiers)

    def _modifier_handler(self, rule_name, str_name, modifiers):
        """
        Handler for to add several modifiers to a string
        :param rule_name: the rule_name to add the modifiers to
        :param str_name: the name of the string to add the modifiers to
        :param modifiers: a list of modifiers
        """
        if modifiers:
            for modifier in modifiers:
                self.yara_rules[rule_name].strings.add_modifier(str_name, modifier)

    def add_condition(self, rule_name, condition):
        """
        Add a raw condition to the specified rule_name
        :param rule_name: the rule_name to add the condition to
        :param condition: the condition as a string
        """
        self._no_rule_name_exception_handler(rule_name)

        self.yara_rules[rule_name].condition.add_raw_condition(condition)

    def build_rule(self, rule_name):
        """
        Build an individual rule in the YaraBuilder object
        :param rule_name: the rule_name to build
        :return: a text string of the built rule
        """
        self._no_rule_name_exception_handler(rule_name)

        return self.yara_rules[rule_name].build_rule()

    def build_rules(self):
        """
        Build all rules in the YaraBuilder object
        :return: a text string of all built rules
        """
        built_rules = []

        for rule in self.yara_rules.values():
            built_rules.append(rule.build_rule())

        return "\n\n".join(built_rules)


def main():  # pragma: no cover
    """
    Method to test if running the module from the command line
    """
    logging.basicConfig(level=logging.DEBUG)

    yara_builder = YaraBuilder()

    yara_builder.create_rule("test_rule1")
    yara_builder.add_meta("test_rule1", "test_name", "test_value")
    yara_builder.add_condition("test_rule1", "filesize > 0")

    yara_builder.create_rule("test_rule2")
    yara_builder.add_text_string("test_rule2", "hello")
    yara_builder.add_text_string("test_rule2", "world")
    yara_builder.add_condition("test_rule2", "any of them")

    print(yara_builder.build_rules())


if __name__ == "__main__":  # pragma: no cover
    main()
