from django.db import models
from django.db.models import ForeignObjectRel


def get_model_relations(model: type[models.Model]) -> list[models.Field]:
    return [
        f
        for f in model._meta.get_fields()  # noqa: SLF001
        if isinstance(f, models.ForeignKey | models.ManyToManyField | models.OneToOneField)
    ]


def get_reverse_model_relations(model: models.Model) -> list[ForeignObjectRel]:
    """
    Return reverse (i.e. autogenerated) foreign and many-to-many relations of a given model.

    Keep note that this list only includes relations which are not present in the code, but generated automatically by
    Django because other models refer to this model in its fields.
    """
    return [f for f in model._meta.get_fields() if isinstance(f, ForeignObjectRel)]  # noqa: SLF001


def get_relations(instance: models.Model) -> list[dict]:
    reverse_fks: list[ForeignObjectRel] = get_reverse_model_relations(instance)

    def get_item(f) -> dict:
        ids: list[int]
        if f.one_to_one is True:
            ids = [getattr(instance, f.name).pk] if hasattr(instance, f.name) else []
        else:
            ids = list(
                getattr(
                    instance,
                    f.related_name or f"{f.related_model._meta.model_name}_set",  # noqa: SLF001
                ).values_list("pk", flat=True),
            )
        try:
            # noinspection PyProtectedMember
            return {
                "model_name": f"{f.related_model._meta.app_label}.{f.related_model._meta.model_name}",  # noqa: SLF001
                "verbose_name": str(f.related_model._meta.verbose_name),  # noqa: SLF001
                "verbose_name_plural": str(f.related_model._meta.verbose_name_plural),  # noqa: SLF001
                "related_field": str(f.remote_field.verbose_name or f.remote_field.name),
                "ids": ids,
            }
        except Exception:
            # noinspection PyProtectedMember
            return {
                "model_name": f"{f.related_model._meta.app_label}.{f.related_model._meta.model_name}",  # noqa: SLF001
                "verbose_name": str(f.related_model._meta.verbose_name),  # noqa: SLF001
                "verbose_name_plural": str(f.related_model._meta.verbose_name_plural),  # noqa: SLF001
                "related_field": "Failed to obtain related field",
                "ids": [],
            }

    return [get_item(f) for f in reverse_fks]
