from typing import Unpack

from moysklad_api.client.default import DefaultParams
from moysklad_api.client.remap import PRODUCTION
from moysklad_api.client.session import BaseSession, Headers, Method
from moysklad_api.loggers import api
from moysklad_api.methods import (
    GetAssortment,
    GetAudit,
    GetCounterparty,
    GetCurrentStock,
    GetEvents,
    GetProduct,
    GetProducts,
    GetPurchaseOrder,
    GetPurchaseOrders,
    GetStock,
    GetToken,
    GetWebhook,
    GetWebhooks,
    MSMethod,
    T,
    UpdateProduct,
    UpdateProducts,
)
from moysklad_api.methods.base import R
from moysklad_api.methods.get_counterparties import GetCounterparties
from moysklad_api.methods.get_demand import GetDemand
from moysklad_api.methods.get_demands import GetDemands
from moysklad_api.types import (
    Assortment,
    Audit,
    Counterparty,
    CurrentStock,
    Demand,
    Event,
    MetaArray,
    Product,
    ProductUpdate,
    PurchaseOrder,
    Stock,
    Token,
    Webhook,
)
from moysklad_api.utils.token import validate_token


class MoyskladAPI:
    def __init__(self, token: str, session: BaseSession | None = None, **kwargs):
        """
        MoyskladAPI class

        :param token: Moy Sklad API token, obtain a new one from <https://api.moysklad.ru/api/remap/1.2/security/token>.
        :raise TokenValidationError: When token has invalid format,
            this exception will be raised
        """
        validate_token(token)
        if session is None:
            session = BaseSession(
                base=PRODUCTION.url,
                timeout=PRODUCTION.timeout,
            )

        self.__token = token
        self._session_headers = self._create_session_headers(self.__token)
        self._session = session
        self._session.headers = self._session_headers

    @property
    def session(self) -> BaseSession:
        return self._session

    @property
    def base(self) -> str:
        return self._session.base

    @property
    def timeout(self) -> int:
        return self._session.timeout

    @property
    def headers(self) -> Headers:
        return self._session.headers

    @property
    def token(self) -> str:
        return self.__token

    @classmethod
    async def from_credentials(
        cls, username: str, password: str, session: BaseSession | None = None
    ) -> "MoyskladAPI":
        if session is None:
            session = BaseSession(
                base=PRODUCTION.url,
                timeout=PRODUCTION.timeout,
            )

        dummy_api = cls(token="dummy", session=session)
        token = await dummy_api.get_token(username, password)
        api.warning(
            "A new token has been obtained. "
            f"Use this token :code:`{token.access_token}` "
            f"for subsequent requests."
        )
        return cls(token=token.access_token, session=session)

    async def close(self):
        await self._session.close()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        import asyncio

        asyncio.run(self.close())

    async def __call__(
        self, method: MSMethod[T], self_method: Method | None = Method.GET
    ) -> T | R:
        return await method.call(self.session, self_method)

    async def get_token(self, username: str, password: str) -> Token:
        call = GetToken(
            username=username,
            password=password,
        )
        return await self(call)

    @staticmethod
    def _create_session_headers(token: str) -> Headers:
        return Headers(token=token)

    async def get_assortment(
        self, **params: Unpack[DefaultParams]
    ) -> MetaArray[Assortment]:
        call = GetAssortment(params=params)
        return await self(call)

    async def get_counterparty(
        self, counterparty_id: str, **params: Unpack[DefaultParams]
    ) -> Counterparty:
        call = GetCounterparty(counterparty_id=counterparty_id, params=params)
        return await self(call)

    async def get_counterparties(
        self, **params: Unpack[DefaultParams]
    ) -> MetaArray[Counterparty]:
        call = GetCounterparties(params=params)
        return await self(call)

    async def get_product(
        self, product_id: str, **params: Unpack[DefaultParams]
    ) -> Product:
        call = GetProduct(product_id=product_id, params=params)
        return await self(call)

    async def get_products(self, **params: Unpack[DefaultParams]) -> MetaArray[Product]:
        call = GetProducts(params=params)
        return await self(call)

    async def update_product(
        self,
        product_id: str,
        name: str | None = None,
        description: str | None = None,
        code: str | None = None,
        external_code: str | None = None,
        archived: bool | None = None,
        article: str | None = None,
        group: str | None = None,
        product_folder: str | None = None,
        sale_prices: list[dict] | None = None,
        attributes: list[dict] | None = None,
        barcodes: list[dict] | None = None,
        min_price: dict | None = None,
        uom: str | None = None,
        tracking_type: str | None = None,
        is_serial_trackable: bool | None = None,
        files: list[dict] | None = None,
        images: list[dict] | None = None,
        packs: list[dict] | None = None,
        owner: str | None = None,
        supplier: str | None = None,
        shared: bool | None = None,
        discount_prohibited: bool | None = None,
        use_parent_vat: bool | None = None,
    ) -> ProductUpdate:
        call = UpdateProduct(
            product_id=product_id,
            name=name,
            description=description,
            code=code,
            external_code=external_code,
            archived=archived,
            article=article,
            group=group,
            product_folder=product_folder,
            sale_prices=sale_prices,
            attributes=attributes,
            barcodes=barcodes,
            min_price=min_price,
            uom=uom,
            tracking_type=tracking_type,
            is_serial_trackable=is_serial_trackable,
            files=files,
            images=images,
            packs=packs,
            owner=owner,
            supplier=supplier,
            shared=shared,
            discount_prohibited=discount_prohibited,
            use_parent_vat=use_parent_vat,
        )
        return await self(call, self_method=Method.PUT)

    async def update_products(
        self, products: list[ProductUpdate]
    ) -> list[ProductUpdate]:
        call = UpdateProducts(
            products=products,
        )
        return await self(call, self_method=Method.POST)

    async def get_stock(
        self,
        current: bool = True,
        **params: Unpack[DefaultParams],
    ) -> CurrentStock | MetaArray[Stock]:
        call = GetCurrentStock(params=params)
        if not current:
            call = GetStock(params=params)
        return await self(call)

    async def get_demand(
        self,
        demand_id: str,
        **params: Unpack[DefaultParams],
    ) -> Demand:
        call = GetDemand(demand_id=demand_id, params=params)
        return await self(call)

    async def get_demands(
        self,
        **params: Unpack[DefaultParams],
    ) -> MetaArray[Demand]:
        call = GetDemands(params=params)
        return await self(call)

    async def get_audit(
        self,
        audit_id: str,
        **params: Unpack[DefaultParams],
    ) -> Audit:
        call = GetAudit(audit_id=audit_id, params=params)
        return await self(call)

    async def get_audit_events(
        self,
        audit_id: str,
        **params: Unpack[DefaultParams],
    ) -> MetaArray[Event]:
        call = GetEvents(audit_id=audit_id, params=params)
        return await self(call)

    async def get_webhook(
        self,
        webhook_id: str,
        **params: Unpack[DefaultParams],
    ) -> Webhook:
        call = GetWebhook(webhook_id=webhook_id, params=params)
        return await self(call)

    async def get_webhooks(
        self,
        **params: Unpack[DefaultParams],
    ) -> Webhook:
        call = GetWebhooks(params=params)
        return await self(call)

    async def get_purchaseorder(
        self,
        purchaseorder_id: str,
        **params: Unpack[DefaultParams],
    ) -> PurchaseOrder:
        call = GetPurchaseOrder(purchaseorder_id=purchaseorder_id, params=params)
        return await self(call)

    async def get_purchaseorders(
        self,
        **params: Unpack[DefaultParams],
    ) -> MetaArray[PurchaseOrder]:
        call = GetPurchaseOrders(params=params)
        return await self(call)
