from .....Internal.Core import Core
from .....Internal.CommandsGroup import CommandsGroup
from .....Internal import Conversions
from .....Internal.StructBase import StructBase
from .....Internal.ArgStruct import ArgStruct


# noinspection PyPep8Naming,PyAttributeOutsideInit,SpellCheckingInspection
class Carrier:
	"""Carrier commands group definition. 8 total commands, 1 Sub-groups, 5 group commands"""

	def __init__(self, core: Core, parent):
		self._core = core
		self._base = CommandsGroup("carrier", core, parent)

	@property
	def listPy(self):
		"""listPy commands group. 3 Sub-classes, 0 commands."""
		if not hasattr(self, '_listPy'):
			from .Carrier_.ListPy import ListPy
			self._listPy = ListPy(self._core, self._base)
		return self._listPy

	def get_count(self) -> int:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:COUNt \n
		Snippet: value: int = driver.source.bb.mccw.carrier.get_count() \n
		No command help available \n
			:return: count: No help available
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:MCCW:CARRier:COUNt?')
		return Conversions.str_to_int(response)

	def set_count(self, count: int) -> None:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:COUNt \n
		Snippet: driver.source.bb.mccw.carrier.set_count(count = 1) \n
		No command help available \n
			:param count: No help available
		"""
		param = Conversions.decimal_value_to_str(count)
		self._core.io.write(f'SOURce<HwInstance>:BB:MCCW:CARRier:COUNt {param}')

	# noinspection PyTypeChecker
	class PhaseStruct(StructBase):
		"""Structure for reading output parameters. Fields: \n
			- Carrier_Index: int: No parameter help available
			- Phase: float: No parameter help available"""
		__meta_args_list = [
			ArgStruct.scalar_int('Carrier_Index'),
			ArgStruct.scalar_float('Phase')]

		def __init__(self):
			StructBase.__init__(self, self)
			self.Carrier_Index: int = None
			self.Phase: float = None

	def get_phase(self) -> PhaseStruct:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:PHASe \n
		Snippet: value: PhaseStruct = driver.source.bb.mccw.carrier.get_phase() \n
		No command help available \n
			:return: structure: for return value, see the help for PhaseStruct structure arguments.
		"""
		return self._core.io.query_struct('SOURce<HwInstance>:BB:MCCW:CARRier:PHASe?', self.__class__.PhaseStruct())

	def set_phase(self, value: PhaseStruct) -> None:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:PHASe \n
		Snippet: driver.source.bb.mccw.carrier.set_phase(value = PhaseStruct()) \n
		No command help available \n
			:param value: see the help for PhaseStruct structure arguments.
		"""
		self._core.io.write_struct('SOURce<HwInstance>:BB:MCCW:CARRier:PHASe', value)

	# noinspection PyTypeChecker
	class PowerStruct(StructBase):
		"""Structure for reading output parameters. Fields: \n
			- Carrier_Index: int: No parameter help available
			- Power: float: No parameter help available"""
		__meta_args_list = [
			ArgStruct.scalar_int('Carrier_Index'),
			ArgStruct.scalar_float('Power')]

		def __init__(self):
			StructBase.__init__(self, self)
			self.Carrier_Index: int = None
			self.Power: float = None

	def get_power(self) -> PowerStruct:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:POWer \n
		Snippet: value: PowerStruct = driver.source.bb.mccw.carrier.get_power() \n
		No command help available \n
			:return: structure: for return value, see the help for PowerStruct structure arguments.
		"""
		return self._core.io.query_struct('SOURce<HwInstance>:BB:MCCW:CARRier:POWer?', self.__class__.PowerStruct())

	def set_power(self, value: PowerStruct) -> None:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:POWer \n
		Snippet: driver.source.bb.mccw.carrier.set_power(value = PowerStruct()) \n
		No command help available \n
			:param value: see the help for PowerStruct structure arguments.
		"""
		self._core.io.write_struct('SOURce<HwInstance>:BB:MCCW:CARRier:POWer', value)

	def get_spacing(self) -> float:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:SPACing \n
		Snippet: value: float = driver.source.bb.mccw.carrier.get_spacing() \n
		No command help available \n
			:return: spacing: No help available
		"""
		response = self._core.io.query_str('SOURce<HwInstance>:BB:MCCW:CARRier:SPACing?')
		return Conversions.str_to_float(response)

	def set_spacing(self, spacing: float) -> None:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:SPACing \n
		Snippet: driver.source.bb.mccw.carrier.set_spacing(spacing = 1.0) \n
		No command help available \n
			:param spacing: No help available
		"""
		param = Conversions.decimal_value_to_str(spacing)
		self._core.io.write(f'SOURce<HwInstance>:BB:MCCW:CARRier:SPACing {param}')

	# noinspection PyTypeChecker
	class StateStruct(StructBase):
		"""Structure for reading output parameters. Fields: \n
			- Carrier_Index: int: No parameter help available
			- State: bool: No parameter help available"""
		__meta_args_list = [
			ArgStruct.scalar_int('Carrier_Index'),
			ArgStruct.scalar_bool('State')]

		def __init__(self):
			StructBase.__init__(self, self)
			self.Carrier_Index: int = None
			self.State: bool = None

	def get_state(self) -> StateStruct:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:STATe \n
		Snippet: value: StateStruct = driver.source.bb.mccw.carrier.get_state() \n
		No command help available \n
			:return: structure: for return value, see the help for StateStruct structure arguments.
		"""
		return self._core.io.query_struct('SOURce<HwInstance>:BB:MCCW:CARRier:STATe?', self.__class__.StateStruct())

	def set_state(self, value: StateStruct) -> None:
		"""SCPI: [SOURce<HW>]:BB:MCCW:CARRier:STATe \n
		Snippet: driver.source.bb.mccw.carrier.set_state(value = StateStruct()) \n
		No command help available \n
			:param value: see the help for StateStruct structure arguments.
		"""
		self._core.io.write_struct('SOURce<HwInstance>:BB:MCCW:CARRier:STATe', value)

	def clone(self) -> 'Carrier':
		"""Clones the group by creating new object from it and its whole existing sub-groups
		Also copies all the existing default Repeated Capabilities setting,
		which you can change independently without affecting the original group"""
		new_group = Carrier(self._core, self._base.parent)
		self._base.synchronize_repcaps(new_group)
		return new_group
