"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackAppManifest = exports.SlackAppManifestShortcutType = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const nodeUrl = require("url");
const aws_cdk_lib_1 = require("aws-cdk-lib");
/**
 * Type of shortcuts
 *
 * @see https://api.slack.com/interactivity/shortcuts
 */
var SlackAppManifestShortcutType;
(function (SlackAppManifestShortcutType) {
    /**
     * Message shortcuts are shown to users in the context menus of messages within Slack
     *
     * @see https://api.slack.com/interactivity/shortcuts/using#message_shortcuts
     */
    SlackAppManifestShortcutType["MESSAGE"] = "message";
    /**
     * Global shortcuts are available to users via the shortcuts button in the composer,
     * and when using search in Slack
     *
     * @see https://api.slack.com/interactivity/shortcuts/using#global_shortcuts
     */
    SlackAppManifestShortcutType["GLOBAL"] = "global";
})(SlackAppManifestShortcutType = exports.SlackAppManifestShortcutType || (exports.SlackAppManifestShortcutType = {}));
;
/**
 * A Slack app manifest
 *
 * @see https://api.slack.com/reference/manifests
 */
class SlackAppManifest {
    constructor(props) {
        this.props = props;
        validateLength('app name', 35, props.name);
        validateLength('app description', 140, props.description);
        validateLength('app long description', 4000, props.longDescription);
        validateColor(props.backgroundColor);
        validateUrl(props.eventSubscriptions?.requestUrl);
        validateItems('bot events', 100, props.eventSubscriptions?.botEvents);
        validateItems('users events', 100, props.eventSubscriptions?.userEvents);
        validateUrl(props.interactivity?.requestUrl);
        validateUrl(props.interactivity?.messageMenuOptionsUrl);
        validateLength('bot display name', 80, props.botUser?.displayName);
        validateItems('shortcuts', 5, props.shortcuts);
        props.shortcuts?.forEach((shortcut) => {
            validateLength('shortcut callback ID', 255, shortcut.callbackId);
            validateLength('shortcut description', 150, shortcut.description);
        });
        validateItems('slash commands', 5, props.slashCommands);
        props.slashCommands?.forEach((command) => {
            validateLength('slash command', 32, command.command);
            validateLength('slash command description', 2000, command.description),
                validateUrl(command.url);
            validateLength('slash command use hint', 1000, command.usageHint);
        });
        validateItems('workflow steps', 10, props.workflowSteps);
        props.workflowSteps?.forEach((step) => {
            validateLength('workflow step name', 50, step.name);
            validateLength('workflow step callback ID', 50, step.callbackId);
        });
        validateItems('unfurls domains', 5, props.unfurlDomains);
        validateItems('OAuth redirect URLs', 1000, props.oauthConfig?.redirectUrls);
        props.oauthConfig?.redirectUrls?.forEach((url) => {
            validateUrl(url, false);
        });
        validateItems('bot scopes', 255, props.oauthConfig?.botScopes);
        validateItems('user scopes', 255, props.oauthConfig?.userScopes);
    }
    render(construct) {
        const schema = {
            _metadata: {
                major_version: this.props.majorVersion,
                minor_version: this.props.minorVersion,
            },
            display_information: {
                name: this.props.name,
                description: this.props.description,
                long_description: this.props.longDescription,
                background_color: prefixWith('#', this.props.backgroundColor)?.toLowerCase(),
            },
            settings: {
                allowed_ip_address_ranges: this.props.allowedIpAddressRanges,
                event_subscriptions: {
                    request_url: this.props.eventSubscriptions?.requestUrl,
                    bot_events: this.props.eventSubscriptions?.botEvents,
                    user_events: this.props.eventSubscriptions?.userEvents,
                },
                interactivity: this.props.interactivity
                    ? {
                        is_enabled: this.props.interactivity?.enabled ?? true,
                        request_url: this.props.interactivity?.requestUrl,
                        message_menu_options_url: this.props.interactivity?.messageMenuOptionsUrl,
                    }
                    : undefined,
                org_deploy_enabled: this.props.orgDeploy,
                socket_mode_enabled: this.props.socketMode,
            },
            features: {
                app_home: {
                    home_tab_enabled: this.props.appHome?.homeTab,
                    messages_tab_enabled: this.props.appHome?.messagesTab,
                    messages_tab_read_only_enabled: this.props.appHome?.messagesTabReadOnly,
                },
                bot_user: this.props.botUser
                    ? {
                        display_name: this.props.botUser?.displayName,
                        always_online: this.props.botUser?.alwaysOnline,
                    }
                    : undefined,
                shortcuts: this.props.shortcuts?.map((shortcut) => ({
                    name: shortcut.name,
                    type: shortcut.type,
                    callback_id: shortcut.callbackId,
                    description: shortcut.description,
                })),
                slash_commands: this.props.slashCommands?.map((command) => ({
                    command: prefixWith('/', command.command),
                    description: command.description,
                    url: command.url,
                    usage_hint: command.usageHint,
                    should_escape: command.shouldEscape,
                })),
                workflow_steps: this.props.workflowSteps?.map((step) => ({
                    name: step.name,
                    callback_id: step.callbackId,
                })),
                unfurl_domains: this.props.unfurlDomains,
            },
            oauth_config: {
                redirect_urls: this.props.oauthConfig?.redirectUrls,
                scopes: {
                    bot: this.props.oauthConfig?.botScopes,
                    users: this.props.oauthConfig?.userScopes,
                },
            },
        };
        return aws_cdk_lib_1.Stack.of(construct).toJsonString(removeUndefined(schema));
    }
}
exports.SlackAppManifest = SlackAppManifest;
_a = JSII_RTTI_SYMBOL_1;
SlackAppManifest[_a] = { fqn: "cloudstructs.SlackAppManifest", version: "0.6.21" };
function prefixWith(prefix, string) {
    if (!string) {
        return undefined;
    }
    if (string.startsWith(prefix)) {
        return string;
    }
    return `${prefix}${string}`;
}
function validateLength(description, max, xs) {
    if (xs && !aws_cdk_lib_1.Token.isUnresolved(xs) && xs.length > max) {
        throw new Error(`Maximum length for ${description} is ${max}, got ${xs.length}: ${xs}`);
    }
}
function validateItems(description, max, xs) {
    if (xs && !aws_cdk_lib_1.Token.isUnresolved(xs) && xs.length > max) {
        throw new Error(`Maximum number of items for ${description} is ${max}, got ${xs.length}`);
    }
}
function validateColor(color) {
    if (color && !aws_cdk_lib_1.Token.isUnresolved(color) && !/^#?([0-9a-fA-F]{3}|[0-9a-fA-F]{6})$/.test(color)) {
        throw new Error(`Invalid hex color: ${color}`);
    }
}
function validateUrl(url, https = true) {
    if (url && !aws_cdk_lib_1.Token.isUnresolved(url)) {
        try {
            const parsed = new nodeUrl.URL(url);
            if (https && parsed.protocol !== 'https:') {
                throw new Error('Invalid protocol');
            }
        }
        catch (err) {
            throw new Error(`${url} is not a valid${https ? ' HTTPS' : ''} url`);
        }
    }
}
function removeUndefined(obj) {
    if (typeof obj === 'string') {
        return obj;
    }
    if (Array.isArray(obj)) {
        const ret = obj
            .map(v => (v && typeof v === 'object') ? removeUndefined(v) : v)
            .filter(v => !(v == null));
        return ret.length !== 0 ? ret : undefined;
    }
    const ret = Object.entries(obj)
        .map(([k, v]) => [k, v && typeof v === 'object' ? removeUndefined(v) : v])
        .reduce((a, [k, v]) => (v == null ? a : (a[k] = v, a)), {});
    return Object.keys(ret).length !== 0 ? ret : undefined;
}
//# sourceMappingURL=data:application/json;base64,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