"""
# Hdl21 Primitive Modules

Primitives are leaf-level Modules typically defined not by users, 
but by simulation tools or device fabricators. 
Prominent examples include MOS transistors, diodes, resistors, and capacitors. 

Primitives divide in two classes, `physical` and `ideal`, indicated by their `primtype` attribute. 
`PrimitiveType.IDEAL` primitives specify circuit-theoretic ideal elements 
e.g. resistors, capacitors, inductors, and notably aphysical elements 
such as ideal voltage and current sources. 

`PrimitiveType.PHYSICAL` primitives in contrast specify abstract versions 
of ultimately physically-realizable elements such as transistors and diodes. 
These elements typically require some external translation, e.g. by a process-technology 
library, to execute in simulations or to be realized in hardware. 

Many element-types (particularly passives) come in both `ideal` and `physical` flavors, 
as typical process-technologies include physical passives, but with far different 
parameterization than ideal passives. For example resistors are commonly specified 
in physical length and width. Capacitors are similarly specified in physical terms, 
often adding metal layers or other physical features. The component-value (R,C,L, etc.) 
for these physically-specified cells is commonly suggestive or optional. 

Both the `Primitive` type and all of its instances are defined in this module. 
The collection of `Primitive` instances is often referred to as Hdl21's "primitive library". 

Summary of the content of the primitive library: 

| Name                           | Description                       | Type     | Aliases                               | Ports        |
| ------------------------------ | --------------------------------- | -------- | ------------------------------------- | ------------ |
| Mos                            | Mos Transistor                    | PHYSICAL | MOS                                   | d, g, s, b   |
| IdealResistor                  | Ideal Resistor                    | IDEAL    | R, Res, Resistor, IdealR, IdealRes    | p, n         |
| PhysicalResistor               | Physical Resistor                 | PHYSICAL | PhyR, PhyRes, ResPhy, PhyResistor     | p, n         |
| ThreeTerminalResistor          | Three Terminal Resistor           | PHYSICAL | Res3, PhyRes3, ResPhy3, PhyResistor3  | p, n, b      |
| IdealCapacitor                 | Ideal Capacitor                   | IDEAL    | C, Cap, Capacitor, IdealC, IdealCap   | p, n         |
| PhysicalCapacitor              | Physical Capacitor                | PHYSICAL | PhyC, PhyCap, CapPhy, PhyCapacitor    | p, n         |
| ThreeTerminalCapacitor         | Three Terminal Capacitor          | PHYSICAL | Cap3, PhyCap3, CapPhy3, PhyCapacitor3 | p, n, b      |
| IdealInductor                  | Ideal Inductor                    | IDEAL    | L, Ind, Inductor, IdealL, IdealInd    | p, n         |
| PhysicalInductor               | Physical Inductor                 | PHYSICAL | PhyL, PhyInd, IndPhy, PhyInductor     | p, n         |
| ThreeTerminalInductor          | Three Terminal Inductor           | PHYSICAL | Ind3, PhyInd3, IndPhy3, PhyInductor3  | p, n, b      |
| PhysicalShort                  | Short-Circuit/ Net-Tie            | PHYSICAL | Short                                 | p, n         |
| DcVoltageSource                | DC Voltage Source                 | IDEAL    | V, Vdc, Vsrc                          | p, n         |
| PulseVoltageSource             | Pulse Voltage Source              | IDEAL    | Vpu, Vpulse                           | p, n         |
| CurrentSource                  | Ideal DC Current Source           | IDEAL    | I, Idc, Isrc                          | p, n         |
| VoltageControlledVoltageSource | Voltage Controlled Voltage Source | IDEAL    | Vcvs, VCVS                            | p, n, cp, cn |
| CurrentControlledVoltageSource | Current Controlled Voltage Source | IDEAL    | Ccvs, CCVS                            | p, n, cp, cn |
| VoltageControlledCurrentSource | Voltage Controlled Current Source | IDEAL    | Vccs, VCCS                            | p, n, cp, cn |
| CurrentControlledCurrentSource | Current Controlled Current Source | IDEAL    | Cccs, CCCS                            | p, n, cp, cn |
| Bipolar                        | Bipolar Transistor                | PHYSICAL | Bjt, BJT                              | c, b, e      |
| Diode                          | Diode                             | PHYSICAL | D                                     | p, n         |

"""

# Non-docstring comment:
#
# That big table above is also included in the Hdl21 package documentation.
# It is generated by the script `Hdl21/scripts/primtable.py`,
# which imports the contents of this module and prints a line per primitive.
# On changes to this module, re-run the script, paste the table here and anywhere else it is used.

# Std-Lib Imports
import copy
from enum import Enum
from dataclasses import replace
from typing import Optional, Any, List, Type, Dict, Union

# PyPi Imports
from pydantic.dataclasses import dataclass

# Local imports
from .default import Default
from .call import param_call
from .params import paramclass, Param, isparamclass, NoParams
from .signal import Port, Signal, Visibility
from .instance import calls_instantiate
from .prefix import Prefix, Prefixed

# # The `Scalar` parameter type
#
# Most primitive parameters "prefer" to be the `Prefixed` type, for reasons outlined in
# https://github.com/dan-fritchman/Hdl21#prefixed-numeric-parameters.
# They often also need a string-valued escape hatch, e.g. when referring to out-of-Hdl21 quantities
# such as parameters in external netlists, or simulation decks.
#
# Note: conversion into `Scalar` from Python's built-in numeric types `int` and `float`
# may be slightly counter-intuitive: both create the *string* variant, not `Prefixed`.
# These would probably preferably convert to `Prefixed` instead, some day.
#
Scalar = Union[Prefixed, str]


class PrimitiveType(Enum):
    """Enumerated Primitive-Types"""

    IDEAL = "IDEAL"
    PHYSICAL = "PHYSICAL"


@dataclass
class Primitive:
    """# Hdl21 Primitive Component

    Primitives are leaf-level Modules typically defined not by users,
    but by simulation tools or device fabricators.
    Prominent examples include MOS transistors, diodes, resistors, and capacitors.
    """

    name: str  # Primitive Name
    desc: str  # String Description
    port_list: List[Signal]  # Ordered Port List
    paramtype: Type  # Class/ Type of valid Parameters
    primtype: PrimitiveType  # Ideal vs Physical Primitive-Type

    def __post_init_post_parse__(self):
        """After type-checking, do plenty more checks on values"""
        if not isparamclass(self.paramtype):
            msg = f"Invalid Primitive param-type {self.paramtype} for {self.name}, must be an `hdl21.paramclass`"
            raise TypeError(msg)
        for p in self.port_list:
            if not p.name:
                raise ValueError(f"Unnamed Primitive Port {p} for {self.name}")
            if p.vis != Visibility.PORT:
                msg = f"Invalid Primitive Port {p.name} on {self.name}; must have PORT visibility"
                raise ValueError(msg)

    def __call__(self, arg: Any = Default, **kwargs) -> "PrimitiveCall":
        params = param_call(callee=self, arg=arg, **kwargs)
        return PrimitiveCall(prim=self, params=params)

    @property
    def Params(self) -> Type:
        return self.paramtype

    @property
    def ports(self) -> Dict[str, Signal]:
        return {p.name: p for p in self.port_list}


@calls_instantiate
@dataclass
class PrimitiveCall:
    """Primitive Call
    A combination of a Primitive and its Parameter-values,
    typically generated by calling the Primitive."""

    prim: Primitive
    params: Any = NoParams

    def __post_init_post_parse__(self):
        # Type-validate our parameters
        if not isinstance(self.params, self.prim.paramtype):
            msg = f"Invalid parameters {self.params} for Primitive {self.prim}. Must be {self.prim.paramtype}"
            raise TypeError(msg)

    @property
    def ports(self) -> dict:
        return self.prim.ports


@dataclass
class PrimLibEntry:
    prim: Primitive
    aliases: List[str]


# Dictionary storing all primitives, keyed by their primary name.
# Stores aliases on the side.
_primitives: Dict[str, PrimLibEntry] = dict()


def _add(prim: Primitive, aliases: List[str]) -> Primitive:
    """Add a primitive to this library.
    Ensures its identifier matches its `name` field, and adds any aliases to the global namespace.
    This is a private function and should be used solely during `hdl21.primitives` import-time."""
    global _primitives

    if prim.name in _primitives or prim.name in globals():
        raise ValueError(f"Duplicate primitive name {prim.name}")

    # Add the combination as a new entry
    entry = PrimLibEntry(prim, aliases)
    _primitives[prim.name] = entry
    globals()[prim.name] = prim

    for alias in aliases:
        if alias in _primitives or alias in globals():
            raise ValueError(f"Duplicate primitive alias {alias}")
        globals()[alias] = prim
    return prim


""" 
Mos Transistor Section 
"""


class MosType(Enum):
    """NMOS/PMOS Type Enumeration"""

    NMOS = "NMOS"
    PMOS = "PMOS"


class MosVth(Enum):
    """MOS Threshold Enumeration"""

    STD = "STD"
    LOW = "LOW"
    HIGH = "HIGH"


@paramclass
class MosParams:
    """MOS Transistor Parameters"""

    w = Param(dtype=Optional[Scalar], desc="Width in resolution units", default=None)
    l = Param(dtype=Optional[Scalar], desc="Length in resolution units", default=None)
    npar = Param(dtype=int, desc="Number of parallel fingers", default=1)
    tp = Param(dtype=MosType, desc="MosType (Nmos/ Pmos)", default=MosType.NMOS)
    vth = Param(dtype=MosVth, desc="Threshold voltage specifier", default=MosVth.STD)
    model = Param(dtype=Optional[str], desc="Model (Name)", default=None)
    # FIXME: whether to include `model`

    def __post_init_post_parse__(self):
        """Value Checks"""
        if self.w <= 0:
            raise ValueError(f"MosParams with invalid width {self.w}")
        if self.l <= 0:
            raise ValueError(f"MosParams with invalid length {self.l}")
        if self.npar <= 0:
            msg = f"MosParams with invalid number parallel fingers {self.npar}"
            raise ValueError(msg)


# Mos Transistor Ports, in SPICE Conventional Order
MosPorts = [
    Port(name="d", desc="Drain"),
    Port(name="g", desc="Gate"),
    Port(name="s", desc="Source"),
    Port(name="b", desc="Bulk"),
]

Mos = _add(
    prim=Primitive(
        name="Mos",
        desc="Mos Transistor",
        port_list=copy.deepcopy(MosPorts),
        paramtype=MosParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["MOS"],
)


def Nmos(arg: Any = Default, **kwargs) -> Primitive:
    """Nmos Constructor. A thin wrapper around `hdl21.primitives.Mos`"""
    mos = Mos(arg, **kwargs)
    mos.params = replace(mos.params, tp=MosType.NMOS)
    return mos


def Pmos(arg: Any = Default, **kwargs) -> Primitive:
    """Pmos Constructor. A thin wrapper around `hdl21.primitives.Mos`"""
    mos = Mos(arg, **kwargs)
    mos.params = replace(mos.params, tp=MosType.PMOS)
    return mos


""" 
Passives
"""

# Oft-reused port list for the passive elements
PassivePorts = [Port(name="p"), Port(name="n")]
# And the three-terminal version
ThreeTerminalPorts = [Port(name="p"), Port(name="n"), Port(name="b")]


@paramclass
class ResistorParams:
    r = Param(dtype=Scalar, desc="Resistance (ohms)")


_add(
    prim=Primitive(
        name="IdealResistor",
        desc="Ideal Resistor",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=ResistorParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["R", "Res", "Resistor", "IdealR", "IdealRes"],
)


@paramclass
class PhysicalResistorParams:
    w = Param(dtype=Optional[Scalar], desc="Width in resolution units", default=None)
    l = Param(dtype=Optional[Scalar], desc="Length in resolution units", default=None)
    model = Param(dtype=Optional[str], desc="Model (Name)", default=None)


_add(
    prim=Primitive(
        name="PhysicalResistor",
        desc="Physical Resistor",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=PhysicalResistorParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["PhyR", "PhyRes", "ResPhy", "PhyResistor"],
)


_add(
    prim=Primitive(
        name="ThreeTerminalResistor",
        desc="Three Terminal Resistor",
        port_list=copy.deepcopy(ThreeTerminalPorts),
        paramtype=PhysicalResistorParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["Res3", "PhyRes3", "ResPhy3", "PhyResistor3"],
)


@paramclass
class IdealCapacitorParams:
    c = Param(dtype=Scalar, desc="Capacitance (F)")


_add(
    prim=Primitive(
        name="IdealCapacitor",
        desc="Ideal Capacitor",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=IdealCapacitorParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["C", "Cap", "Capacitor", "IdealC", "IdealCap"],
)


@paramclass
class PhysicalCapacitorParams:
    c = Param(dtype=Scalar, desc="Capacitance (F)")


_add(
    prim=Primitive(
        name="PhysicalCapacitor",
        desc="Physical Capacitor",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=PhysicalCapacitorParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["PhyC", "PhyCap", "CapPhy", "PhyCapacitor"],
)


_add(
    prim=Primitive(
        name="ThreeTerminalCapacitor",
        desc="Three Terminal Capacitor",
        port_list=copy.deepcopy(ThreeTerminalPorts),
        paramtype=PhysicalCapacitorParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["Cap3", "PhyCap3", "CapPhy3", "PhyCapacitor3"],
)


@paramclass
class IdealInductorParams:
    l = Param(dtype=Scalar, desc="Inductance (H)")


_add(
    prim=Primitive(
        name="IdealInductor",
        desc="Ideal Inductor",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=IdealInductorParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["L", "Ind", "Inductor", "IdealL", "IdealInd"],
)


@paramclass
class PhysicalInductorParams:
    l = Param(dtype=Scalar, desc="Inductance (H)")


_add(
    Primitive(
        name="PhysicalInductor",
        desc="Physical Inductor",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=PhysicalInductorParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["PhyL", "PhyInd", "IndPhy", "PhyInductor"],
)


_add(
    prim=Primitive(
        name="ThreeTerminalInductor",
        desc="Three Terminal Inductor",
        port_list=copy.deepcopy(ThreeTerminalPorts),
        paramtype=PhysicalInductorParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["Ind3", "PhyInd3", "IndPhy3", "PhyInductor3"],
)


@paramclass
class PhysicalShortParams:
    layer = Param(dtype=Optional[Union[int, str]], desc="Metal layer", default=None)
    w = Param(dtype=Optional[Scalar], desc="Width in resolution units", default=None)
    l = Param(dtype=Optional[Scalar], desc="Length in resolution units", default=None)


_add(
    prim=Primitive(
        name="PhysicalShort",
        desc="Short-Circuit/ Net-Tie",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=PhysicalShortParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["Short"],
)


""" 
Sources
"""


@paramclass
class DcVoltageSourceParams:
    """`DcVoltageSource` Parameters"""

    dc = Param(dtype=Scalar, default=0 * Prefix.UNIT, desc="DC Value (V)")
    ac = Param(dtype=Optional[Scalar], default=None, desc="AC Amplitude (V)")


_add(
    prim=Primitive(
        name="DcVoltageSource",
        desc="DC Voltage Source",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=DcVoltageSourceParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=[
        "V",
        "Vdc",
        "Vsrc",
    ],
)


@paramclass
class PulseVoltageSourceParams:
    """`PulseVoltageSource` Parameters"""

    delay = Param(dtype=Optional[Scalar], default=None, desc="Time Delay (s)")
    v1 = Param(dtype=Optional[Scalar], default=None, desc="One Value (V)")
    v2 = Param(dtype=Optional[Scalar], default=None, desc="Zero Value (V)")
    period = Param(dtype=Optional[Scalar], default=None, desc="Period (s)")
    rise = Param(dtype=Optional[Scalar], default=None, desc="Rise time (s)")
    fall = Param(dtype=Optional[Scalar], default=None, desc="Fall time (s)")
    width = Param(dtype=Optional[Scalar], default=None, desc="Pulse width (s)")


_add(
    prim=Primitive(
        name="PulseVoltageSource",
        desc="Pulse Voltage Source",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=PulseVoltageSourceParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["Vpu", "Vpulse"],
)


@paramclass
class SineVoltageSourceParams:
    """`SineVoltageSource` Parameters"""

    voff = Param(dtype=Optional[Scalar], default=None, desc="Offset (V)")
    vamp = Param(dtype=Optional[Scalar], default=None, desc="Amplitude (V)")
    freq = Param(dtype=Optional[Scalar], default=None, desc="Frequency (Hz)")
    td = Param(dtype=Optional[Scalar], default=None, desc="Delay (s)")
    phase = Param(dtype=Optional[Scalar], default=None, desc="Phase at td (degrees)")


_add(
    prim=Primitive(
        name="SineVoltageSource",
        desc="Sine Voltage Source",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=SineVoltageSourceParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["Vsin"],
)


@paramclass
class CurrentSourceParams:
    dc = Param(dtype=Optional[Scalar], default=0, desc="DC Value (A)")


_add(
    Primitive(
        name="CurrentSource",
        desc="Ideal DC Current Source",
        port_list=copy.deepcopy(PassivePorts),
        paramtype=CurrentSourceParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["I", "Idc", "Isrc"],
)


"""
Controlled Sources 
"""


@paramclass
class ControlledSourceParams:
    gain = Param(dtype=Scalar, default=1 * Prefix.UNIT, desc="Gain in SI Units")


# Controlled Sources Port List
ControlledSourcePorts = [
    Port(name="p", desc="Output, Positive"),
    Port(name="n", desc="Output, Negative"),
    Port(name="cp", desc="Control, Positive"),
    Port(name="cn", desc="Control, Negative"),
]

_add(
    prim=Primitive(
        name="VoltageControlledVoltageSource",
        desc="Voltage Controlled Voltage Source",
        port_list=copy.deepcopy(ControlledSourcePorts),
        paramtype=ControlledSourceParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["Vcvs", "VCVS"],
)
_add(
    prim=Primitive(
        name="CurrentControlledVoltageSource",
        desc="Current Controlled Voltage Source",
        port_list=copy.deepcopy(ControlledSourcePorts),
        paramtype=ControlledSourceParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["Ccvs", "CCVS"],
)
_add(
    prim=Primitive(
        name="VoltageControlledCurrentSource",
        desc="Voltage Controlled Current Source",
        port_list=copy.deepcopy(ControlledSourcePorts),
        paramtype=ControlledSourceParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["Vccs", "VCCS"],
)
_add(
    prim=Primitive(
        name="CurrentControlledCurrentSource",
        desc="Current Controlled Current Source",
        port_list=copy.deepcopy(ControlledSourcePorts),
        paramtype=ControlledSourceParams,
        primtype=PrimitiveType.IDEAL,
    ),
    aliases=["Cccs", "CCCS"],
)

""" 
Bipolar Section 
"""


class BipolarType(Enum):
    """Bipolar Junction Transistor NPN/PNP Type Enumeration"""

    NPN = "NPN"
    PNP = "PNP"


@paramclass
class BipolarParams:
    """Bipolar Transistor Parameters"""

    w = Param(dtype=Optional[Scalar], desc="Width in resolution units", default=None)
    l = Param(dtype=Optional[Scalar], desc="Length in resolution units", default=None)
    tp = Param(
        dtype=BipolarType, desc="Bipolar Type (NPN/ PNP)", default=BipolarType.NPN
    )

    def __post_init_post_parse__(self):
        """Value Checks"""
        if self.w <= 0:
            raise ValueError(f"BipolarParams with invalid width {self.w}")
        if self.l <= 0:
            raise ValueError(f"BipolarParams with invalid length {self.l}")


BipolarPorts = [Port(name="c"), Port(name="b"), Port(name="e")]

Bipolar = _add(
    prim=Primitive(
        name="Bipolar",
        desc="Bipolar Transistor",
        port_list=copy.deepcopy(BipolarPorts),
        paramtype=BipolarParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["Bjt", "BJT"],
)


def Npn(arg: Any = Default, **kwargs) -> Primitive:
    """Npn Constructor. A thin wrapper around `hdl21.primitives.Bipolar`"""
    bip = Bipolar(arg, **kwargs)
    bip.params = replace(bip.params, tp=BipolarType.NPN)
    return bip


def Pnp(arg: Any = Default, **kwargs) -> Primitive:
    """Pnp Constructor. A thin wrapper around `hdl21.primitives.Bipolar`"""
    bip = Bipolar(arg, **kwargs)
    bip.params = replace(bip.params, tp=BipolarType.PNP)
    return bip


""" 
Diodes
"""


@paramclass
class DiodeParams:
    w = Param(dtype=Optional[Scalar], desc="Width in resolution units", default=None)
    l = Param(dtype=Optional[Scalar], desc="Length in resolution units", default=None)
    model = Param(dtype=Optional[str], desc="Model (Name)", default=None)
    # FIXME: whether to include `model`


_add(
    prim=Primitive(
        name="Diode",
        desc="Diode",
        # Despite not really being "passive", Diode does use the same `PassivePorts` list.
        port_list=copy.deepcopy(PassivePorts),
        paramtype=DiodeParams,
        primtype=PrimitiveType.PHYSICAL,
    ),
    aliases=["D"],
)
