"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ClusterResourceProvider = void 0;
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const cr = require("@aws-cdk/custom-resources");
const lambda_layer_node_proxy_agent_1 = require("@aws-cdk/lambda-layer-node-proxy-agent");
const HANDLER_DIR = path.join(__dirname, 'cluster-resource-handler');
const HANDLER_RUNTIME = lambda.Runtime.NODEJS_12_X;
/**
 * A custom resource provider that handles cluster operations. It serves
 * multiple custom resources such as the cluster resource and the fargate
 * resource.
 *
 * @internal
 */
class ClusterResourceProvider extends core_1.NestedStack {
    constructor(scope, id, props) {
        super(scope, id);
        // The NPM dependency proxy-agent is required in order to support proxy routing with the AWS JS SDK.
        const nodeProxyAgentLayer = new lambda_layer_node_proxy_agent_1.NodeProxyAgentLayer(this, 'NodeProxyAgentLayer');
        const onEvent = new lambda.Function(this, 'OnEventHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'onEvent handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: props.environment,
            handler: 'index.onEvent',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
            // Allow user to override the layer.
            layers: props.onEventLayer ? [props.onEventLayer] : [nodeProxyAgentLayer],
        });
        const isComplete = new lambda.Function(this, 'IsCompleteHandler', {
            code: lambda.Code.fromAsset(HANDLER_DIR),
            description: 'isComplete handler for EKS cluster resource provider',
            runtime: HANDLER_RUNTIME,
            environment: props.environment,
            handler: 'index.isComplete',
            timeout: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
            layers: [nodeProxyAgentLayer],
        });
        this.provider = new cr.Provider(this, 'Provider', {
            onEventHandler: onEvent,
            isCompleteHandler: isComplete,
            totalTimeout: core_1.Duration.hours(1),
            queryInterval: core_1.Duration.minutes(1),
            vpc: props.subnets ? props.vpc : undefined,
            vpcSubnets: props.subnets ? { subnets: props.subnets } : undefined,
            securityGroups: props.securityGroup ? [props.securityGroup] : undefined,
        });
        props.adminRole.grant(onEvent.role, 'sts:AssumeRole');
        props.adminRole.grant(isComplete.role, 'sts:AssumeRole');
    }
    static getOrCreate(scope, props) {
        var _a;
        const stack = core_1.Stack.of(scope);
        const uid = '@aws-cdk/aws-eks.ClusterResourceProvider';
        return (_a = stack.node.tryFindChild(uid)) !== null && _a !== void 0 ? _a : new ClusterResourceProvider(stack, uid, props);
    }
    /**
     * The custom resource service token for this provider.
     */
    get serviceToken() { return this.provider.serviceToken; }
}
exports.ClusterResourceProvider = ClusterResourceProvider;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2x1c3Rlci1yZXNvdXJjZS1wcm92aWRlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImNsdXN0ZXItcmVzb3VyY2UtcHJvdmlkZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNkJBQTZCO0FBRzdCLDhDQUE4QztBQUM5Qyx3Q0FBNkQ7QUFDN0QsZ0RBQWdEO0FBQ2hELDBGQUE2RTtBQU83RSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO0FBQ3JFLE1BQU0sZUFBZSxHQUFHLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDO0FBc0NuRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLHVCQUF3QixTQUFRLGtCQUFXO0lBYXRELFlBQW9CLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQW1DO1FBQ25GLEtBQUssQ0FBQyxLQUFzQixFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWxDLG9HQUFvRztRQUNwRyxNQUFNLG1CQUFtQixHQUFHLElBQUksbURBQW1CLENBQUMsSUFBSSxFQUFFLHFCQUFxQixDQUFDLENBQUM7UUFFakYsTUFBTSxPQUFPLEdBQUcsSUFBSSxNQUFNLENBQUMsUUFBUSxDQUFDLElBQUksRUFBRSxnQkFBZ0IsRUFBRTtZQUMxRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDO1lBQ3hDLFdBQVcsRUFBRSxtREFBbUQ7WUFDaEUsT0FBTyxFQUFFLGVBQWU7WUFDeEIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO1lBQzlCLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLE9BQU8sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUM1QixHQUFHLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMxQyxVQUFVLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2xFLGNBQWMsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN2RSxvQ0FBb0M7WUFDcEMsTUFBTSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLG1CQUFtQixDQUFDO1NBQzFFLENBQUMsQ0FBQztRQUVILE1BQU0sVUFBVSxHQUFHLElBQUksTUFBTSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUU7WUFDaEUsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQztZQUN4QyxXQUFXLEVBQUUsc0RBQXNEO1lBQ25FLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLFdBQVcsRUFBRSxLQUFLLENBQUMsV0FBVztZQUM5QixPQUFPLEVBQUUsa0JBQWtCO1lBQzNCLE9BQU8sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUM1QixHQUFHLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMxQyxVQUFVLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2xFLGNBQWMsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUN2RSxNQUFNLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQztTQUM5QixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsUUFBUSxHQUFHLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ2hELGNBQWMsRUFBRSxPQUFPO1lBQ3ZCLGlCQUFpQixFQUFFLFVBQVU7WUFDN0IsWUFBWSxFQUFFLGVBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQy9CLGFBQWEsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUNsQyxHQUFHLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUMxQyxVQUFVLEVBQUUsS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2xFLGNBQWMsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUN4RSxDQUFDLENBQUM7UUFFSCxLQUFLLENBQUMsU0FBUyxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSyxFQUFFLGdCQUFnQixDQUFDLENBQUM7UUFDdkQsS0FBSyxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLElBQUssRUFBRSxnQkFBZ0IsQ0FBQyxDQUFDO0lBQzVELENBQUM7SUF4RE0sTUFBTSxDQUFDLFdBQVcsQ0FBQyxLQUFnQixFQUFFLEtBQW1DOztRQUM3RSxNQUFNLEtBQUssR0FBRyxZQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sR0FBRyxHQUFHLDBDQUEwQyxDQUFDO1FBQ3ZELGFBQU8sS0FBSyxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUE0QixtQ0FBSSxJQUFJLHVCQUF1QixDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDbkgsQ0FBQztJQXNERDs7T0FFRztJQUNILElBQVcsWUFBWSxLQUFLLE9BQU8sSUFBSSxDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDO0NBQ2pFO0FBaEVELDBEQWdFQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG5pbXBvcnQgKiBhcyBlYzIgZnJvbSAnQGF3cy1jZGsvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBEdXJhdGlvbiwgTmVzdGVkU3RhY2ssIFN0YWNrIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgKiBhcyBjciBmcm9tICdAYXdzLWNkay9jdXN0b20tcmVzb3VyY2VzJztcbmltcG9ydCB7IE5vZGVQcm94eUFnZW50TGF5ZXIgfSBmcm9tICdAYXdzLWNkay9sYW1iZGEtbGF5ZXItbm9kZS1wcm94eS1hZ2VudCc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QgfSBmcm9tICdjb25zdHJ1Y3RzJztcblxuLy8gdjIgLSBrZWVwIHRoaXMgaW1wb3J0IGFzIGEgc2VwYXJhdGUgc2VjdGlvbiB0byByZWR1Y2UgbWVyZ2UgY29uZmxpY3Qgd2hlbiBmb3J3YXJkIG1lcmdpbmcgd2l0aCB0aGUgdjIgYnJhbmNoLlxuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lXG5pbXBvcnQgeyBDb25zdHJ1Y3QgYXMgQ29yZUNvbnN0cnVjdCB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuXG5jb25zdCBIQU5ETEVSX0RJUiA9IHBhdGguam9pbihfX2Rpcm5hbWUsICdjbHVzdGVyLXJlc291cmNlLWhhbmRsZXInKTtcbmNvbnN0IEhBTkRMRVJfUlVOVElNRSA9IGxhbWJkYS5SdW50aW1lLk5PREVKU18xMl9YO1xuXG5leHBvcnQgaW50ZXJmYWNlIENsdXN0ZXJSZXNvdXJjZVByb3ZpZGVyUHJvcHMge1xuICAvKipcbiAgICogVGhlIElBTSByb2xlIHRvIGFzc3VtZSBpbiBvcmRlciB0byBpbnRlcmFjdCB3aXRoIHRoZSBjbHVzdGVyLlxuICAgKi9cbiAgcmVhZG9ubHkgYWRtaW5Sb2xlOiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIFRoZSBWUEMgdG8gcHJvdmlzaW9uIHRoZSBmdW5jdGlvbnMgaW4uXG4gICAqL1xuICByZWFkb25seSB2cGM/OiBlYzIuSVZwYztcblxuICAvKipcbiAgICogVGhlIHN1Ym5ldHMgdG8gcGxhY2UgdGhlIGZ1bmN0aW9ucyBpbi5cbiAgICovXG4gIHJlYWRvbmx5IHN1Ym5ldHM/OiBlYzIuSVN1Ym5ldFtdO1xuXG4gIC8qKlxuICAgKiBFbnZpcm9ubWVudCB0byBhZGQgdG8gdGhlIGhhbmRsZXIuXG4gICAqL1xuICByZWFkb25seSBlbnZpcm9ubWVudD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgLyoqXG4gICAqIEFuIEFXUyBMYW1iZGEgbGF5ZXIgdGhhdCBpbmNsdWRlcyB0aGUgTlBNIGRlcGVuZGVuY3kgYHByb3h5LWFnZW50YC5cbiAgICpcbiAgICogSWYgbm90IGRlZmluZWQsIGEgZGVmYXVsdCBsYXllciB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBvbkV2ZW50TGF5ZXI/OiBsYW1iZGEuSUxheWVyVmVyc2lvbjtcblxuICAvKipcbiAgICogVGhlIHNlY3VyaXR5IGdyb3VwIHRvIGFzc29jaWF0ZSB3aXRoIHRoZSBmdW5jdGlvbnMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gc2VjdXJpdHkgZ3JvdXAuXG4gICAqL1xuICByZWFkb25seSBzZWN1cml0eUdyb3VwPzogZWMyLklTZWN1cml0eUdyb3VwO1xufVxuXG4vKipcbiAqIEEgY3VzdG9tIHJlc291cmNlIHByb3ZpZGVyIHRoYXQgaGFuZGxlcyBjbHVzdGVyIG9wZXJhdGlvbnMuIEl0IHNlcnZlc1xuICogbXVsdGlwbGUgY3VzdG9tIHJlc291cmNlcyBzdWNoIGFzIHRoZSBjbHVzdGVyIHJlc291cmNlIGFuZCB0aGUgZmFyZ2F0ZVxuICogcmVzb3VyY2UuXG4gKlxuICogQGludGVybmFsXG4gKi9cbmV4cG9ydCBjbGFzcyBDbHVzdGVyUmVzb3VyY2VQcm92aWRlciBleHRlbmRzIE5lc3RlZFN0YWNrIHtcblxuICBwdWJsaWMgc3RhdGljIGdldE9yQ3JlYXRlKHNjb3BlOiBDb25zdHJ1Y3QsIHByb3BzOiBDbHVzdGVyUmVzb3VyY2VQcm92aWRlclByb3BzKSB7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZihzY29wZSk7XG4gICAgY29uc3QgdWlkID0gJ0Bhd3MtY2RrL2F3cy1la3MuQ2x1c3RlclJlc291cmNlUHJvdmlkZXInO1xuICAgIHJldHVybiBzdGFjay5ub2RlLnRyeUZpbmRDaGlsZCh1aWQpIGFzIENsdXN0ZXJSZXNvdXJjZVByb3ZpZGVyID8/IG5ldyBDbHVzdGVyUmVzb3VyY2VQcm92aWRlcihzdGFjaywgdWlkLCBwcm9wcyk7XG4gIH1cblxuICAvKipcbiAgICogVGhlIGN1c3RvbSByZXNvdXJjZSBwcm92aWRlciB0byB1c2UgZm9yIGN1c3RvbSByZXNvdXJjZXMuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcHJvdmlkZXI6IGNyLlByb3ZpZGVyO1xuXG4gIHByaXZhdGUgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IENsdXN0ZXJSZXNvdXJjZVByb3ZpZGVyUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSBhcyBDb3JlQ29uc3RydWN0LCBpZCk7XG5cbiAgICAvLyBUaGUgTlBNIGRlcGVuZGVuY3kgcHJveHktYWdlbnQgaXMgcmVxdWlyZWQgaW4gb3JkZXIgdG8gc3VwcG9ydCBwcm94eSByb3V0aW5nIHdpdGggdGhlIEFXUyBKUyBTREsuXG4gICAgY29uc3Qgbm9kZVByb3h5QWdlbnRMYXllciA9IG5ldyBOb2RlUHJveHlBZ2VudExheWVyKHRoaXMsICdOb2RlUHJveHlBZ2VudExheWVyJyk7XG5cbiAgICBjb25zdCBvbkV2ZW50ID0gbmV3IGxhbWJkYS5GdW5jdGlvbih0aGlzLCAnT25FdmVudEhhbmRsZXInLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5mcm9tQXNzZXQoSEFORExFUl9ESVIpLFxuICAgICAgZGVzY3JpcHRpb246ICdvbkV2ZW50IGhhbmRsZXIgZm9yIEVLUyBjbHVzdGVyIHJlc291cmNlIHByb3ZpZGVyJyxcbiAgICAgIHJ1bnRpbWU6IEhBTkRMRVJfUlVOVElNRSxcbiAgICAgIGVudmlyb25tZW50OiBwcm9wcy5lbnZpcm9ubWVudCxcbiAgICAgIGhhbmRsZXI6ICdpbmRleC5vbkV2ZW50JyxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMSksXG4gICAgICB2cGM6IHByb3BzLnN1Ym5ldHMgPyBwcm9wcy52cGMgOiB1bmRlZmluZWQsXG4gICAgICB2cGNTdWJuZXRzOiBwcm9wcy5zdWJuZXRzID8geyBzdWJuZXRzOiBwcm9wcy5zdWJuZXRzIH0gOiB1bmRlZmluZWQsXG4gICAgICBzZWN1cml0eUdyb3VwczogcHJvcHMuc2VjdXJpdHlHcm91cCA/IFtwcm9wcy5zZWN1cml0eUdyb3VwXSA6IHVuZGVmaW5lZCxcbiAgICAgIC8vIEFsbG93IHVzZXIgdG8gb3ZlcnJpZGUgdGhlIGxheWVyLlxuICAgICAgbGF5ZXJzOiBwcm9wcy5vbkV2ZW50TGF5ZXIgPyBbcHJvcHMub25FdmVudExheWVyXSA6IFtub2RlUHJveHlBZ2VudExheWVyXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGlzQ29tcGxldGUgPSBuZXcgbGFtYmRhLkZ1bmN0aW9uKHRoaXMsICdJc0NvbXBsZXRlSGFuZGxlcicsIHtcbiAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmZyb21Bc3NldChIQU5ETEVSX0RJUiksXG4gICAgICBkZXNjcmlwdGlvbjogJ2lzQ29tcGxldGUgaGFuZGxlciBmb3IgRUtTIGNsdXN0ZXIgcmVzb3VyY2UgcHJvdmlkZXInLFxuICAgICAgcnVudGltZTogSEFORExFUl9SVU5USU1FLFxuICAgICAgZW52aXJvbm1lbnQ6IHByb3BzLmVudmlyb25tZW50LFxuICAgICAgaGFuZGxlcjogJ2luZGV4LmlzQ29tcGxldGUnLFxuICAgICAgdGltZW91dDogRHVyYXRpb24ubWludXRlcygxKSxcbiAgICAgIHZwYzogcHJvcHMuc3VibmV0cyA/IHByb3BzLnZwYyA6IHVuZGVmaW5lZCxcbiAgICAgIHZwY1N1Ym5ldHM6IHByb3BzLnN1Ym5ldHMgPyB7IHN1Ym5ldHM6IHByb3BzLnN1Ym5ldHMgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHNlY3VyaXR5R3JvdXBzOiBwcm9wcy5zZWN1cml0eUdyb3VwID8gW3Byb3BzLnNlY3VyaXR5R3JvdXBdIDogdW5kZWZpbmVkLFxuICAgICAgbGF5ZXJzOiBbbm9kZVByb3h5QWdlbnRMYXllcl0sXG4gICAgfSk7XG5cbiAgICB0aGlzLnByb3ZpZGVyID0gbmV3IGNyLlByb3ZpZGVyKHRoaXMsICdQcm92aWRlcicsIHtcbiAgICAgIG9uRXZlbnRIYW5kbGVyOiBvbkV2ZW50LFxuICAgICAgaXNDb21wbGV0ZUhhbmRsZXI6IGlzQ29tcGxldGUsXG4gICAgICB0b3RhbFRpbWVvdXQ6IER1cmF0aW9uLmhvdXJzKDEpLFxuICAgICAgcXVlcnlJbnRlcnZhbDogRHVyYXRpb24ubWludXRlcygxKSxcbiAgICAgIHZwYzogcHJvcHMuc3VibmV0cyA/IHByb3BzLnZwYyA6IHVuZGVmaW5lZCxcbiAgICAgIHZwY1N1Ym5ldHM6IHByb3BzLnN1Ym5ldHMgPyB7IHN1Ym5ldHM6IHByb3BzLnN1Ym5ldHMgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHNlY3VyaXR5R3JvdXBzOiBwcm9wcy5zZWN1cml0eUdyb3VwID8gW3Byb3BzLnNlY3VyaXR5R3JvdXBdIDogdW5kZWZpbmVkLFxuICAgIH0pO1xuXG4gICAgcHJvcHMuYWRtaW5Sb2xlLmdyYW50KG9uRXZlbnQucm9sZSEsICdzdHM6QXNzdW1lUm9sZScpO1xuICAgIHByb3BzLmFkbWluUm9sZS5ncmFudChpc0NvbXBsZXRlLnJvbGUhLCAnc3RzOkFzc3VtZVJvbGUnKTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgY3VzdG9tIHJlc291cmNlIHNlcnZpY2UgdG9rZW4gZm9yIHRoaXMgcHJvdmlkZXIuXG4gICAqL1xuICBwdWJsaWMgZ2V0IHNlcnZpY2VUb2tlbigpIHsgcmV0dXJuIHRoaXMucHJvdmlkZXIuc2VydmljZVRva2VuOyB9XG59XG4iXX0=