from moonshot.src.bookmark.bookmark import Bookmark
from moonshot.src.bookmark.bookmark_arguments import BookmarkArguments


def api_insert_bookmark(
    name: str,
    prompt: str,
    prepared_prompt: str,
    response: str,
    context_strategy: str = "",
    prompt_template: str = "",
    attack_module: str = "",
    metric: str = "",
) -> dict:
    """
    Inserts a new bookmark into the database.

    This function constructs a BookmarkArguments object with the provided details and
    invokes the add_bookmark method of a Bookmark instance to persist the new bookmark.

    Args:
        name (str): The unique name of the bookmark.
        prompt (str): The associated prompt text for the bookmark.
        prepared_prompt (str): The prepared prompt text for the bookmark.
        response (str): The corresponding response text for the bookmark.
        context_strategy (str, optional): The strategy used for context management in the bookmark. Defaults to "".
        prompt_template (str, optional): The template used for generating the prompt. Defaults to "".
        attack_module (str, optional): The attack module linked with the bookmark. Defaults to "".
        metric (str, optional): The metric associated with the bookmark. Defaults to "".

    Returns:
        dict: A dictionary containing the details of the newly inserted bookmark.
    """
    # Create a new BookmarkArguments object
    bookmark_args = BookmarkArguments(
        id=0,  # id will be auto-generated by the database
        name=name,
        prompt=prompt,
        prepared_prompt=prepared_prompt,
        response=response,
        context_strategy=context_strategy,
        prompt_template=prompt_template,
        attack_module=attack_module,
        metric=metric,
        bookmark_time="",  # bookmark_time will be set to current time in add_bookmark method
    )
    return Bookmark().add_bookmark(bookmark_args)


def api_get_all_bookmarks() -> list[dict]:
    """
    Retrieves a list of all bookmarks from the database.

    Returns:
        list[dict]: A list of bookmarks, each represented as a dictionary.
    """
    return Bookmark().get_all_bookmarks()


def api_get_bookmark(bookmark_name: str) -> dict:
    """
    Retrieves the details of a specific bookmark by its name.

    Args:
        bookmark_name (str): The name of the bookmark to retrieve.

    Returns:
        dict: The bookmark details corresponding to the provided name.
    """
    return Bookmark().get_bookmark(bookmark_name)


def api_delete_bookmark(bookmark_name: str) -> dict:
    """
    Removes a bookmark from the database using its name.

    Args:
        bookmark_name (str): The name of the bookmark to be removed.

    Returns:
        dict: A dictionary containing the details of the deleted bookmark.
    """
    return Bookmark().delete_bookmark(bookmark_name)


def api_delete_all_bookmark() -> dict:
    """
    Removes all bookmarks from the database.

    Returns:
        dict: A dictionary indicating the result of the delete operation.
    """
    return Bookmark().delete_all_bookmark()


def api_export_bookmarks(export_file_name: str = "bookmarks") -> str:
    """
    Exports bookmarks to a specified file.

    Args:
        export_file_name (str): The name of the file to export the bookmarks to.

    Returns:
        str: The filepath of where the file is written.
    """
    return Bookmark().export_bookmarks(export_file_name)
