import time
import json
import requests
from urllib.parse import urljoin
from uuid import uuid4
from datetime import datetime, timedelta

from azure.storage.blob import generate_blob_sas, BlobSasPermissions, BlobServiceClient
from django.utils import timezone


class BlobService:
    def __init__(self, account_name: str, account_key: str, container: str, conn_str: str):
        self.account_name = account_name
        self.account_key = account_key
        self.container_name = container
        self.connect_str = conn_str
        self.blob_service_client = BlobServiceClient.from_connection_string(self.connect_str)

    def post_blob(self, file_path: str) -> str:
        blob_client = self.blob_service_client.get_blob_client(
            container=self.container_name,
            blob=file_path
        )
        with open(file_path, "rb") as data:
            blob_client.upload_blob(
                data,
                overwrite=True
            )
        return file_path

    def get_blob_sas(self, name: str) -> str:
        sas_blob = generate_blob_sas(
            account_name=self.account_name,
            container_name=self.container_name,
            blob_name=name,
            account_key=self.account_key,
            permission=BlobSasPermissions(read=True),
            expiry=datetime.utcnow() + timedelta(hours=3)
        )
        url = 'https://{}.blob.core.windows.net/{}/{}?{}'.format(
            self.account_name,
            self.container_name,
            name,
            sas_blob
        )
        return url


class AzureTranscribe:
    def __init__(self, base_url: str, token: str):
        self.base_url = base_url
        self.token = token
        self.headers = {
            "Content-Type": "application/json",
            "Ocp-Apim-Subscription-Key": self.token
        }

    def create_transcription(self, sas_url: str, language: str = 'en-US') -> dict:
        url = urljoin(self.base_url, 'transcriptions')
        payload = {
            "contentUrls": [
                sas_url
            ],
            "locale": language,
            "displayName": str(uuid4())
        }
        response = requests.post(url, headers=self.headers, data=json.dumps(payload))
        response.raise_for_status()
        return response.json()['self']

    def check_status(self, transcription_url: str, time_sleep: float = 15, time_out: float = 600) -> str:
        start = timezone.now()
        while True:
            response = requests.get(transcription_url, headers=self.headers)
            if response.json()['status'] == 'Succeeded':
                return response.json()['links']['files']
            time.sleep(time_sleep)
            if timezone.now() > start + timedelta(seconds=time_out):
                raise TimeoutError

    def get_result(self, files_url: str) -> str:
        response = requests.get(files_url, headers=self.headers)
        file_url = response.json()['values'][1]['links']['contentUrl']
        response = requests.get(file_url)
        return response.json()['combinedRecognizedPhrases'][0]['display']
