/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 */

#if !defined(SIMDE_X86_AVX512_SQRT_H)
#define SIMDE_X86_AVX512_SQRT_H

#include "mov.h"
#include "types.h"

HEDLEY_DIAGNOSTIC_PUSH
SIMDE_DISABLE_UNWANTED_DIAGNOSTICS
SIMDE_BEGIN_DECLS_

SIMDE_FUNCTION_ATTRIBUTES
simde__m512 simde_mm512_sqrt_ps(simde__m512 a) {
#if defined(SIMDE_X86_AVX512F_NATIVE)
  return _mm512_sqrt_ps(a);
#else
  simde__m512_private r_, a_ = simde__m512_to_private(a);

#if defined(SIMDE_X86_AVX_NATIVE)
  r_.m256[0] = simde_mm256_sqrt_ps(a_.m256[0]);
  r_.m256[1] = simde_mm256_sqrt_ps(a_.m256[1]);
#elif defined(simde_math_sqrtf)
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.f32) / sizeof(r_.f32[0])); i++) {
    r_.f32[i] = simde_math_sqrtf(a_.f32[i]);
  }
#else
  HEDLEY_UNREACHABLE();
#endif

  return simde__m512_from_private(r_);
#endif
}
#if defined(SIMDE_X86_AVX512F_ENABLE_NATIVE_ALIASES)
#define _mm512_sqrt_ps(a) simde_mm512_sqrt_ps(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde__m512 simde_mm512_mask_sqrt_ps(simde__m512 src, simde__mmask16 k,
                                     simde__m512 a) {
#if defined(SIMDE_X86_AVX512F_NATIVE)
  return _mm512_mask_sqrt_ps(src, k, a);
#else
  return simde_mm512_mask_mov_ps(src, k, simde_mm512_sqrt_ps(a));
#endif
}
#if defined(SIMDE_X86_AVX512F_ENABLE_NATIVE_ALIASES)
#undef _mm512_mask_sqrt_ps
#define _mm512_mask_sqrt_ps(src, k, a) simde_mm512_mask_sqrt_ps(src, k, a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde__m512d simde_mm512_sqrt_pd(simde__m512d a) {
#if defined(SIMDE_X86_AVX512F_NATIVE)
  return _mm512_sqrt_pd(a);
#else
  simde__m512d_private r_, a_ = simde__m512d_to_private(a);

#if defined(SIMDE_X86_AVX_NATIVE)
  r_.m256d[0] = simde_mm256_sqrt_pd(a_.m256d[0]);
  r_.m256d[1] = simde_mm256_sqrt_pd(a_.m256d[1]);
#elif defined(simde_math_sqrt)
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.f64) / sizeof(r_.f64[0])); i++) {
    r_.f64[i] = simde_math_sqrt(a_.f64[i]);
  }
#else
  HEDLEY_UNREACHABLE();
#endif

  return simde__m512d_from_private(r_);
#endif
}
#if defined(SIMDE_X86_AVX512F_ENABLE_NATIVE_ALIASES)
#define _mm512_sqrt_pd(a) simde_mm512_sqrt_pd(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde__m512d simde_mm512_mask_sqrt_pd(simde__m512d src, simde__mmask8 k,
                                      simde__m512d a) {
#if defined(SIMDE_X86_AVX512F_NATIVE)
  return _mm512_mask_sqrt_pd(src, k, a);
#else
  return simde_mm512_mask_mov_pd(src, k, simde_mm512_sqrt_pd(a));
#endif
}
#if defined(SIMDE_X86_AVX512F_ENABLE_NATIVE_ALIASES)
#undef _mm512_mask_sqrt_pd
#define _mm512_mask_sqrt_pd(src, k, a) simde_mm512_mask_sqrt_pd(src, k, a)
#endif

SIMDE_END_DECLS_
HEDLEY_DIAGNOSTIC_POP

#endif /* !defined(SIMDE_X86_AVX512_SQRT_H) */
