/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Sean Maher <seanptmaher@gmail.com>
 */

/*
 * Implementor's notes: seanptmaher
 *
 * this intrinsic has many (7 pages worth) of variations on it, so
 * this file might become a dumping ground for various
 * implementations. I'm only going to write the `cvt` intrinsics for
 * now, but off the top of my head, we've got:
 *
 * - vcvt* <- round towards 0
 * - vcvtn* <- round to nearest with ties to even
 * - vcvtm* <- round towards minus infinity
 * - vcvtp* <- round towards plus infinity
 * - vcvta* <- round to nearest with ties to away
 *
 * and there are a bunch of variations on these. More information can
 * be found here:
 * https://developer.arm.com/architectures/instruction-sets/simd-isas/neon/intrinsics?page=1&search=vcvt
 */

#if !defined(SIMDE_ARM_NEON_CVT_H)
#define SIMDE_ARM_NEON_CVT_H

#include "types.h"

HEDLEY_DIAGNOSTIC_PUSH
SIMDE_DISABLE_UNWANTED_DIAGNOSTICS
SIMDE_BEGIN_DECLS_

SIMDE_FUNCTION_ATTRIBUTES
int32_t simde_vcvts_s32_f32(simde_float32 a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
  return vcvts_s32_f32(a);
#else
  return HEDLEY_STATIC_CAST(int32_t, a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvts_s32_f32
#define vcvts_s32_f32(a) simde_vcvts_s32_f32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
uint32_t simde_vcvts_u32_f32(simde_float32 a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvts_u32_f32(a);
#else
  return HEDLEY_STATIC_CAST(uint32_t, (a < 0) ? 0 : a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvts_u32_f32
#define vcvts_u32_f32(a) simde_vcvts_u32_f32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float32 simde_vcvts_f32_s32(int32_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
  return vcvts_f32_s32(a);
#else
  return HEDLEY_STATIC_CAST(simde_float32, a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvts_f32_s32
#define vcvts_f32_s32(a) simde_vcvts_f32_s32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float32 simde_vcvts_f32_u32(uint32_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvts_f32_u32(a);
#else
  return HEDLEY_STATIC_CAST(simde_float32, a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvts_f32_u32
#define vcvts_f32_u32(a) simde_vcvts_f32_u32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
int64_t simde_vcvtd_s64_f64(simde_float64 a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
  return vcvtd_s64_f64(a);
#else
  return HEDLEY_STATIC_CAST(int64_t, a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvtd_s64_f64
#define vcvtd_s64_f64(a) simde_vcvtd_s64_f64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
uint64_t simde_vcvtd_u64_f64(simde_float64 a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvtd_u64_f64(a);
#else
  return HEDLEY_STATIC_CAST(uint64_t, (a < 0) ? 0 : a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvtd_u64_f64
#define vcvtd_u64_f64(a) simde_vcvtd_u64_f64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float64 simde_vcvtd_f64_s64(int64_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
  return vcvtd_f64_s64(a);
#else
  return HEDLEY_STATIC_CAST(simde_float64, a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvtd_f64_s64
#define vcvtd_f64_s64(a) simde_vcvtd_f64_s64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float64 simde_vcvtd_f64_u64(uint64_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvtd_f64_u64(a);
#else
  return HEDLEY_STATIC_CAST(simde_float64, a);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvtd_f64_u64
#define vcvtd_f64_u64(a) simde_vcvtd_f64_u64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int32x2_t simde_vcvt_s32_f32(simde_float32x2_t a) {
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
  return vcvt_s32_f32(a);
#else
  simde_float32x2_private a_ = simde_float32x2_to_private(a);
  simde_int32x2_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvts_s32_f32(a_.values[i]);
  }
  return simde_int32x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vcvt_s32_f32
#define vcvt_s32_f32(a) simde_vcvt_s32_f32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint32x2_t simde_vcvt_u32_f32(simde_float32x2_t a) {
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvt_u32_f32(a);
#else
  simde_float32x2_private a_ = simde_float32x2_to_private(a);
  simde_uint32x2_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvts_u32_f32(a_.values[i]);
  }
  return simde_uint32x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vcvt_u32_f32
#define vcvt_u32_f32(a) simde_vcvt_u32_f32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int64x1_t simde_vcvt_s64_f64(simde_float64x1_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
  return vcvt_s64_f64(a);
#else
  simde_float64x1_private a_ = simde_float64x1_to_private(a);
  simde_int64x1_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvtd_s64_f64(a_.values[i]);
  }
  return simde_int64x1_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvt_s64_f64
#define vcvt_s64_f64(a) simde_vcvt_s64_f64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint64x1_t simde_vcvt_u64_f64(simde_float64x1_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvt_u64_f64(a);
#else
  simde_float64x1_private a_ = simde_float64x1_to_private(a);
  simde_uint64x1_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvtd_u64_f64(a_.values[i]);
  }
  return simde_uint64x1_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvt_u64_f64
#define vcvt_u64_f64(a) simde_vcvt_u64_f64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int32x4_t simde_vcvtq_s32_f32(simde_float32x4_t a) {
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
  return vcvtq_s32_f32(a);
#else
  simde_float32x4_private a_ = simde_float32x4_to_private(a);
  simde_int32x4_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvts_s32_f32(a_.values[i]);
  }
  return simde_int32x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vcvtq_s32_f32
#define vcvtq_s32_f32(a) simde_vcvtq_s32_f32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint32x4_t simde_vcvtq_u32_f32(simde_float32x4_t a) {
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvtq_u32_f32(a);
#else
  simde_float32x4_private a_ = simde_float32x4_to_private(a);
  simde_uint32x4_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvts_u32_f32(a_.values[i]);
  }
  return simde_uint32x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vcvtq_u32_f32
#define vcvtq_u32_f32(a) simde_vcvtq_u32_f32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_int64x2_t simde_vcvtq_s64_f64(simde_float64x2_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
  return vcvtq_s64_f64(a);
#else
  simde_float64x2_private a_ = simde_float64x2_to_private(a);
  simde_int64x2_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvtd_s64_f64(a_.values[i]);
  }
  return simde_int64x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvtq_s64_f64
#define vcvtq_s64_f64(a) simde_vcvtq_s64_f64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_uint64x2_t simde_vcvtq_u64_f64(simde_float64x2_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvtq_u64_f64(a);
#else
  simde_float64x2_private a_ = simde_float64x2_to_private(a);
  simde_uint64x2_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvtd_u64_f64(a_.values[i]);
  }
  return simde_uint64x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvtq_u64_f64
#define vcvtq_u64_f64(a) simde_vcvtq_u64_f64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float32x2_t simde_vcvt_f32_s32(simde_int32x2_t a) {
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
  return vcvt_f32_s32(a);
#else
  simde_int32x2_private a_ = simde_int32x2_to_private(a);
  simde_float32x2_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvts_f32_s32(a_.values[i]);
  }
  return simde_float32x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vcvt_f32_s32
#define vcvt_f32_s32(a) simde_vcvt_f32_s32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float32x2_t simde_vcvt_f32_u32(simde_uint32x2_t a) {
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvt_f32_u32(a);
#else
  simde_uint32x2_private a_ = simde_uint32x2_to_private(a);
  simde_float32x2_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvts_f32_u32(a_.values[i]);
  }
  return simde_float32x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vcvt_f32_u32
#define vcvt_f32_u32(a) simde_vcvt_f32_u32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float64x1_t simde_vcvt_f64_s64(simde_int64x1_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
  return vcvt_f64_s64(a);
#else
  simde_int64x1_private a_ = simde_int64x1_to_private(a);
  simde_float64x1_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvtd_f64_s64(a_.values[i]);
  }
  return simde_float64x1_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvt_f64_s64
#define vcvt_f64_s64(a) simde_vcvt_f64_s64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float64x1_t simde_vcvt_f64_u64(simde_uint64x1_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvt_f64_u64(a);
#else
  simde_uint64x1_private a_ = simde_uint64x1_to_private(a);
  simde_float64x1_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvtd_f64_u64(a_.values[i]);
  }
  return simde_float64x1_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvt_f64_u64
#define vcvt_f64_u64(a) simde_vcvt_f64_u64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float32x4_t simde_vcvtq_f32_s32(simde_int32x4_t a) {
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE)
  return vcvtq_f32_s32(a);
#else
  simde_int32x4_private a_ = simde_int32x4_to_private(a);
  simde_float32x4_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvts_f32_s32(a_.values[i]);
  }
  return simde_float32x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vcvtq_f32_s32
#define vcvtq_f32_s32(a) simde_vcvtq_f32_s32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float32x4_t simde_vcvtq_f32_u32(simde_uint32x4_t a) {
#if defined(SIMDE_ARM_NEON_A32V7_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvtq_f32_u32(a);
#else
  simde_uint32x4_private a_ = simde_uint32x4_to_private(a);
  simde_float32x4_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvts_f32_u32(a_.values[i]);
  }
  return simde_float32x4_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A32V7_ENABLE_NATIVE_ALIASES)
#undef vcvtq_f32_u32
#define vcvtq_f32_u32(a) simde_vcvtq_f32_u32(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float64x2_t simde_vcvtq_f64_s64(simde_int64x2_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE)
  return vcvtq_f64_s64(a);
#else
  simde_int64x2_private a_ = simde_int64x2_to_private(a);
  simde_float64x2_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvtd_f64_s64(a_.values[i]);
  }
  return simde_float64x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvtq_f64_s64
#define vcvtq_f64_s64(a) simde_vcvtq_f64_s64(a)
#endif

SIMDE_FUNCTION_ATTRIBUTES
simde_float64x2_t simde_vcvtq_f64_u64(simde_uint64x2_t a) {
#if defined(SIMDE_ARM_NEON_A64V8_NATIVE) && !defined(SIMDE_BUG_CLANG_46844)
  return vcvtq_f64_u64(a);
#else
  simde_uint64x2_private a_ = simde_uint64x2_to_private(a);
  simde_float64x2_private r_;
  SIMDE_VECTORIZE
  for (size_t i = 0; i < (sizeof(r_.values) / sizeof(r_.values[0])); i++) {
    r_.values[i] = simde_vcvtd_f64_u64(a_.values[i]);
  }
  return simde_float64x2_from_private(r_);
#endif
}
#if defined(SIMDE_ARM_NEON_A64V8_ENABLE_NATIVE_ALIASES)
#undef vcvtq_f64_u64
#define vcvtq_f64_u64(a) simde_vcvtq_f64_u64(a)
#endif

SIMDE_END_DECLS_
HEDLEY_DIAGNOSTIC_POP

#endif /* SIMDE_ARM_NEON_CVT_H */
