"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let stack;
describe('VersionQuery', () => {
    beforeEach(() => {
        stack = new core_1.Stack(undefined, undefined);
    });
    describe('constructor', () => {
        test('throws not implemented error for empty version string', () => {
            // WHEN
            expect(() => {
                new lib_1.VersionQuery(stack, 'version', {
                    version: '',
                });
            }).toThrowError(/MethodNotSupportedException: This method is currently not implemented./);
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function'));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource', {
                DeadlineVersion: '',
            }));
        });
        test('throws not implemented error for valid version string', () => {
            // WHEN
            expect(() => {
                new lib_1.VersionQuery(stack, 'version', {
                    version: '1.2',
                });
            }).toThrowError(/MethodNotSupportedException: This method is currently not implemented./);
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function'));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource', {
                DeadlineVersion: '1.2',
            }));
        });
        test('throws not implemented error without props', () => {
            // WHEN
            expect(() => {
                new lib_1.VersionQuery(stack, 'version');
            }).toThrowError(/MethodNotSupportedException: This method is currently not implemented./);
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function'));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource'));
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource', {
                DeadlineVersion: '',
            }));
        });
    });
    // GIVEN
    const majorVersion = 1;
    const minorVersion = 2;
    const releaseVersion = 3;
    const patchVersion = 4;
    const expectedVersionString = `${majorVersion}.${minorVersion}.${releaseVersion}.${patchVersion}`;
    let version;
    function exactVersionTests() {
        test('does not create a custom resource', () => {
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResourceLike('AWS::CloudFormation::CustomResource'));
        });
        test('does not create a lambda', () => {
            // THEN
            assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Function'));
        });
        test('populates version components', () => {
            // THEN
            expect(version.majorVersion).toEqual(1);
            expect(version.minorVersion).toEqual(2);
            expect(version.releaseVersion).toEqual(3);
        });
        test('provides linux installers', () => {
            // GIVEN
            const linuxFullVersionString = version.linuxFullVersionString();
            // THEN
            expect(version.linuxInstallers).toBeDefined();
            expect(linuxFullVersionString).toBeDefined();
            expect(linuxFullVersionString).toMatch(expectedVersionString);
            expect(version.linuxInstallers.repository).toBeDefined();
            expect(version.linuxInstallers.repository.s3Bucket.bucketName).toMatch('thinkbox-installers');
            expect(version.linuxInstallers.repository.objectKey).toMatch(`DeadlineRepository-${expectedVersionString}-linux-x64-installer.run`);
        });
    }
    describe('.exact()', () => {
        beforeEach(() => {
            version = lib_1.VersionQuery.exact(stack, 'version', {
                majorVersion,
                minorVersion,
                releaseVersion,
                patchVersion,
            });
        });
        exactVersionTests();
    });
    describe('.exactString()', () => {
        beforeEach(() => {
            version = lib_1.VersionQuery.exactString(stack, 'version', expectedVersionString);
        });
        exactVersionTests();
        test.each([
            [''],
            ['abc'],
            ['1'],
            ['1.2'],
            ['1.2.3'],
            ['1.2.3.4a'],
            ['a1.2.3.4a'],
            ['a1.2.3.4'],
            [' 1.2.3.4 '],
            ['a.b.c.d'],
            ['-1.0.2.3'],
            ['.1.0.2.3'],
        ])('throws an error on invalid versions %p', (versionStr) => {
            // WHEN
            function when() {
                lib_1.VersionQuery.exactString(stack, 'version', versionStr);
            }
            // THEN
            expect(when).toThrowError(new RegExp(`^"${versionStr}" is not a valid version$`));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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