"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const env = {
    region: 'us-east-1',
};
let app;
let certificateSecret;
let deadlineVersion;
let dependencyStack;
let dockerContainer;
let images;
let lfCluster;
let licenses;
let rcsImage;
let renderQueue;
let stack;
let ubl;
let vpc;
let workerFleet;
describe('UsageBasedLicensing', () => {
    beforeEach(() => {
        // GIVEN
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DependencyStack', { env });
        deadlineVersion = lib_1.VersionQuery.exact(dependencyStack, 'Version', {
            majorVersion: 10,
            minorVersion: 1,
            releaseVersion: 9,
            patchVersion: 1,
        });
        expect(deadlineVersion.linuxFullVersionString).toBeDefined();
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'VPC');
        rcsImage = aws_ecs_1.ContainerImage.fromDockerImageAsset(new aws_ecr_assets_1.DockerImageAsset(dependencyStack, 'Image', {
            directory: __dirname,
        }));
        renderQueue = new lib_1.RenderQueue(dependencyStack, 'RQ-NonDefaultPort', {
            version: deadlineVersion,
            vpc,
            images: { remoteConnectionServer: rcsImage },
            repository: new lib_1.Repository(dependencyStack, 'RepositoryNonDefault', {
                vpc,
                version: deadlineVersion,
            }),
        });
        lfCluster = new aws_ecs_1.Cluster(dependencyStack, 'licenseForwarderCluster', {
            vpc,
        });
        certificateSecret = aws_secretsmanager_1.Secret.fromSecretArn(lfCluster, 'CertSecret', 'arn:aws:secretsmanager:us-west-2:675872700355:secret:CertSecret-j1kiFz');
        dockerContainer = new aws_ecr_assets_1.DockerImageAsset(lfCluster, 'license-forwarder', {
            directory: __dirname,
        });
        images = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(dockerContainer),
        };
        const workerStack = new core_1.Stack(app, 'WorkerStack', { env });
        workerFleet = new lib_1.WorkerInstanceFleet(workerStack, 'workerFleet', {
            vpc,
            workerMachineImage: new aws_ec2_1.GenericWindowsImage({
                'us-east-1': 'ami-any',
            }),
            renderQueue,
            securityGroup: aws_ec2_1.SecurityGroup.fromSecurityGroupId(dependencyStack, 'SG', 'sg-123456789', {
                allowAllOutbound: false,
            }),
        });
        licenses = [lib_1.UsageBasedLicense.forMaya()];
        stack = new core_1.Stack(app, 'Stack', { env });
        // WHEN
        ubl = new lib_1.UsageBasedLicensing(stack, 'UBL', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc,
        });
    });
    test('creates an ECS cluster', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Cluster'));
    });
    describe('creates an ASG', () => {
        test('defaults', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '1',
                MaxSize: '1',
                VPCZoneIdentifier: [
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet1Subnet*`),
                    },
                    {
                        'Fn::ImportValue': assert_1.stringLike(`${dependencyStack.stackName}:ExportsOutputRefVPCPrivateSubnet2Subnet*`),
                    },
                ],
            }));
        });
        test('capacity can be specified', () => {
            // WHEN
            const isolatedStack = new core_1.Stack(app, 'MyStack', { env });
            new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                certificateSecret,
                desiredCount: 2,
                images,
                licenses,
                renderQueue,
                vpc,
            });
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: '2',
                MaxSize: '2',
            }));
        });
        test('gives write access to log group', () => {
            // GIVEN
            const logGroup = ubl.node.findChild('UBLLogGroup');
            const asgRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.asg.role.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: assert_1.arrayWith('logs:CreateLogStream', 'logs:PutLogEvents'),
                        Effect: 'Allow',
                        Resource: stack.resolve(logGroup.logGroupArn),
                    }),
                    Version: '2012-10-17',
                },
                Roles: assert_1.arrayWith({ Ref: asgRoleLogicalId }),
            }));
        });
    });
    describe('creates an ECS service', () => {
        test('associated with the cluster', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                Cluster: { Ref: stack.getLogicalId(ubl.cluster.node.defaultChild) },
            }));
        });
        describe('DesiredCount', () => {
            test('defaults to 1', () => {
                // THEN
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: 1,
                }));
            });
            test('can be specified', () => {
                // GIVEN
                const desiredCount = 2;
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
                // WHEN
                new lib_1.UsageBasedLicensing(isolatedStack, 'UBL', {
                    certificateSecret,
                    images,
                    licenses,
                    renderQueue,
                    vpc,
                    desiredCount,
                });
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desiredCount,
                }));
            });
        });
        test('sets launch type to EC2', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                LaunchType: 'EC2',
            }));
        });
        test('sets distinct instance placement constraint', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                PlacementConstraints: assert_1.arrayWith({ Type: 'distinctInstance' }),
            }));
        });
        test('uses the task definition', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                TaskDefinition: { Ref: stack.getLogicalId(ubl.service.taskDefinition.node.defaultChild) },
            }));
        });
        test('with the correct deployment configuration', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                DeploymentConfiguration: {
                    MaximumPercent: 100,
                    MinimumHealthyPercent: 0,
                },
            }));
        });
    });
    describe('creates a task definition', () => {
        test('container name is LicenseForwarderContainer', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Name: 'LicenseForwarderContainer',
                    },
                ],
            }));
        });
        test('container is marked essential', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Essential: true,
                    },
                ],
            }));
        });
        test('with increased ulimits', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Ulimits: [
                            {
                                HardLimit: 200000,
                                Name: 'nofile',
                                SoftLimit: 200000,
                            },
                            {
                                HardLimit: 64000,
                                Name: 'nproc',
                                SoftLimit: 64000,
                            },
                        ],
                    },
                ],
            }));
        });
        test('with awslogs log driver', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        LogConfiguration: {
                            LogDriver: 'awslogs',
                            Options: {
                                'awslogs-group': {},
                                'awslogs-stream-prefix': 'LicenseForwarder',
                                'awslogs-region': env.region,
                            },
                        },
                    },
                ],
            }));
        });
        test('configures UBL certificates', () => {
            // GIVEN
            const taskRoleLogicalId = core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild);
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_CERTIFICATES_URI',
                            Value: certificateSecret.secretArn,
                        }),
                    },
                ],
                TaskRoleArn: {
                    'Fn::GetAtt': [
                        taskRoleLogicalId,
                        'Arn',
                    ],
                },
            }));
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: certificateSecret.secretArn,
                        },
                    ],
                    Version: '2012-10-17',
                },
                Roles: [
                    { Ref: core_1.Stack.of(ubl).getLogicalId(ubl.service.taskDefinition.taskRole.node.defaultChild) },
                ],
            }));
        });
        test('uses host networking', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                NetworkMode: 'host',
            }));
        });
        test('is marked EC2 compatible only', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                RequiresCompatibilities: ['EC2'],
            }));
        });
    });
    test('License Forwarder subnet selection', () => {
        // GIVEN
        const publicSubnetIds = ['PublicSubnet1', 'PublicSubnet2'];
        const vpcFromAttributes = aws_ec2_1.Vpc.fromVpcAttributes(dependencyStack, 'AttrVpc', {
            availabilityZones: ['us-east-1a', 'us-east-1b'],
            vpcId: 'vpcid',
            publicSubnetIds,
        });
        stack = new core_1.Stack(app, 'IsolatedStack', { env });
        // WHEN
        new lib_1.UsageBasedLicensing(stack, 'licenseForwarder', {
            certificateSecret,
            images,
            licenses,
            renderQueue,
            vpc: vpcFromAttributes,
            vpcSubnets: { subnetType: aws_ec2_1.SubnetType.PUBLIC },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: publicSubnetIds,
        }));
    });
    describe('license limits', () => {
        test('multiple licenses with limits', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
            // WHEN
            new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                vpc,
                images,
                certificateSecret,
                renderQueue,
                licenses: [
                    lib_1.UsageBasedLicense.forMaya(10),
                    lib_1.UsageBasedLicense.forVray(10),
                ],
            });
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'UBL_LIMITS',
                            Value: 'maya:10;vray:10',
                        }),
                    },
                ],
            }));
        });
        test.each([
            ['3dsMax', lib_1.UsageBasedLicense.for3dsMax(10), [27002]],
            ['Arnold', lib_1.UsageBasedLicense.forArnold(10), [5056, 7056]],
            ['Cinema4D', lib_1.UsageBasedLicense.forCinema4D(10), [5057, 7057]],
            ['Clarisse', lib_1.UsageBasedLicense.forClarisse(10), [40500]],
            ['Houdini', lib_1.UsageBasedLicense.forHoudini(10), [1715]],
            ['Katana', lib_1.UsageBasedLicense.forKatana(10), [4101, 6101]],
            ['KeyShot', lib_1.UsageBasedLicense.forKeyShot(10), [27003, 2703]],
            ['Krakatoa', lib_1.UsageBasedLicense.forKrakatoa(10), [27000, 2700]],
            ['Mantra', lib_1.UsageBasedLicense.forMantra(10), [1716]],
            ['Maxwell', lib_1.UsageBasedLicense.forMaxwell(10), [5055, 7055]],
            ['Maya', lib_1.UsageBasedLicense.forMaya(10), [27002, 2702]],
            ['Nuke', lib_1.UsageBasedLicense.forNuke(10), [4101, 6101]],
            ['RealFlow', lib_1.UsageBasedLicense.forRealFlow(10), [5055, 7055]],
            ['RedShift', lib_1.UsageBasedLicense.forRedShift(10), [5054, 7054]],
            ['Vray', lib_1.UsageBasedLicense.forVray(10), [30306]],
            ['Yeti', lib_1.UsageBasedLicense.forYeti(10), [5053, 7053]],
        ])('Test open port for license type %s', (_licenseName, license, ports) => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack', { env });
            // WHEN
            ubl = new lib_1.UsageBasedLicensing(isolatedStack, 'licenseForwarder', {
                vpc,
                certificateSecret,
                licenses: [
                    license,
                ],
                renderQueue,
                images,
            });
            ubl.grantPortAccess(workerFleet, [license]);
            // THEN
            ports.forEach(port => {
                const ublAsgSecurityGroup = ubl.asg.connections.securityGroups[0].node.defaultChild;
                const ublAsgSecurityGroupLogicalId = isolatedStack.getLogicalId(ublAsgSecurityGroup);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: port,
                    GroupId: {
                        'Fn::GetAtt': [
                            ublAsgSecurityGroupLogicalId,
                            'GroupId',
                        ],
                    },
                    SourceSecurityGroupId: {
                        'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(workerFleet)}:ExportsOutputFnGetAttworkerFleetInstanceSecurityGroupB00C2885GroupId60416F0A`),
                    },
                }));
            });
        });
        test('requires one usage based license', () => {
            // Without any licenses
            expect(() => {
                new lib_1.UsageBasedLicensing(dependencyStack, 'licenseForwarder', {
                    vpc,
                    images,
                    certificateSecret: certificateSecret,
                    licenses: [],
                    renderQueue,
                });
            }).toThrowError('Should be specified at least one license with defined limit.');
        });
    });
    describe('configures render queue', () => {
        test('adds ingress rule for asg', () => {
            const ublAsgSg = ubl.asg.connections.securityGroups[0].node.defaultChild;
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                FromPort: 8080,
                ToPort: 8080,
                GroupId: {
                    'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLBSecurityGroup*`),
                },
                SourceSecurityGroupId: {
                    'Fn::GetAtt': [
                        core_1.Stack.of(ubl).getLogicalId(ublAsgSg),
                        'GroupId',
                    ],
                },
            }));
        });
        test('sets RENDER_QUEUE_URI environment variable', () => {
            // THEN
            assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                ContainerDefinitions: [
                    {
                        Environment: assert_1.arrayWith({
                            Name: 'RENDER_QUEUE_URI',
                            Value: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'http://',
                                        {
                                            'Fn::ImportValue': assert_1.stringLike(`${core_1.Stack.of(renderQueue).stackName}:ExportsOutputFnGetAttRQNonDefaultPortLB*`),
                                        },
                                        ':8080',
                                    ],
                                ],
                            },
                        }),
                    },
                ],
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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