"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('Stage', () => {
    // GIVEN
    const STAGE_PATH = path.join(__dirname, 'assets');
    /**
     * A sub-class of Stage that makes the constructor public for testing purposes.
     *
     * This is to be able to specify a manifest rather than load it from the file-system via {@link Stage.fromDirectory}.
     */
    class StageWithPulicConstructor extends lib_1.Stage {
        constructor(props) {
            super(props);
        }
    }
    beforeEach(() => {
        jest.resetModules();
    });
    describe('.fromDirectory', () => {
        // GIVEN
        const manifest = {
            schema: 1,
            version: '1.2.3.4',
            recipes: {},
        };
        const readFileSync = jest.spyOn(fs, 'readFileSync');
        let stage;
        beforeAll(() => {
            readFileSync.mockReturnValue(JSON.stringify(manifest));
            stage = require('../lib').Stage.fromDirectory(STAGE_PATH); // eslint-disable-line
        });
        afterAll(() => {
            readFileSync.mockRestore();
            jest.resetModules();
        });
        test('has dirPath', () => {
            // THEN
            expect(stage.dirPath).toBe(STAGE_PATH);
        });
        test('loads manifest.json from directory', () => {
            // THEN
            const expectedPath = path.join(STAGE_PATH, 'manifest.json');
            expect(readFileSync).toHaveBeenCalledWith(expectedPath, expect.objectContaining({
                encoding: 'utf8',
            }));
            expect(stage.manifest).toEqual(manifest);
        });
    });
    test('has manifest', () => {
        const manifest = {
            schema: 1,
            version: '1.2.3.4',
            recipes: {
                a: {
                    title: 'a-title',
                    description: 'a-description',
                    buildArgs: {
                        argone: 'a-argone-value',
                        argtwo: 'a-argtwo-value',
                    },
                },
                b: {
                    title: 'b-title',
                    description: 'b-description',
                    buildArgs: {
                        argone: 'b-argone-value',
                        argtwo: 'b-argtwo-value',
                    },
                },
            },
        };
        const stage = new StageWithPulicConstructor({
            manifest,
            path: STAGE_PATH,
        });
        // THEN
        expect(stage.manifest).toEqual(manifest);
    });
    describe('manifest validation', () => {
        test.each([
            [
                'mainfest wrong type',
                {
                    manifest: 1,
                    expectedError: /Expected object but got/,
                },
            ],
            [
                'manifest version not string',
                {
                    manifest: {
                        version: 1,
                        recipes: {},
                    },
                },
            ],
            [
                'missing schema',
                {
                    manifest: {
                        version: '1.2.3.4',
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "schema" key/,
                },
            ],
            [
                'wrong schema type', {
                    manifest: {
                        schema: 'a',
                        version: '1.2.3.4',
                        recipes: {},
                    },
                    expectedError: /Expected a numeric "schema" but got:/,
                },
            ],
            [
                'missing version', {
                    manifest: {
                        schema: 1,
                        recipes: {},
                    },
                    expectedError: /Manifest contains no "version" key/,
                },
            ],
        ])('%s', (_name, testcase) => {
            const { manifest, expectedError } = testcase;
            // WHEN
            function when() {
                new StageWithPulicConstructor({
                    path: STAGE_PATH,
                    manifest,
                });
            }
            expect(when).toThrow(expectedError);
        });
    });
    describe('valid recipes', () => {
        test.each([
            [
                'build args and no target',
                {
                    title: 'Eugene\'s Favorite Cookies',
                    description: 'C is for cookie. That\'s good enough for me',
                    buildArgs: {
                        flour: '3 cups',
                        butter: '1 cup',
                        sugar: '1/2 cup',
                        egg: '2',
                        bakingpowder: '1 tsp',
                        bakingsoda: '1 tsb',
                        vanilla: '1 tsp',
                        salt: '1 pinch',
                    },
                },
            ],
            [
                'buildArgs and a target',
                {
                    title: 'Josh\'s S\'mores',
                    description: 'a secret family recipe',
                    buildArgs: {
                        grahamcracker: '2',
                        marshmellow: '1',
                        chocolate: '2',
                        campfire: '1',
                    },
                    target: 'simplicity',
                },
            ],
            [
                'target and no buildArgs',
                {
                    title: 'Jericho\s special brownie batch with a strawberry milkshake',
                    description: 'a slight improvement to the original recipe',
                    target: 'target',
                },
            ],
            [
                'no target or buildArgs',
                {
                    title: 'Yash\'s Tequila Lime Shot (TLS)',
                    description: 'A sure-fire way to brighten your day',
                },
            ],
        ])('%s', (_scenario, recipe) => {
            // GIVEN
            const manifest = {
                recipes: {
                    recipeName: recipe,
                },
                schema: 1,
                version: '1.2.3.4',
            };
            // WHEN
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // THEN
            expect(stage.manifest.recipes.recipeName).toEqual(recipe);
        });
    });
    describe('.imageFromRecipe()', () => {
        let app;
        let stack;
        beforeEach(() => {
            // GIVEN
            app = new core_1.App();
            stack = new core_1.Stack(app, 'Stack');
        });
        test('creates DockerImageAssets from existing recipes', () => {
            jest.resetModules();
            const mockDockerImageAssetConstructor = jest.fn();
            jest.mock('@aws-cdk/aws-ecr-assets', () => {
                class DockerImageAsset {
                    constructor(...args) {
                        mockDockerImageAssetConstructor(...args);
                    }
                }
                return {
                    DockerImageAsset,
                };
            });
            // Import the class under test now that the ec2 mock is in place
            const ReloadedStage = jest.requireActual('../lib').Stage; // eslint-disable-line
            class ReloadedStageWithPublicConstructor extends ReloadedStage {
                constructor(props) {
                    super(props);
                }
            }
            // GIVEN
            const recipeName = 'myname';
            const recipe = {
                description: 'some description',
                title: 'some title',
                buildArgs: {
                    a: 'a',
                    b: 'b',
                },
            };
            const stage = new ReloadedStageWithPublicConstructor({
                path: STAGE_PATH,
                manifest: {
                    version: '1.2.3.4',
                    schema: 1,
                    recipes: {
                        [recipeName]: recipe,
                    },
                },
            });
            // WHEN
            stage.imageFromRecipe(stack, 'Image', recipeName);
            // THEN
            expect(mockDockerImageAssetConstructor).toHaveBeenCalledWith(stack, 'Image', {
                ...recipe,
                directory: STAGE_PATH,
            });
            expect(mockDockerImageAssetConstructor.mock.calls[0][2]).not.toMatchObject({
                target: expect.anything(),
            });
        });
        test('throws exception when recipe does not exist', () => {
            // GIVEN
            const manifest = {
                schema: 1,
                version: '1.2.3.4',
                recipes: {},
            };
            const invalidRecipeName = 'this-recipe-does-not-exist';
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            function when() {
                stage.imageFromRecipe(stack, 'Image', invalidRecipeName);
            }
            // THEN
            expect(when).toThrowError(`No such recipe: ${invalidRecipeName}`);
        });
    });
    describe('.getVersion()', () => {
        test('returns a version in the success case', () => {
            var _a;
            // GIVEN
            const app = new core_1.App();
            const stack = new core_1.Stack(app, 'Stack');
            const manifest = {
                schema: 1,
                version: '1.2.3.4',
                recipes: {},
            };
            const stage = new StageWithPulicConstructor({
                manifest,
                path: STAGE_PATH,
            });
            // WHEN
            const version = stage.getVersion(stack, 'Version');
            const linuxFullVersionString = version.linuxFullVersionString();
            // THEN
            expect(version.majorVersion).toEqual(1);
            expect(version.minorVersion).toEqual(2);
            expect(version.releaseVersion).toEqual(3);
            expect(version.linuxInstallers).toBeDefined();
            expect((_a = version.linuxInstallers) === null || _a === void 0 ? void 0 : _a.patchVersion).toEqual(4);
            expect(linuxFullVersionString).toBeDefined();
            expect(linuxFullVersionString).toEqual('1.2.3.4');
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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