"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_certificatemanager_1 = require("@aws-cdk/aws-certificatemanager");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const __1 = require("../..");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
describe('RenderQueue', () => {
    let app;
    let dependencyStack;
    let stack;
    let vpc;
    let rcsImage;
    let images;
    let repository;
    let version;
    let renderQueueCommon;
    // GIVEN
    beforeEach(() => {
        app = new core_1.App();
        dependencyStack = new core_1.Stack(app, 'DepStack');
        vpc = new aws_ec2_1.Vpc(dependencyStack, 'Vpc');
        version = lib_1.VersionQuery.exact(dependencyStack, 'Version', {
            majorVersion: 10,
            minorVersion: 1,
            releaseVersion: 9,
            patchVersion: 1,
        });
        repository = new lib_1.Repository(dependencyStack, 'Repo', {
            version,
            vpc,
        });
        stack = new core_1.Stack(app, 'Stack');
        rcsImage = aws_ecs_1.ContainerImage.fromAsset(__dirname);
        images = {
            remoteConnectionServer: rcsImage,
        };
        renderQueueCommon = new lib_1.RenderQueue(stack, 'RenderQueueCommon', {
            images,
            repository,
            version,
            vpc,
        });
    });
    afterEach(() => {
        jest.resetAllMocks();
    });
    test('creates cluster', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Cluster'));
    });
    test('creates service', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::Service'));
    });
    test('creates task definition', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ECS::TaskDefinition'));
    });
    test('closed ingress by default', () => {
        // THEN
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::EC2::SecurityGroup', {
            // The openListener=true option would create an ingress rule in the listener's SG.
            // make sure that we don't have that.
            SecurityGroupIngress: assert_1.arrayWith(assert_1.objectLike({})),
        }));
    });
    test('creates load balancer with default values', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
            LoadBalancerAttributes: [
                {
                    Key: 'deletion_protection.enabled',
                    Value: 'true',
                },
            ],
            Scheme: 'internal',
        }));
    });
    test('creates a log group with default prefix of "/renderfarm/"', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/RenderQueueCommon',
            RetentionInDays: 3,
        }));
    });
    test('configure the container log driver', () => {
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
            ContainerDefinitions: [
                assert_1.objectLike({
                    LogConfiguration: {
                        LogDriver: 'awslogs',
                        Options: {
                            'awslogs-group': {
                                'Fn::Select': [
                                    6,
                                    {
                                        'Fn::Split': [
                                            ':',
                                            {
                                                'Fn::Join': [
                                                    '',
                                                    [
                                                        'arn:',
                                                        { Ref: 'AWS::Partition' },
                                                        ':logs:',
                                                        { Ref: 'AWS::Region' },
                                                        ':',
                                                        { Ref: 'AWS::AccountId' },
                                                        ':log-group:',
                                                        {
                                                            'Fn::GetAtt': assert_1.arrayWith('LogGroupName'),
                                                        },
                                                    ],
                                                ],
                                            },
                                        ],
                                    },
                                ],
                            },
                            'awslogs-stream-prefix': 'RCS',
                            'awslogs-region': { Ref: 'AWS::Region' },
                        },
                    },
                }),
            ],
        }));
    });
    test('child dependencies added', () => {
        // GIVEN
        const host = new aws_ec2_1.Instance(stack, 'Host', {
            vpc,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // WHEN
        renderQueueCommon.addChildDependency(host);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
            DependsOn: assert_1.arrayWith('RenderQueueCommonLBPublicListener935F5635', 'RenderQueueCommonRCSTask2A4D5EA5'),
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    describe('renderQueueSize.min', () => {
        describe('defaults to 1', () => {
            function assertSpecifiesMinSize(stackToAssert) {
                assert_1.expect(stackToAssert).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    MinSize: '1',
                }));
            }
            test('renderQueueSize unspecified', () => {
                // THEN
                assertSpecifiesMinSize(stack);
            });
            test('renderQueueSize.min unspecified', () => {
                // GIVEN
                const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', {
                    images,
                    repository,
                    version,
                    vpc,
                    renderQueueSize: {},
                });
                // THEN
                assertSpecifiesMinSize(isolatedStack);
            });
        });
        // Asserts that only one RCS container and ASG instance can be created.
        // Deadline currently requires that successive API requests are serviced by a single RCS.
        test.each([
            [0],
            [2],
        ])('clamps to 1 - using %d', (min) => {
            // GIVEN
            const props = {
                images,
                repository,
                version,
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow('renderQueueSize.min');
        });
        // Asserts that when the renderQueueSize.min prop is specified, the underlying ASG's min property is set accordingly.
        test('configures minimum number of ASG instances', () => {
            // GIVEN
            const min = 1;
            const props = {
                images,
                repository,
                version,
                vpc,
                renderQueueSize: {
                    min,
                },
            };
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                MinSize: min.toString(),
            }));
        });
    });
    describe('renderQueueSize.desired', () => {
        describe('defaults', () => {
            test('unset ASG desired', () => {
                assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: assert_1.ABSENT,
                }));
            });
        });
        test('caps at 1', () => {
            // GIVEN
            const props = {
                images,
                repository,
                version,
                vpc,
                renderQueueSize: {
                    desired: 2,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(stack, 'RenderQueue', props);
            })
                // THEN
                .toThrow('renderQueueSize.desired cannot be greater than 1');
        });
        describe('is specified', () => {
            const desired = 1;
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    renderQueueSize: {
                        desired,
                    },
                };
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('configures desired number of ASG instances', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
                    DesiredCapacity: desired.toString(),
                }));
            });
            // THEN
            test('configures desired number of ECS tasks in the service', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::Service', {
                    DesiredCount: desired,
                }));
            });
        });
    });
    describe('trafficEncryption', () => {
        describe('defaults', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {},
                };
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('to HTTPS internally between ALB and RCS', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('to HTTP externally between clients and ALB', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('when interalProtocol is HTTPS', () => {
            let isolatedStack;
            let renderQueue;
            let caCertPemLogicalId;
            let caCertPkcsLogicalId;
            let caCertPkcsPassphraseLogicalId;
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS,
                    },
                };
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
                caCertPemLogicalId = isolatedStack.getLogicalId(renderQueue.node.findChild('TlsCaCertPem').node.defaultChild);
                const caCertPkcs = renderQueue.node.findChild('TlsRcsCertBundle');
                const caCertPkcsPassphrase = caCertPkcs.node.findChild('Passphrase');
                caCertPkcsLogicalId = isolatedStack.getLogicalId(caCertPkcs.node.defaultChild);
                caCertPkcsPassphraseLogicalId = isolatedStack.getLogicalId(caCertPkcsPassphrase.node.defaultChild);
            });
            // THEN
            test('ALB connects with HTTPS to port 4433', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTPS',
                    Port: 4433,
                }));
            });
            test('creates RCS cert', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_X509Generator', {
                    ServiceToken: {
                        'Fn::GetAtt': assert_1.arrayWith('Arn'),
                    },
                    DistinguishedName: { CN: 'renderfarm.local' },
                    Secret: {
                        NamePrefix: 'IsolatedStack/RenderQueue/TlsCaCertPem',
                    },
                }));
            });
            test('grants read access to secrets containing the certs and passphrase', () => {
                const taskDef = renderQueue.node.findChild('RCSTask');
                const taskRoleLogicalId = isolatedStack.getLogicalId(taskDef.taskRole.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
                    PolicyDocument: {
                        Statement: assert_1.arrayWith({
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Action: [
                                'secretsmanager:GetSecretValue',
                                'secretsmanager:DescribeSecret',
                            ],
                            Effect: 'Allow',
                            Resource: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                        Version: '2012-10-17',
                    },
                    Roles: assert_1.arrayWith({ Ref: taskRoleLogicalId }),
                }));
            });
            test('configures environment variables for cert secret URIs', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ECS::TaskDefinition', {
                    ContainerDefinitions: assert_1.arrayWith(assert_1.deepObjectLike({
                        Environment: assert_1.arrayWith({
                            Name: 'RCS_TLS_CA_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPemLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_URI',
                            Value: {
                                'Fn::GetAtt': [
                                    caCertPkcsLogicalId,
                                    'Cert',
                                ],
                            },
                        }, {
                            Name: 'RCS_TLS_CERT_PASSPHRASE_URI',
                            Value: { Ref: caCertPkcsPassphraseLogicalId },
                        }),
                    })),
                }));
            });
        });
        describe('when internal protocol is HTTP', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                    },
                };
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('no certs are created', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('Custom::RFDK_X509Generator'));
            });
            test('ALB connects with HTTP to port 8080', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
                    Protocol: 'HTTP',
                    Port: 8080,
                }));
            });
        });
        describe('externalProtocol is HTTPS', () => {
            let isolatedStack;
            const CERT_ARN = 'certarn';
            const CA_ARN = 'caarn';
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('configures the ALB listener to use the specified ACM certificate', () => {
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                    Certificates: assert_1.arrayWith({
                        CertificateArn: CERT_ARN,
                    }),
                }));
            });
            test('raises an error when a cert is specified without a hostname', () => {
                // GIVEN
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Cert', 'certArn'),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert2', CA_ARN),
                        },
                    },
                };
                // WHEN
                expect(() => {
                    new lib_1.RenderQueue(stack, 'RenderQueue', props);
                })
                    // THEN
                    .toThrow(/A hostname must be provided when the external protocol is HTTPS/);
            });
        });
        describe('externalProtocol is HTTPS importing cert', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                    subject: {
                        cn: `ca.${ZONE_NAME}`,
                    },
                });
                const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                    subject: {
                        cn: `server.${ZONE_NAME}`,
                    },
                    signingCertificate: caCert,
                });
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    trafficEncryption: {
                        externalTLS: {
                            rfdkCertificate: serverCert,
                        },
                        internalProtocol: aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP,
                    },
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('sets the listener port to 4433', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Port: 4433,
                }));
            });
            test('sets the listener protocol to HTTPS', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
                    Protocol: 'HTTPS',
                }));
            });
            test('Imports Cert to ACM', () => {
                // THEN
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
                    X509CertificatePem: {
                        Cert: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Cert',
                            ],
                        },
                        Key: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'Key',
                            ],
                        },
                        Passphrase: {
                            Ref: 'ServerCertPassphraseE4C3CB38',
                        },
                        CertChain: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                }));
            });
        });
        test('Throws if given ACM cert and RFDK Cert', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const CA_ARN = 'caArn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const caCert = new __1.X509CertificatePem(isolatedStack, 'CaCert', {
                subject: {
                    cn: `ca.${ZONE_NAME}`,
                },
            });
            const serverCert = new __1.X509CertificatePem(isolatedStack, 'ServerCert', {
                subject: {
                    cn: `server.${ZONE_NAME}`,
                },
                signingCertificate: caCert,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                        acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(isolatedStack, 'CA_Cert', CA_ARN),
                        rfdkCertificate: serverCert,
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if no Cert given', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                trafficEncryption: {
                    externalTLS: {},
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS/);
        });
        test('Throws if ACM Cert is given without a cert chain', () => {
            // GIVEN
            const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certArn';
            const zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                vpc,
                zoneName: ZONE_NAME,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                trafficEncryption: {
                    externalTLS: {
                        acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(isolatedStack, 'Certificate', CERT_ARN),
                    },
                },
                hostname: {
                    zone,
                },
            };
            // WHEN
            expect(() => {
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            })
                // THEN
                .toThrow(/externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate./);
        });
    });
    describe('Client Connection', () => {
        describe('externalProtocol is http', () => {
            let isolatedStack;
            let zone;
            const ZONE_NAME = 'renderfarm.local';
            let rq;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/http:\/\/.*:8080$/);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '" --render-queue "http://',
                            {
                                'Fn::GetAtt': [
                                    'RenderQueueLB235D35F4',
                                    'DNSName',
                                ],
                            },
                            ':8080" \n' +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '" --render-queue "http://',
                            {
                                'Fn::GetAtt': [
                                    'RenderQueueLB235D35F4',
                                    'DNSName',
                                ],
                            },
                            ':8080"  2>&1\n' +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 8080,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::Instance', {
                    DependsOn: assert_1.arrayWith('RenderQueueLBPublicListenerBBF15D5F', 'RenderQueueRCSTaskA9AE70D3'),
                }, assert_1.ResourcePart.CompleteDefinition));
            });
        });
        describe('externalProtocol is https', () => {
            let isolatedStack;
            let zone;
            let rq;
            const ZONE_NAME = 'renderfarm.local';
            const CERT_ARN = 'certarn';
            const CA_ARN = 'caarn';
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                zone = new aws_route53_1.PrivateHostedZone(isolatedStack, 'RenderQueueZone', {
                    vpc,
                    zoneName: ZONE_NAME,
                });
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    hostname: {
                        zone,
                    },
                    trafficEncryption: {
                        externalTLS: {
                            acmCertificate: aws_certificatemanager_1.Certificate.fromCertificateArn(stack, 'Certificate', CERT_ARN),
                            acmCertificateChain: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'CA_Cert', CA_ARN),
                        },
                    },
                };
                // WHEN
                rq = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            test('ECS can connect', () => {
                // WHEN
                const hosts = [new aws_ec2_1.Instance(isolatedStack, 'Host', {
                        vpc,
                        instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                        machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                    })];
                const role = new aws_iam_1.Role(isolatedStack, 'Role', { assumedBy: new aws_iam_1.AccountRootPrincipal() });
                const env = rq.configureClientECS({
                    hosts,
                    grantee: role,
                });
                // THEN
                expect(env).toHaveProperty('RENDER_QUEUE_URI');
                expect(env.RENDER_QUEUE_URI).toMatch(/https:\/\/.*:4433$/);
                expect(env).toHaveProperty('RENDER_QUEUE_TLS_CA_CERT_URI', CA_ARN);
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(hosts[0].connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Linux Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\')\naws s3 cp \'s3://',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' \'/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\'\n' +
                                'if [ -f \"/etc/profile.d/deadlineclient.sh\" ]; then\n' +
                                '  source \"/etc/profile.d/deadlineclient.sh\"\n' +
                                'fi\n' +
                                '"${DEADLINE_PATH}/deadlinecommand" -executeScriptNoGui "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '" --render-queue "https://',
                            {
                                'Fn::GetAtt': [
                                    'RenderQueueLB235D35F4',
                                    'DNSName',
                                ],
                            },
                            ':4433" --tls-ca "caarn"\n' +
                                'rm -f "/tmp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'if service --status-all | grep -q "Deadline 10 Launcher"; then\n' +
                                '  service deadline10launcher restart\n' +
                                'fi',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
            test('Windows Instance can connect', () => {
                // WHEN
                const host = new aws_ec2_1.Instance(isolatedStack, 'Host', {
                    vpc,
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R4, aws_ec2_1.InstanceSize.LARGE),
                    machineImage: aws_ec2_1.MachineImage.latestWindows(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_CORE_BASE),
                });
                rq.configureClientInstance({
                    host,
                });
                // THEN
                const userData = isolatedStack.resolve(host.userData.render());
                expect(userData).toStrictEqual({
                    'Fn::Join': [
                        '',
                        [
                            '<powershell>mkdir (Split-Path -Path \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' ) -ea 0\n' +
                                'Read-S3Object -BucketName \'',
                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Bucket },
                            '\' -key \'',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -file \'C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '\' -ErrorAction Stop\n' +
                                '$ErrorActionPreference = "Stop"\n' +
                                '$DEADLINE_PATH = (get-item env:"DEADLINE_PATH").Value\n' +
                                '& "$DEADLINE_PATH/deadlinecommand.exe" -executeScriptNoGui "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '" --render-queue "https://',
                            {
                                'Fn::GetAtt': [
                                    'RenderQueueLB235D35F4',
                                    'DNSName',
                                ],
                            },
                            ':4433" --tls-ca \"caarn\" 2>&1\n' +
                                'Remove-Item -Path "C:/temp/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.RQ_CONNECTION_ASSET.Key },
                                        ],
                                    },
                                ],
                            },
                            '"\n' +
                                'If (Get-Service "deadline10launcherservice" -ErrorAction SilentlyContinue) {\n' +
                                '  Restart-Service "deadline10launcherservice"\n' +
                                '} Else {\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" -shutdownall 2>&1\n' +
                                '  & "$DEADLINE_PATH/deadlinelauncher.exe" 2>&1\n' +
                                '}</powershell>',
                        ],
                    ],
                });
                // Make sure we execute the script with the correct args
                assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::EC2::SecurityGroupIngress', {
                    IpProtocol: 'tcp',
                    ToPort: 4433,
                    SourceSecurityGroupId: {
                        'Fn::GetAtt': [
                            isolatedStack.getLogicalId(host.connections.securityGroups[0].node.defaultChild),
                            'GroupId',
                        ],
                    },
                }));
            });
        });
    });
    test('can specify subnets', () => {
        // GIVEN
        const subnets = [
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet1', {
                subnetId: 'SubnetID1',
                availabilityZone: 'us-west-2a',
            }),
            aws_ec2_1.Subnet.fromSubnetAttributes(dependencyStack, 'Subnet2', {
                subnetId: 'SubnetID2',
                availabilityZone: 'us-west-2b',
            }),
        ];
        const props = {
            images,
            repository,
            version,
            vpc,
            vpcSubnets: {
                subnets,
            },
        };
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup', {
            VPCZoneIdentifier: assert_1.arrayWith('SubnetID1', 'SubnetID2'),
        }));
    });
    test('can specify instance type', () => {
        // GIVEN
        const props = {
            images,
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.C5, aws_ec2_1.InstanceSize.LARGE),
            repository,
            version,
            vpc,
        };
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'c5.large',
        }));
    });
    test('no deletion protection', () => {
        // GIVEN
        const props = {
            images,
            repository,
            version,
            vpc,
            deletionProtection: false,
        };
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'deletion_protection.enabled',
                Value: 'true',
            }),
            Scheme: assert_1.ABSENT,
            Type: assert_1.ABSENT,
        })));
    });
    test('drop invalid http header fields enabled', () => {
        // GIVEN
        const props = {
            images,
            repository,
            version,
            vpc,
        };
        const isolatedStack = new core_1.Stack(app, 'IsolatedStack');
        // WHEN
        new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
        // THEN
        assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
            LoadBalancerAttributes: assert_1.arrayWith({
                Key: 'routing.http.drop_invalid_header_fields.enabled',
                Value: 'true',
            }),
        }));
    });
    describe('hostname', () => {
        // GIVEN
        const zoneName = 'mydomain.local';
        describe('not specified', () => {
            let isolatedStack;
            beforeEach(() => {
                // GIVEN
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                };
                // WHEN
                new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('does not create a record set', () => {
                assert_1.expect(isolatedStack).notTo(assert_1.haveResource('AWS::Route53::RecordSet'));
            });
        });
        describe('specified with zone but no hostname', () => {
            let zone;
            let isolatedStack;
            let renderQueue;
            beforeEach(() => {
                // GIVEN
                zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                    vpc,
                    zoneName,
                });
                isolatedStack = new core_1.Stack(app, 'IsolatedStack');
                const props = {
                    images,
                    repository,
                    version,
                    vpc,
                    hostname: {
                        zone,
                    },
                };
                // WHEN
                renderQueue = new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            });
            // THEN
            test('creates a record set using default hostname', () => {
                const loadBalancerLogicalId = dependencyStack.getLogicalId(renderQueue.loadBalancer.node.defaultChild);
                assert_1.expect(isolatedStack).to(assert_1.haveResource('AWS::Route53::RecordSet', {
                    Name: `renderqueue.${zoneName}.`,
                    Type: 'A',
                    AliasTarget: assert_1.objectLike({
                        HostedZoneId: {
                            'Fn::GetAtt': [
                                loadBalancerLogicalId,
                                'CanonicalHostedZoneID',
                            ],
                        },
                    }),
                }));
            });
        });
        test.each([
            ['rq.somedomain.local'],
            ['1startswithnumber'],
            ['-startswithhyphen'],
            ['endswith-'],
            ['contains$symbol'],
            ['thisstringisexactlysixtyfourcharacterswhichisonelargerthanthemax'],
        ])('.hostname validation - %s', (hostname) => {
            // GIVEN
            const zone = new aws_route53_1.PrivateHostedZone(dependencyStack, 'Zone', {
                zoneName: 'somedomain.local',
                vpc,
            });
            const props = {
                images,
                repository,
                version,
                vpc,
                hostname: {
                    hostname,
                    zone,
                },
            };
            // WHEN
            function when() {
                new lib_1.RenderQueue(stack, 'NewRenderQueue', props);
            }
            // THEN
            expect(when).toThrow(/Invalid RenderQueue hostname/);
        });
    });
    describe('Access Logs', () => {
        let isolatedStack;
        let isolatedVpc;
        let isolatedRepository;
        let isolatedVersion;
        let isolatedimages;
        let accessBucket;
        beforeEach(() => {
            // GIVEN
            isolatedStack = new core_1.Stack(app, 'IsolatedStack', {
                env: {
                    region: 'us-east-1',
                },
            });
            isolatedVpc = new aws_ec2_1.Vpc(isolatedStack, 'Vpc');
            isolatedVersion = lib_1.VersionQuery.exact(isolatedStack, 'Version', {
                majorVersion: 10,
                minorVersion: 1,
                releaseVersion: 9,
                patchVersion: 1,
            });
            isolatedRepository = new lib_1.Repository(isolatedStack, 'Repo', {
                version: isolatedVersion,
                vpc: isolatedVpc,
            });
            isolatedimages = {
                remoteConnectionServer: rcsImage,
            };
            accessBucket = new aws_s3_1.Bucket(isolatedStack, 'AccessBucket');
        });
        test('enabling access logs sets attributes and policies', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
        test('enabling access logs works with prefix', () => {
            // GIVEN
            const props = {
                images: isolatedimages,
                repository: isolatedRepository,
                version: isolatedVersion,
                vpc: isolatedVpc,
                accessLogs: {
                    destinationBucket: accessBucket,
                    prefix: 'PREFIX_STRING',
                },
            };
            // WHEN
            new lib_1.RenderQueue(isolatedStack, 'RenderQueue', props);
            // THEN
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
                LoadBalancerAttributes: assert_1.arrayWith({
                    Key: 'access_logs.s3.enabled',
                    Value: 'true',
                }, {
                    Key: 'access_logs.s3.bucket',
                    Value: {
                        Ref: 'AccessBucketE2803D76',
                    },
                }, {
                    Key: 'access_logs.s3.prefix',
                    Value: 'PREFIX_STRING',
                }),
            }));
            assert_1.expect(isolatedStack).to(assert_1.haveResourceLike('AWS::S3::BucketPolicy', {
                Bucket: {
                    Ref: 'AccessBucketE2803D76',
                },
                PolicyDocument: {
                    Statement: assert_1.arrayWith({
                        Action: 's3:PutObject',
                        Condition: {
                            StringEquals: {
                                's3:x-amz-acl': 'bucket-owner-full-control',
                            },
                        },
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }, {
                        Action: 's3:GetBucketAcl',
                        Effect: 'Allow',
                        Principal: {
                            Service: 'delivery.logs.amazonaws.com',
                        },
                        Resource: {
                            'Fn::GetAtt': [
                                'AccessBucketE2803D76',
                                'Arn',
                            ],
                        },
                    }, {
                        Action: [
                            's3:PutObject*',
                            's3:Abort*',
                        ],
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::127311923021:root',
                                    ],
                                ],
                            },
                        },
                        Resource: {
                            'Fn::Join': [
                                '',
                                [
                                    {
                                        'Fn::GetAtt': [
                                            'AccessBucketE2803D76',
                                            'Arn',
                                        ],
                                    },
                                    '/PREFIX_STRING/AWSLogs/',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    }),
                },
            }));
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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