"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.WorkerInstanceFleet = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
/**
 *  A new or imported Deadline Worker Fleet.
 */
class WorkerInstanceFleetBase extends core_1.Construct {
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * Resources Deployed
 * ------------------------
 * 1) An AutoScalingGroup to maintain the number of instances;
 * 2) An Instance Role and corresponding IAM Policy;
 * 3) A script asset which is uploaded to your deployment bucket used to configure the worker so it can connect to the Render Queue
 * 4) An aws-rfdk.CloudWatchAgent to configure sending logs to cloudwatch.
 *
 * @ResourcesDeployed
 */
class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.stack = core_1.Stack.of(scope);
        this.validateProps(props);
        // Launching the fleet with deadline workers.
        this.fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'Default', {
            vpc: props.vpc,
            instanceType: (props.instanceType ? props.instanceType : aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE)),
            machineImage: props.workerMachineImage,
            keyName: props.keyName,
            vpcSubnets: props.vpcSubnets ? props.vpcSubnets : {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: props.minCapacity,
            maxCapacity: props.maxCapacity,
            desiredCapacity: props.desiredCapacity,
            resourceSignalTimeout: WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT,
            healthCheck: aws_autoscaling_1.HealthCheck.elb({
                grace: WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL,
            }),
            role: props.role,
            spotPrice: (_a = props.spotPrice) === null || _a === void 0 ? void 0 : _a.toString(),
            blockDevices: props.blockDevices,
        });
        this.targetCapacity = parseInt(this.fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = this.fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: this.fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [this.fleet.autoScalingGroupArn],
                })],
        });
        this.fleet.node.defaultChild.metricsCollection = [{
                granularity: '1Minute',
                metrics: ['GroupDesiredCapacity'],
            }];
        if (props.securityGroup) {
            this.fleet.addSecurityGroup(props.securityGroup);
        }
        this.grantPrincipal = this.fleet.grantPrincipal;
        this.connections = this.fleet.connections;
        this.connections.allowToDefaultPort(props.renderQueue);
        let healthCheckPort = core_2.HealthMonitor.DEFAULT_HEALTH_CHECK_PORT;
        if (props.healthCheckConfig && props.healthCheckConfig.port) {
            healthCheckPort = props.healthCheckConfig.port;
        }
        // Configure the health monitoring if provided
        this.configureHealthMonitor(props, healthCheckPort);
        // Updating the user data with installation logs stream.
        this.configureCloudWatchLogStream(this.fleet, id, props.logGroupProps);
        props.renderQueue.configureClientInstance({
            host: this.fleet,
        });
        // Updating the user data with deadline repository installation commands.
        this.configureWorkerScript(this.fleet, props, healthCheckPort);
        // Updating the user data with successful cfn-signal commands.
        this.fleet.userData.addSignalOnExitCommand(this.fleet);
    }
    /**
     * Add the security group to all workers
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.fleet.addSecurityGroup(securityGroup);
    }
    configureCloudWatchLogStream(fleetInstance, id, logGroupProps) {
        const prefix = (logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) ? logGroupProps.logGroupPrefix : WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, `${id}LogGroupWrapper`, `${id}`, defaultedLogGroupProps);
        logGroup.grantWrite(fleetInstance);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(core_1.Duration.seconds(15));
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'UserdataExecution', 'C:\\ProgramData\\Amazon\\EC2-Windows\\Launch\\Log\\UserdataExecution.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlineslave*.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', 'C:\\ProgramData\\Thinkbox\\Deadline10\\logs\\deadlinelauncher*.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'WorkerLogs', '/var/log/Thinkbox/Deadline10/deadlineslave*.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'LauncherLogs', '/var/log/Thinkbox/Deadline10/deadlinelauncher*.log');
        new core_2.CloudWatchAgent(this, 'WorkerFleetLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: fleetInstance,
        });
    }
    configureWorkerScript(fleetInstance, props, healthCheckPort) {
        const configureWorkerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'WorkerConfigurationScript', {
            osType: fleetInstance.osType,
            baseName: 'configureWorker',
            rootDir: path.join(__dirname, '..', 'scripts/'),
        });
        // Converting to lower case, as groups and pools are all stored in lower case in deadline.
        const groups = props.groups ? props.groups.map(val => val.toLowerCase()).join(',') : '';
        const pools = props.pools ? props.pools.map(val => val.toLowerCase()).join(',') : '';
        configureWorkerScriptAsset.executeOn({
            host: fleetInstance,
            args: [
                `'${healthCheckPort}'`,
                `'${groups}'`,
                `'${pools}'`,
                `'${props.region || ''}'`,
            ],
        });
    }
    validateProps(props) {
        this.validateSpotPrice(props.spotPrice);
        this.validateArrayGroupsPoolsSyntax(props.groups, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'groups');
        this.validateArrayGroupsPoolsSyntax(props.pools, /^(?!none$)[a-zA-Z0-9-_]+$/i, 'pools');
        this.validateRegion(props.region, /^(?!none$|all$|unrecognized$)[a-zA-Z0-9-_]+$/i);
        this.validateBlockDevices(props.blockDevices);
    }
    validateSpotPrice(spotPrice) {
        if (spotPrice && !(spotPrice >= WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT && spotPrice <= WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT)) {
            throw new Error(`Invalid value: ${spotPrice} for property 'spotPrice'. Valid values can be any decimal between ${WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT} and ${WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT}.`);
        }
    }
    validateRegion(region, regex) {
        if (region && !regex.test(region)) {
            throw new Error(`Invalid value: ${region} for property 'region'. Valid characters are A-Z, a-z, 0-9, - and _. ‘All’, ‘none’ and ‘unrecognized’ are reserved names that cannot be used.`);
        }
    }
    validateArrayGroupsPoolsSyntax(array, regex, property) {
        if (array) {
            array.forEach(value => {
                if (!regex.test(value)) {
                    throw new Error(`Invalid value: ${value} for property '${property}'. Valid characters are A-Z, a-z, 0-9, - and _. Also, group 'none' is reserved as the default group.`);
                }
            });
        }
    }
    validateBlockDevices(blockDevices) {
        if (blockDevices === undefined) {
            this.node.addWarning(`The worker-fleet ${this.node.id} is being created without being provided any block devices so the Source AMI's devices will be used. ` +
                'Workers can have access to sensitive data so it is recommended to either explicitly encrypt the devices on the worker fleet or to ensure the source AMI\'s Drives are encrypted.');
        }
        else {
            blockDevices.forEach(device => {
                if (device.volume.ebsDevice === undefined) {
                    // Suppressed or Ephemeral Block Device
                    return;
                }
                // encrypted is not exposed as part of ebsDeviceProps so we need to confirm it exists then access it via [].
                // eslint-disable-next-line dot-notation
                if (('encrypted' in device.volume.ebsDevice === false) || ('encrypted' in device.volume.ebsDevice && !device.volume.ebsDevice['encrypted'])) {
                    this.node.addWarning(`The BlockDevice "${device.deviceName}" on the worker-fleet ${this.node.id} is not encrypted. ` +
                        'Workers can have access to sensitive data so it is recommended to encrypt the devices on the worker fleet.');
                }
            });
        }
    }
    configureHealthMonitor(props, healthCheckPort) {
        if (props.healthMonitor) {
            props.healthMonitor.registerFleet(this, props.healthCheckConfig || {
                port: healthCheckPort,
            });
        }
        else {
            this.node.addWarning(`The worker-fleet ${this.node.id} is being created without a health monitor attached to it. This means that the fleet will not automatically scale-in to 0 if the workers are unhealthy.`);
        }
    }
}
exports.WorkerInstanceFleet = WorkerInstanceFleet;
/**
 * The min limit for spot price.
 */
WorkerInstanceFleet.SPOT_PRICE_MIN_LIMIT = 0.001;
/**
 * The max limit for spot price.
 */
WorkerInstanceFleet.SPOT_PRICE_MAX_LIMIT = 255;
/**
 * This determines worker's health based on any hardware or software issues of EC2 instance.
 * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
 * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
 */
WorkerInstanceFleet.DEFAULT_HEALTH_CHECK_INTERVAL = core_1.Duration.minutes(1);
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
WorkerInstanceFleet.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
 * to take for launch and execute the user-data for deadline worker configuration. As we are setting
 * failure signals in the user-data, any failure will terminate deployment immediately.
 */
WorkerInstanceFleet.RESOURCE_SIGNAL_TIMEOUT = core_1.Duration.minutes(15);
//# sourceMappingURL=data:application/json;base64,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