/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { AutoScalingGroup, BlockDevice } from '@aws-cdk/aws-autoscaling';
import { IMetric } from '@aws-cdk/aws-cloudwatch';
import { Connections, IConnectable, IMachineImage, InstanceType, ISecurityGroup, IVpc, SubnetSelection } from '@aws-cdk/aws-ec2';
import { IApplicationLoadBalancerTarget } from '@aws-cdk/aws-elasticloadbalancingv2';
import { IGrantable, IPolicy, IPrincipal, IRole } from '@aws-cdk/aws-iam';
import { Construct, IResource, Stack } from '@aws-cdk/core';
import { HealthCheckConfig, IHealthMonitor, IMonitorableFleet, LogGroupFactoryProps } from '../../core';
import { IRenderQueue } from './render-queue';
/**
 * Interface for Deadline Worker Fleet.
 */
export interface IWorkerFleet extends IResource, IConnectable, IGrantable {
}
/**
 * Properties for the Deadline Worker Fleet.
 */
export interface WorkerInstanceFleetProps {
    /**
     * VPC to launch the worker fleet in.
     */
    readonly vpc: IVpc;
    /**
     * Security Group to assign to this fleet.
     *
     * @default - create new security group
     */
    readonly securityGroup?: ISecurityGroup;
    /**
     * An IAM role to associate with the instance profile assigned to its resources.
     *
     * The role must be assumable by the service principal `ec2.amazonaws.com`:
     *
     * @example
     *
     *    const role = new iam.Role(this, 'MyRole', {
     *      assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com')
     *    });
     *
     * @default - A role will automatically be created, it can be accessed via the `role` property
     */
    readonly role?: IRole;
    /**
     * AMI of the deadline worker to launch.
     */
    readonly workerMachineImage: IMachineImage;
    /**
     * Type of instance to launch for executing repository installer.
     *
     * @default - a T2-Large type will be used.
     */
    readonly instanceType?: InstanceType;
    /**
     * Where to place the instance within the VPC.
     *
     * @default - Private subnets.
     */
    readonly vpcSubnets?: SubnetSelection;
    /**
     * Name of SSH keypair to grant access to instance.
     *
     * @default - No SSH access will be possible.
     */
    readonly keyName?: string;
    /**
     * Initial amount of workers in the fleet.
     *
     * If this is set to a number, every deployment will reset the amount of
     * workers to this number. It is recommended to leave this value blank.
     *
     * @default minCapacity, and leave unchanged during deployment
     */
    readonly desiredCapacity?: number;
    /**
     * Minimum number of instances in the fleet.
     *
     * @default 1
     */
    readonly minCapacity?: number;
    /**
     * Maximum number of instances in the fleet.
     *
     * @default desiredCapacity, or minCapacity if desiredCapacity is not set
     */
    readonly maxCapacity?: number;
    /**
     * Endpoint for the RenderQueue, to which the worker fleet needs to be connected.
     */
    readonly renderQueue: IRenderQueue;
    /**
     * Deadline groups these workers needs to be assigned to. The group is
     * created if it does not already exist.
     *
     * @default - Worker is not assigned to any group
     */
    readonly groups?: string[];
    /**
     * Deadline pools these workers needs to be assigned to. The pool is created
     * if it does not already exist.
     *
     * @default - Worker is not assigned to any pool.
     */
    readonly pools?: string[];
    /**
     * Deadline region these workers needs to be assigned to.
     *
     * @default - Worker is not assigned to any region
     */
    readonly region?: string;
    /**
     * Properties for setting up the Deadline Worker's LogGroup
     * @default - LogGroup will be created with all properties' default values and a prefix of "/renderfarm/".
     */
    readonly logGroupProps?: LogGroupFactoryProps;
    /**
     * Health Monitor component to monitor the health of instances.
     *
     * @default - Health Monitoring is turned-off
     */
    readonly healthMonitor?: IHealthMonitor;
    /**
     * Properties for configuring a health check
     *
     * @default properties of HealthCheckConfig applies
     */
    readonly healthCheckConfig?: HealthCheckConfig;
    /**
     * The maximum hourly price($) to be paid for each Spot instance.
     * min - 0.001; max - 255
     *
     * @default - launches on-demand EC2 instances.
     */
    readonly spotPrice?: number;
    readonly blockDevices?: BlockDevice[];
}
/**
 *  A new or imported Deadline Worker Fleet.
 */
declare abstract class WorkerInstanceFleetBase extends Construct implements IWorkerFleet, IMonitorableFleet {
    /**
     * The security groups/rules used to allow network connections to the file system.
     */
    abstract readonly connections: Connections;
    /**
     * The principal to grant permissions to.
     */
    abstract readonly grantPrincipal: IPrincipal;
    /**
     * The stack in which this worker fleet is defined.
     */
    abstract readonly stack: Stack;
    /**
     * The ASG object created by the construct.
     */
    abstract readonly fleet: AutoScalingGroup;
    /**
     * This field expects the base capacity metric of the fleet against
     * which, the healthy percent will be calculated.
     *
     * eg.: GroupDesiredCapacity for an ASG
     */
    abstract readonly targetCapacityMetric: IMetric;
    /**
     * This field expects the component of type INetworkLoadBalancerTarget
     * which can be attached to Network Load Balancer for monitoring.
     *
     * eg. An AutoScalingGroup
     */
    abstract readonly targetToMonitor: IApplicationLoadBalancerTarget;
    /**
     * This field expects a policy which can be attached to the lambda
     * execution role so that it is capable of suspending the fleet.
     *
     * eg.: autoscaling:UpdateAutoScalingGroup permission for an ASG
     */
    abstract readonly targetUpdatePolicy: IPolicy;
    /**
     * This field expects the maximum instance count this fleet can have.
     */
    abstract readonly targetCapacity: number;
    /**
     * This field expects the scope in which to create the monitoring resource
     * like TargetGroups, Listener etc.
     */
    abstract readonly targetScope: Construct;
}
/**
 * This construct reperesents a fleet of Deadline Workers.
 *
 * The construct consists of an Auto Scaling Group (ASG) of instances using a provided AMI which has Deadline and any number
 * of render applications installed.  Whenever an instance in the ASG start it will connect Deadline to the desired render queue.
 *
 * When the worker fleet is deployed if it has been provided a HealthMonitor the Worker fleet will register itself against the Monitor
 * to ensure that the fleet remains healthy.
 *
 * Resources Deployed
 * ------------------------
 * 1) An AutoScalingGroup to maintain the number of instances;
 * 2) An Instance Role and corresponding IAM Policy;
 * 3) A script asset which is uploaded to your deployment bucket used to configure the worker so it can connect to the Render Queue
 * 4) An aws-rfdk.CloudWatchAgent to configure sending logs to cloudwatch.
 *
 * @ResourcesDeployed
 */
export declare class WorkerInstanceFleet extends WorkerInstanceFleetBase {
    /**
     * The min limit for spot price.
     */
    static readonly SPOT_PRICE_MIN_LIMIT = 0.001;
    /**
     * The max limit for spot price.
     */
    static readonly SPOT_PRICE_MAX_LIMIT = 255;
    /**
     * This determines worker's health based on any hardware or software issues of EC2 instance.
     * Resource Tracker does deep ping every 5 minutes. These checks should be more frequent so
     * that any EC2 level issues are identified ASAP. Hence setting it to 1 minute.
     */
    private static DEFAULT_HEALTH_CHECK_INTERVAL;
    /**
     * Default prefix for a LogGroup if one isn't provided in the props.
     */
    private static readonly DEFAULT_LOG_GROUP_PREFIX;
    /**
     * Setting the default signal timeout to 15 min. This is the max time, a single instance is expected
     * to take for launch and execute the user-data for deadline worker configuration. As we are setting
     * failure signals in the user-data, any failure will terminate deployment immediately.
     */
    private static readonly RESOURCE_SIGNAL_TIMEOUT;
    /**
     * The ASG object created by the construct.
     */
    readonly fleet: AutoScalingGroup;
    /**
     * The security groups/rules used to allow network connections to the file system.
     */
    readonly connections: Connections;
    /**
     * The principal to grant permissions to.
     */
    readonly grantPrincipal: IPrincipal;
    /**
     * The stack in which this worker fleet is defined.
     */
    readonly stack: Stack;
    /**
     * This field implements the base capacity metric of the fleet against
     * which, the healthy percent will be calculated.
     *
     * eg.: GroupDesiredCapacity for an ASG
     */
    readonly targetCapacityMetric: IMetric;
    /**
     * This field implements the component of type INetworkLoadBalancerTarget
     * which can be attached to Network Load Balancer for monitoring.
     *
     * eg. An AutoScalingGroup
     */
    readonly targetToMonitor: IApplicationLoadBalancerTarget;
    /**
     * This field implements a policy which can be attached to the lambda
     * execution role so that it is capable of suspending the fleet.
     *
     * eg.: autoscaling:UpdateAutoScalingGroup permission for an ASG
     */
    readonly targetUpdatePolicy: IPolicy;
    /**
     * This field implements the maximum instance count this fleet can have.
     */
    readonly targetCapacity: number;
    /**
     * This field implements the scope in which to create the monitoring resource
     * like TargetGroups, Listener etc.
     */
    readonly targetScope: Construct;
    constructor(scope: Construct, id: string, props: WorkerInstanceFleetProps);
    /**
     * Add the security group to all workers
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup: ISecurityGroup): void;
    private configureCloudWatchLogStream;
    private configureWorkerScript;
    private validateProps;
    private validateSpotPrice;
    private validateRegion;
    private validateArrayGroupsPoolsSyntax;
    private validateBlockDevices;
    private configureHealthMonitor;
}
export {};
