"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionQuery = void 0;
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
class VersionQueryBase extends core_1.Construct {
    /**
     * @inheritdoc
     */
    linuxFullVersionString() {
        if (!this.linuxInstallers) {
            return undefined;
        }
        return this.fullVersionString(this.linuxInstallers.patchVersion);
    }
    fullVersionString(patchVersion) {
        return `${this.releaseVersionString}.${patchVersion}`;
    }
    /**
     * Helper to concatenate the major, minor, and release version values into a release version string.
     */
    get releaseVersionString() {
        return `${this.majorVersion}.${this.minorVersion}.${this.releaseVersion}`;
    }
}
VersionQueryBase.INSTALLER_BUCKET = 'thinkbox-installers';
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
class VersionQuery extends VersionQueryBase {
    constructor(scope, id, props) {
        super(scope, id);
        throw new Error(`MethodNotSupportedException: This method is currently not implemented. Input: ${JSON.stringify(props)}`);
    }
    /**
     * Parses a version string of the format:
     *
     *    <major>.<minor>.<release>.<patch>
     *
     * and extracts the components.
     *
     * @param versionstr The input version string
     */
    static parseVersionString(versionstr) {
        const match = VersionQuery.RE_FULL_VERSION.exec(versionstr);
        if (!match) {
            throw new Error(`"${versionstr}" is not a valid version`);
        }
        return {
            majorVersion: parseInt(match.groups.major, 10),
            minorVersion: parseInt(match.groups.minor, 10),
            releaseVersion: parseInt(match.groups.release, 10),
            patchVersion: parseInt(match.groups.patch, 10),
        };
    }
    /**
     * Specify a Deadline version from a fully-qualified Deadline patch version.
     *
     * This only provides the Linux repository installer based on its conventional
     * S3 object path.
     *
     * @remark Thinkbox reserves the right to revoke patch versions of Deadline and suceeed them with a new patch version.
     * For this reason, using this method may fail if Thinkbox revokes the specific patch version of the Deadline
     * installer in the event of a critical issue such as a security vulnerability.
     *
     * Use at your own risk.
     *
     * @param scope The parent scope
     * @param id The construct ID
     * @param versionComponents The individual components of the Deadline release version
     */
    static exact(scope, id, versionComponents) {
        class ExactVersion extends VersionQueryBase {
            constructor() {
                super(scope, id);
                const installerBucket = aws_s3_1.Bucket.fromBucketName(this, 'ThinkboxInstallers', ExactVersion.INSTALLER_BUCKET);
                const { majorVersion, minorVersion, releaseVersion, patchVersion } = versionComponents;
                this.majorVersion = majorVersion;
                this.minorVersion = minorVersion;
                this.releaseVersion = releaseVersion;
                const fullVersionString = this.fullVersionString(patchVersion);
                const objectKey = `Deadline/${fullVersionString}/Linux/DeadlineRepository-${fullVersionString}-linux-x64-installer.run`;
                this.linuxInstallers = {
                    patchVersion,
                    repository: {
                        s3Bucket: installerBucket,
                        objectKey,
                    },
                };
            }
        }
        return new ExactVersion();
    }
    /**
     * Specify Version from a fully-qualified Deadline release version string.
     *
     * This only provides the Linux repository installer based on its conventional
     * S3 object path.
     *
     * @param scope The parent scope
     * @param id The construct ID
     * @param versionString A fully qualified version string (e.g. 10.1.9.2)
     */
    static exactString(scope, id, versionString) {
        return VersionQuery.exact(scope, id, VersionQuery.parseVersionString(versionString));
    }
}
exports.VersionQuery = VersionQuery;
/**
 * Regular expression for matching a Deadline release version number
 */
VersionQuery.RE_FULL_VERSION = /^(?<major>\d+)\.(?<minor>\d+)\.(?<release>\d+)\.(?<patch>\d+)$/;
//# sourceMappingURL=data:application/json;base64,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