/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import { Construct } from '@aws-cdk/core';
import { IPatchVersion, IVersion, PlatformInstallers } from './version-ref';
/**
 * Properties for the Deadline Version
 */
export interface VersionQueryProps {
    /**
     * String containing the complete or partial deadline version.
     *
     * @default - the latest available version of deadline installer.
     */
    readonly version?: string;
}
/**
 * The abstract class for new or imported(custom) Deadline Version.
 */
declare abstract class VersionQueryBase extends Construct implements IVersion {
    protected static readonly INSTALLER_BUCKET = "thinkbox-installers";
    /**
     * @inheritdoc
     */
    abstract readonly majorVersion: number;
    /**
     * @inheritdoc
     */
    abstract readonly minorVersion: number;
    /**
     * @inheritdoc
     */
    abstract readonly releaseVersion: number;
    /**
     * @inheritdoc
     */
    abstract readonly linuxInstallers?: PlatformInstallers;
    /**
     * @inheritdoc
     */
    linuxFullVersionString(): string | undefined;
    protected fullVersionString(patchVersion: number): string;
    /**
     * Helper to concatenate the major, minor, and release version values into a release version string.
     */
    protected get releaseVersionString(): string;
}
/**
 * This class encapsulates information about a particular version of Thinkbox's Deadline software.
 * Information such as the version number, and where to get installers for that version from Amazon S3.
 *
 * The version of an official release of Deadline is always four numeric version components separated by dots.
 * ex: 10.1.8.5. We refer to the components in this version, in order from left-to-right, as the
 * major, minor, release, and patch versions. For example, Deadline version 10.1.8.5 is majorVersion 10, minorVersion 1,
 * releaseVersion 8, and patchVersion 5.
 *
 * All of the installers provided by an instance of this class must be for the same Deadline release (ex: 10.1.8),
 * but the patch versions may differ between operating systems depending on the particulars of that release of Deadline.
 * This class provides a simple way to query a version of Deadline prior to or during deployment of a
 * CDK app.
 *
 * You pass an instance of this class to various Deadline constructs in this library to tell those
 * constructs which version of Deadline you want them to use, and be configured for.
 */
export declare class VersionQuery extends VersionQueryBase {
    /**
     * Parses a version string of the format:
     *
     *    <major>.<minor>.<release>.<patch>
     *
     * and extracts the components.
     *
     * @param versionstr The input version string
     */
    static parseVersionString(versionstr: string): IPatchVersion;
    /**
     * Specify a Deadline version from a fully-qualified Deadline patch version.
     *
     * This only provides the Linux repository installer based on its conventional
     * S3 object path.
     *
     * @remark Thinkbox reserves the right to revoke patch versions of Deadline and suceeed them with a new patch version.
     * For this reason, using this method may fail if Thinkbox revokes the specific patch version of the Deadline
     * installer in the event of a critical issue such as a security vulnerability.
     *
     * Use at your own risk.
     *
     * @param scope The parent scope
     * @param id The construct ID
     * @param versionComponents The individual components of the Deadline release version
     */
    static exact(scope: Construct, id: string, versionComponents: IPatchVersion): IVersion;
    /**
     * Specify Version from a fully-qualified Deadline release version string.
     *
     * This only provides the Linux repository installer based on its conventional
     * S3 object path.
     *
     * @param scope The parent scope
     * @param id The construct ID
     * @param versionString A fully qualified version string (e.g. 10.1.9.2)
     */
    static exactString(scope: Construct, id: string, versionString: string): IVersion;
    /**
     * Regular expression for matching a Deadline release version number
     */
    private static readonly RE_FULL_VERSION;
    /**
     * @inheritdoc
     */
    readonly majorVersion: number;
    /**
     * @inheritdoc
     */
    readonly minorVersion: number;
    /**
     * @inheritdoc
     */
    readonly releaseVersion: number;
    /**
     * @inheritdoc
     */
    readonly linuxInstallers?: PlatformInstallers;
    constructor(scope: Construct, id: string, props?: VersionQueryProps);
}
export {};
