"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ThinkboxDockerRecipes = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const core_1 = require("@aws-cdk/core");
/**
 * An enum that is associated with AWS Thinkbox managed recipes that are available in the stage manifest.
 */
var ThinkboxManagedDeadlineDockerRecipes;
(function (ThinkboxManagedDeadlineDockerRecipes) {
    /**
     * The Docker Image Asset for the Remote Connection Server.
     */
    ThinkboxManagedDeadlineDockerRecipes["RemoteConnectionServer"] = "rcs";
    /**
     * The Docker Image Asset for the License Forwarder.
     */
    ThinkboxManagedDeadlineDockerRecipes["LicenseForwarder"] = "license-forwarder";
})(ThinkboxManagedDeadlineDockerRecipes || (ThinkboxManagedDeadlineDockerRecipes = {}));
/**
 * An API for interacting with staged Deadline Docker recipes provided by AWS Thinkbox.
 *
 * This provides container images as required by RFDK's Deadline constructs such as
 *
 * * {@link @aws-rfdk/deadline#RenderQueue}
 * * {@link @aws-rfdk/deadline#UsageBasedLicensing}
 *
 * @example Construct a RenderQueue
 *
 * import { App, Stack, Vpc } from '@aws-rfdk/core';
 * import { RenderQueue, Repository, ThinkboxDockerRecipes } from '@aws-rfdk/deadline';
 * const app = new App();
 * const stack = new Stack(app, 'Stack');
 * const vpc = new Vpc(app, stack);
 * const recipes = new ThinkboxDockerRecipes(stack, 'Recipes', {
 *   path: '/path/to/staged/recipes'
 * });
 * const repository = new Repository(stack, 'Repository', {
 *   vpc,
 *   version: recipes.version
 * });
 *
 * const renderQueue = new RenderQueue(stack, 'RenderQueue', {
 *   images: recipes.renderQueueImages,
 *   // ...
 * });
 */
class ThinkboxDockerRecipes extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.version = props.stage.getVersion(this, 'Version');
        for (const recipe of [ThinkboxManagedDeadlineDockerRecipes.RemoteConnectionServer, ThinkboxManagedDeadlineDockerRecipes.LicenseForwarder]) {
            if (!props.stage.manifest.recipes[recipe]) {
                throw new Error(`Could not find ${recipe} recipe`);
            }
        }
        this.remoteConnectionServer = props.stage.imageFromRecipe(this, 'RemoteConnectionServer', ThinkboxManagedDeadlineDockerRecipes.RemoteConnectionServer.toString());
        this.licenseForwarder = props.stage.imageFromRecipe(this, 'LicenseForwarder', ThinkboxManagedDeadlineDockerRecipes.LicenseForwarder.toString());
        this.renderQueueImages = {
            remoteConnectionServer: aws_ecs_1.ContainerImage.fromDockerImageAsset(this.remoteConnectionServer),
        };
        this.ublImages = {
            licenseForwarder: aws_ecs_1.ContainerImage.fromDockerImageAsset(this.licenseForwarder),
        };
    }
}
exports.ThinkboxDockerRecipes = ThinkboxDockerRecipes;
//# sourceMappingURL=data:application/json;base64,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