"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Stage = void 0;
const fs = require("fs");
const path = require("path");
const aws_ecr_assets_1 = require("@aws-cdk/aws-ecr-assets");
const version_1 = require("./version");
/**
 * Class for interacting with the Deadline stage directory
 *
 * The stage is a directory that conforms to a conventional structure that RFDK
 * requires to deploy Deadline. It should contain a manifest file, the Deadline
 * installers, and any supporting files required for building the Deadline
 * container.
 */
class Stage {
    /**
     * Constructs a Stage instance
     *
     * @param dirPath The path to the Deadline staging directory
     * @param manifest The parsed manifest file
     */
    constructor(props) {
        Stage.validateManifest(props.manifest);
        this.dirPath = props.path;
        this.manifest = props.manifest;
    }
    /**
     * Loads and parses the manifest file from a given path
     * @param manifestPath The path to the manifest JSON file
     */
    static loadManifest(manifestPath) {
        const content = fs.readFileSync(manifestPath, { encoding: 'utf8' });
        return JSON.parse(content);
    }
    /**
     * Returns a {@link Stage} loaded using the specified directory as the Docker build context
     * and loads and uses the manifest named `manifest.json` in the directory.
     *
     * @param stagePath The path to the Deadline stage directory
     */
    static fromDirectory(stagePath) {
        const manifest = Stage.loadManifest(path.join(stagePath, Stage.MANIFEST_REL_PATH));
        return new Stage({
            manifest,
            path: stagePath,
        });
    }
    /**
     * Ensures that the manifest is a valid manifest object schema.
     *
     * If the object does not fit the schema, then an Error is thrown.
     *
     * @param rawManifest The parsed manifest object
     */
    static validateManifest(rawManifest) {
        if (rawManifest !== undefined && typeof rawManifest !== 'object') {
            throw new TypeError(`Expected object but got ${typeof rawManifest}`);
        }
        const schema = rawManifest.schema;
        if (schema === undefined) {
            throw new Error('Manifest contains no "schema" key');
        }
        else if (typeof schema !== 'number') {
            throw new TypeError(`Expected a numeric "schema" but got: ${typeof schema}`);
        }
        const version = rawManifest.version;
        if (version === undefined) {
            throw new Error('Manifest contains no "version" key');
        }
        else if (typeof version !== 'string') {
            throw new TypeError(`Expected a string "version" but got: ${typeof version}`);
        }
        return true;
    }
    /**
     * Creates a {@link Version} based on the manifest version
     *
     * @param scope The parent scope
     * @param id The construct ID
     */
    getVersion(scope, id) {
        return version_1.VersionQuery.exactString(scope, id, this.manifest.version);
    }
    /**
     * Construct a {@link DockerImageAsset} instance from a recipe in the Stage
     * @param scope The scope for the {@link DockerImageAsset}
     * @param id The construct ID of the {@link DockerImageAsset}
     * @param recipeName The name of the recipe
     */
    imageFromRecipe(scope, id, recipeName) {
        const recipe = this.manifest.recipes[recipeName];
        if (!recipe) {
            throw new Error(`No such recipe: ${recipeName}`);
        }
        return new aws_ecr_assets_1.DockerImageAsset(scope, id, {
            directory: this.dirPath,
            ...recipe,
        });
    }
}
exports.Stage = Stage;
/**
 * The path to the manifest JSON file relative to the {@link Stage}'s directory'.
 */
Stage.MANIFEST_REL_PATH = 'manifest.json';
//# sourceMappingURL=data:application/json;base64,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