"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const path = require("path");
const url_1 = require("url");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_docdb_1 = require("@aws-cdk/aws-docdb");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_efs_1 = require("@aws-cdk/aws-efs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const database_connection_1 = require("./database-connection");
/**
 * This construct represents the main Deadline Repository which contains the central database and file system
 * that Deadline requires.
 *
 * When deployed this construct will start up a single instance which will run the Deadline Repository installer to
 * initialize the file system and database, the logs of which will be forwarded to Cloudwatch via a CloudWatchAgent.
 * After the installation is complete the instance will be shutdown.
 *
 * Whenever the stack is updated if a change is detected in the installer a new instance will be started, which will perform
 * a check on the existing Deadline Repository.  If they are compatible with the new installer an update will be performed
 * and the deployment will continue, otherwise the the deployment will be cancelled.
 * In either case the instance will be cleaned up.
 *
 * Resources Deployed
 * ------------------------
 * 1) Encrypted EFS File System - If no IFileSystem is provided;
 * 2) DocumentDB and DatabaseConnection - If no database connection is provided;
 * 3) Auto Scaling Group (ASG) with min & max capacity of 1 instance;
 * 4) Instance Role and corresponding IAM Policy
 * 5) A Script Asset which is uploaded to your deployment bucket to run the installer
 * 6) An aws-rfdk.CloudWatchAgent to configure sending logs to cloudwatch.
 *
 * @ResourcesDeployed
 */
class Repository extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m, _o, _p;
        super(scope, id);
        if (props.database && ((_a = props.backupOptions) === null || _a === void 0 ? void 0 : _a.databaseRetention)) {
            this.node.addWarning('Backup retention for database will not be applied since a database is not being created by this construct');
        }
        if (props.fileSystem && ((_b = props.removalPolicy) === null || _b === void 0 ? void 0 : _b.filesystem)) {
            this.node.addWarning('RemovalPolicy for filesystem will not be applied since a filesystem is not being created by this construct');
        }
        if (props.database && ((_c = props.removalPolicy) === null || _c === void 0 ? void 0 : _c.database)) {
            this.node.addWarning('RemovalPolicy for database will not be applied since a database is not being created by this construct');
        }
        this.version = props.version;
        // Set up the Filesystem and Database components of the repository
        this.fileSystem = (_d = props.fileSystem) !== null && _d !== void 0 ? _d : new core_2.MountableEfs(this, {
            filesystem: new aws_efs_1.FileSystem(this, 'FileSystem', {
                vpc: props.vpc,
                vpcSubnets: (_e = props.vpcSubnets) !== null && _e !== void 0 ? _e : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
                encrypted: true,
                lifecyclePolicy: aws_efs_1.LifecyclePolicy.AFTER_14_DAYS,
                removalPolicy: (_g = (_f = props.removalPolicy) === null || _f === void 0 ? void 0 : _f.filesystem) !== null && _g !== void 0 ? _g : core_1.RemovalPolicy.RETAIN,
            }),
        });
        if (props.database) {
            this.databaseConnection = props.database;
            if (props.databaseAuditLogging !== undefined) {
                this.node.addWarning(`The parameter databaseAuditLogging only has an effect when the Repository is creating its own database. 
        Please ensure that the Database provided is configured correctly.`);
            }
        }
        else {
            const databaseAuditLogging = (_h = props.databaseAuditLogging) !== null && _h !== void 0 ? _h : true;
            /**
             * This option is part of enabling audit logging for DocumentDB; the other required part is the enabling of the CloudWatch exports below.
             *
             * For more information about audit logging in DocumentDB, see:  https://docs.aws.amazon.com/documentdb/latest/developerguide/event-auditing.html
             */
            const parameterGroup = databaseAuditLogging ? new aws_docdb_1.ClusterParameterGroup(this, 'ParameterGroup', {
                description: 'DocDB cluster parameter group with enabled audit logs',
                family: 'docdb3.6',
                parameters: {
                    audit_logs: 'enabled',
                },
            }) : undefined;
            const instances = (_j = props.documentDbInstanceCount) !== null && _j !== void 0 ? _j : Repository.DEFAULT_NUM_DOCDB_INSTANCES;
            const dbCluster = new aws_docdb_1.DatabaseCluster(this, 'DocumentDatabase', {
                masterUser: { username: 'DocDBUser' },
                instanceProps: {
                    instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.R5, aws_ec2_1.InstanceSize.LARGE),
                    vpc: props.vpc,
                    vpcSubnets: (_k = props.vpcSubnets) !== null && _k !== void 0 ? _k : { subnetType: aws_ec2_1.SubnetType.PRIVATE, onePerAz: true },
                },
                instances,
                backup: {
                    retention: (_m = (_l = props.backupOptions) === null || _l === void 0 ? void 0 : _l.databaseRetention) !== null && _m !== void 0 ? _m : Repository.DEFAULT_DATABASE_RETENTION_PERIOD,
                },
                parameterGroup,
                removalPolicy: (_p = (_o = props.removalPolicy) === null || _o === void 0 ? void 0 : _o.database) !== null && _p !== void 0 ? _p : core_1.RemovalPolicy.RETAIN,
            });
            if (databaseAuditLogging) {
                /**
                 * This option enable export audit logs to Amazon CloudWatch.
                 * This is second options that required for enable audit log.
                 */
                const cfnDB = dbCluster.node.findChild('Resource');
                cfnDB.enableCloudwatchLogsExports = ['audit'];
            }
            /* istanbul ignore next */
            if (!dbCluster.secret) {
                /* istanbul ignore next */
                throw new Error('DBCluster failed to get set up properly -- missing login secret.');
            }
            // This is a workaround because of the bug in CDK implementation:
            // autoMinorVersionUpgrade should be true by default but it's not.
            // This code can be removed once fixed in CDK.
            for (let i = 1; i <= instances; i++) {
                const docdbInstance = dbCluster.node.tryFindChild(`Instance${i}`);
                docdbInstance.autoMinorVersionUpgrade = true;
            }
            this.databaseConnection = database_connection_1.DatabaseConnection.forDocDB({
                database: dbCluster,
                login: dbCluster.secret,
            });
        }
        // Launching the instance which installs the deadline repository in the stack.
        this.installerGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Installer', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            vpcSubnets: {
                subnetType: aws_ec2_1.SubnetType.PRIVATE,
            },
            minCapacity: 1,
            maxCapacity: 1,
            resourceSignalTimeout: (props.repositoryInstallationTimeout || core_1.Duration.minutes(15)),
            updateType: aws_autoscaling_1.UpdateType.REPLACING_UPDATE,
            replacingUpdateMinSuccessfulInstancesPercent: 100,
        });
        this.node.defaultChild = this.installerGroup;
        // Ensure the DB is serving before we try to connect to it.
        this.databaseConnection.addChildDependency(this.installerGroup);
        // Updating the user data with installation logs stream -- ALWAYS DO THIS FIRST.
        this.configureCloudWatchLogStream(this.installerGroup, `${id}`, props.logGroupProps);
        this.setupDirectConnect(this.installerGroup, Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH);
        this.rootPrefix = props.repositoryInstallationPrefix || Repository.DEFAULT_REPO_PREFIX;
        if (path.posix.isAbsolute(this.rootPrefix)) {
            // If the input path is absolute, then we make it relative (to the root of the repo file-system)
            this.rootPrefix = path.posix.relative(path.posix.sep, this.rootPrefix);
        }
        const repositoryInstallationPath = path.posix.normalize(path.posix.join(Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH, this.rootPrefix));
        // Updating the user data with deadline repository installation commands.
        this.configureRepositoryInstallerScript(this.installerGroup, repositoryInstallationPath, props.version);
        this.configureSelfTermination();
        // Updating the user data with successful cfn-signal commands.
        this.installerGroup.userData.addSignalOnExitCommand(this.installerGroup);
    }
    /**
     * @inheritdoc
     */
    configureClientECS(props) {
        var _a, _b;
        const hostMountPoint = (_a = props.containerInstances.filesystemMountPoint) !== null && _a !== void 0 ? _a : '/mnt/repo';
        const containerMountPoint = (_b = props.containers.filesystemMountPoint) !== null && _b !== void 0 ? _b : `/opt/Thinkbox/DeadlineRepository${this.version.majorVersion}`;
        // Set up a direct connection on the host machine. This:
        //  - grants IAM permissions to the role associated with the instance profile access to
        //    - the file-system
        //    - the DB secret containing the credentials
        //  - adds a security group ingress rule to the DB cluster and file-system
        //  - adds userdata commands to mount the repository file-system on the host
        props.containerInstances.hosts.forEach(host => {
            this.setupDirectConnect(host, hostMountPoint);
        });
        // Build up a mapping of environment variables that are used to configure the container's direct connection to the
        // repository
        const containerEnvironment = {
            REPO_URI: url_1.pathToFileURL(containerMountPoint).toString(),
        };
        // The role associated with the task definition needs access to connect to the database
        this.databaseConnection.grantRead(props.containers.taskDefinition.taskRole);
        // Add any environment variables specified by the connection
        Object.entries(this.databaseConnection.containerEnvironment).forEach((entry) => {
            const [envVarName, envVarValue] = entry;
            containerEnvironment[envVarName] = envVarValue;
        });
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.containers.taskDefinition.node.addDependency(this);
        // Configure a named volume in the task-definition that points to the container host's mount-point of the repository
        // file-system
        props.containers.taskDefinition.addVolume({
            name: Repository.ECS_VOLUME_NAME,
            host: {
                sourcePath: path.posix.normalize(path.posix.join(hostMountPoint, this.rootPrefix)),
            },
        });
        // Return the container connection. This data structure contains all the pieces needed to create containers
        // that can directly connect to the repository.
        return {
            containerEnvironment,
            readOnlyMountPoint: {
                containerPath: containerMountPoint,
                readOnly: true,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
            readWriteMountPoint: {
                containerPath: containerMountPoint,
                readOnly: false,
                sourceVolume: Repository.ECS_VOLUME_NAME,
            },
        };
    }
    /**
     * @inheritdoc
     */
    configureClientInstance(props) {
        var _a;
        // Add an explicit dependency on the Repository. This ensures that deployments of the Repository construct precede
        // deployments of the client and the repository is fully setup.
        props.host.node.addDependency(this);
        this.setupDirectConnect(props.host, props.mountPoint);
        const stack = core_1.Stack.of(this);
        const uuid = 'f625e47b-7aed-4879-9861-513a72145525';
        const uniqueId = 'DeadlineRepository' + props.host.osType + uuid.replace(/[-]/g, '');
        const configureDirectConnect = (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : core_2.ScriptAsset.fromPathConvention(stack, uniqueId, {
            osType: props.host.osType,
            baseName: 'configureRepositoryDirectConnect',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        configureDirectConnect.grantRead(props.host);
        this.databaseConnection.addConnectionDBArgs(props.host);
        const repoPath = path.posix.normalize(path.posix.join(props.mountPoint, this.rootPrefix));
        configureDirectConnect.executeOn({
            host: props.host,
            args: [`"${repoPath}"`],
        });
    }
    /**
     * Set up direct connect to this repo for the given host. Specifically:
     *  - IAM permissions & security group access to the database.
     *  - mounting the repository filesystem
     *
     * @param host Host to setup.
     * @param repositoryMountPoint Absolute directory at which to mount the repo filesystem.
     *
     * @remark Only allowable for Windows hosts.
     */
    setupDirectConnect(host, repositoryMountPoint) {
        if (host.osType === aws_ec2_1.OperatingSystemType.WINDOWS) {
            throw new Error('Deadline direct connect on Windows hosts is not yet supported by the RFDK.');
        }
        this.databaseConnection.grantRead(host);
        this.databaseConnection.allowConnectionsFrom(host);
        this.fileSystem.mountToLinuxInstance(host, {
            location: repositoryMountPoint,
        });
    }
    /**
     * Adds UserData commands to configure the CloudWatch Agent running on the instance that performs the repository
     * installation.
     *
     * The commands configure the agent to stream the following logs to a new CloudWatch log group:
     *   - The cloud-init log
     *   - The Deadline Repo's installer log
     *
     * @param installerGroup The instance that performs the Deadline Repository installation
     * @param logGroupProps
     */
    configureCloudWatchLogStream(installerGroup, groupName, logGroupProps) {
        const prefix = (logGroupProps === null || logGroupProps === void 0 ? void 0 : logGroupProps.logGroupPrefix) ? logGroupProps.logGroupPrefix : Repository.DEFAULT_LOG_GROUP_PREFIX;
        const defaultedLogGroupProps = {
            ...logGroupProps,
            logGroupPrefix: prefix,
        };
        const logGroup = core_2.LogGroupFactory.createOrFetch(this, 'RepositoryLogGroupWrapper', groupName, defaultedLogGroupProps);
        logGroup.grantWrite(installerGroup);
        const cloudWatchConfigurationBuilder = new core_2.CloudWatchConfigBuilder(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL);
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'cloud-init-output', '/var/log/cloud-init-output.log');
        cloudWatchConfigurationBuilder.addLogsCollectList(logGroup.logGroupName, 'deadlineRepositoryInstallationLogs', '/tmp/bitrock_installer.log');
        new core_2.CloudWatchAgent(this, 'RepositoryInstallerLogsConfig', {
            cloudWatchConfig: cloudWatchConfigurationBuilder.generateCloudWatchConfiguration(),
            host: installerGroup,
        });
    }
    configureSelfTermination() {
        const tagKey = 'resourceLogicalId';
        /*
        Add a policy to the ASG that allows it to modify itself. We cannot add the ASG name in resources
        as it will cause cyclic dependency. Hence, using Condition Keys
        */
        const tagCondition = {};
        tagCondition[`autoscaling:ResourceTag/${tagKey}`] = this.node.uniqueId;
        core_1.Tag.add(this.installerGroup, tagKey, this.node.uniqueId);
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'autoscaling:UpdateAutoScalingGroup',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
        // Following policy is required to read the aws tags within the instance
        this.installerGroup.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'ec2:DescribeTags',
            ],
            resources: ['*'],
        }));
        // wait for the log flush interval to make sure that all the logs gets flushed.
        // this wait can be avoided in future by using a life-cycle-hook on 'TERMINATING' state.
        const terminationDelay = Math.ceil(Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL.toMinutes({ integral: false }));
        this.installerGroup.userData.addOnExitCommands(`sleep ${terminationDelay}m`);
        // fetching the instance id and asg name and then setting all the capacity to 0 to terminate the installer.
        this.installerGroup.userData.addOnExitCommands('INSTANCE="$(curl http://169.254.169.254/latest/meta-data/instance-id)"');
        this.installerGroup.userData.addOnExitCommands('ASG="$(aws --region ' + core_1.Stack.of(this).region + ' ec2 describe-tags --filters "Name=resource-id,Values=${INSTANCE}" "Name=key,Values=aws:autoscaling:groupName" --query "Tags[0].Value" --output text)"');
        this.installerGroup.userData.addOnExitCommands('aws --region ' + core_1.Stack.of(this).region + ' autoscaling update-auto-scaling-group --auto-scaling-group-name ${ASG} --min-size 0 --max-size 0 --desired-capacity 0');
    }
    configureRepositoryInstallerScript(installerGroup, installPath, version) {
        var _a;
        const installerScriptAsset = core_2.ScriptAsset.fromPathConvention(this, 'DeadlineRepositoryInstallerScript', {
            osType: installerGroup.osType,
            baseName: 'installDeadlineRepository',
            rootDir: path.join(__dirname, '..', 'scripts'),
        });
        this.databaseConnection.addInstallerDBArgs(installerGroup);
        if (!((_a = version.linuxInstallers) === null || _a === void 0 ? void 0 : _a.repository)) {
            throw new Error('Version given to Repository must provide a Linux Repository installer.');
        }
        const linuxVersionString = version.linuxFullVersionString();
        if (!linuxVersionString) {
            throw new Error('Version given to Repository must provide a full Linux version string.');
        }
        version.linuxInstallers.repository.s3Bucket.grantRead(installerGroup, version.linuxInstallers.repository.objectKey);
        installerScriptAsset.executeOn({
            host: installerGroup,
            args: [
                `"s3://${version.linuxInstallers.repository.s3Bucket.bucketName}/${version.linuxInstallers.repository.objectKey}"`,
                `"${installPath}"`,
                linuxVersionString,
            ],
        });
    }
}
exports.Repository = Repository;
/**
 * Default file system mount path for repository
 */
Repository.DEFAULT_FILE_SYSTEM_MOUNT_PATH = '/mnt/efs/fs1';
/**
 * Default installation prefix for deadline repository.
 */
Repository.DEFAULT_REPO_PREFIX = 'DeadlineRepository';
/**
 * Default prefix for a LogGroup if one isn't provided in the props.
 */
Repository.DEFAULT_LOG_GROUP_PREFIX = '/renderfarm/';
/**
 * How often Cloudwatch logs will be flushed.
 */
Repository.CLOUDWATCH_LOG_FLUSH_INTERVAL = core_1.Duration.seconds(15);
/**
 * The name of the volume used in ECS task definitions to mount the repository file-system mounted on EC2 hosts into
 * containers.
 */
Repository.ECS_VOLUME_NAME = 'RepositoryFilesystem';
/**
 * The default number of DocDB instances if one isn't provided in the props.
 */
Repository.DEFAULT_NUM_DOCDB_INSTANCES = 1;
/**
 * Default retention period for DocumentDB automated backups if one isn't provided in the props.
 */
Repository.DEFAULT_DATABASE_RETENTION_PERIOD = core_1.Duration.days(15);
//# sourceMappingURL=data:application/json;base64,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