"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.RenderQueue = void 0;
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_ecs_patterns_1 = require("@aws-cdk/aws-ecs-patterns");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const core_2 = require("../../core");
const rq_connection_1 = require("./rq-connection");
/**
 * Base class for Render Queue providers
 */
class RenderQueueBase extends core_1.Construct {
}
/**
 * The RenderQueue construct deploys an Elastic Container Service (ECS) service that serves Deadline's REST HTTP API
 * to Deadline Clients.
 *
 * Most Deadline clients will connect to a Deadline render farm via the the RenderQueue. The API provides Deadline
 * clients access to Deadline's database and repository file-system in a way that is secure, performant, and scalable.
 *
 * Resources Deployed
 * ------------------------
 * 1) An ECS cluster
 * 2) An EC2 auto-scaling group that provides the EC2 container instances that host the ECS service
 * 3) An ECS service with a task definition that deploys the RCS container
 * 4) A CloudWatch bucket for streaming logs from the RCS container
 * 5) An application load balancer, listener and target group that balance incoming traffic among the RCS containers
 *
 * @ResourcesDeployed
 */
class RenderQueue extends RenderQueueBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        super(scope, id);
        // The RCS does not currently support horizontal scaling behind a load-balancer, so we limit to at most one instance
        if (props.renderQueueSize && props.renderQueueSize.min !== undefined && props.renderQueueSize.min > 1) {
            throw new Error(`renderQueueSize.min cannot be greater than 1 - got ${props.renderQueueSize.min}`);
        }
        if (props.renderQueueSize && props.renderQueueSize.desired !== undefined && props.renderQueueSize.desired > 1) {
            throw new Error(`renderQueueSize.desired cannot be greater than 1 - got ${props.renderQueueSize.desired}`);
        }
        let externalProtocol;
        if ((_a = props.trafficEncryption) === null || _a === void 0 ? void 0 : _a.externalTLS) {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
            if ((props.trafficEncryption.externalTLS.acmCertificate === undefined) ===
                (props.trafficEncryption.externalTLS.rfdkCertificate === undefined)) {
                throw new Error('Exactly one of externalTLS.acmCertificate and externalTLS.rfdkCertificate must be provided when using externalTLS.');
            }
            else if (props.trafficEncryption.externalTLS.rfdkCertificate) {
                if (props.trafficEncryption.externalTLS.rfdkCertificate.certChain === undefined) {
                    throw new Error('Provided rfdkCertificate does not contain a certificate chain.');
                }
                this.clientCert = new core_2.ImportedAcmCertificate(this, 'AcmCert', props.trafficEncryption.externalTLS.rfdkCertificate);
                this.certChain = props.trafficEncryption.externalTLS.rfdkCertificate.certChain;
            }
            else {
                if (props.trafficEncryption.externalTLS.acmCertificateChain === undefined) {
                    throw new Error('externalTLS.acmCertificateChain must be provided when using externalTLS.acmCertificate.');
                }
                this.clientCert = props.trafficEncryption.externalTLS.acmCertificate;
                this.certChain = props.trafficEncryption.externalTLS.acmCertificateChain;
            }
        }
        else {
            externalProtocol = aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP;
        }
        const internalProtocol = (_c = (_b = props.trafficEncryption) === null || _b === void 0 ? void 0 : _b.internalProtocol) !== null && _c !== void 0 ? _c : aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS;
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS && !props.hostname) {
            throw new Error('A hostname must be provided when the external protocol is HTTPS');
        }
        this.cluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            vpc: props.vpc,
        });
        const minCapacity = (_e = (_d = props.renderQueueSize) === null || _d === void 0 ? void 0 : _d.min) !== null && _e !== void 0 ? _e : 1;
        if (minCapacity < 1) {
            throw new Error(`renderQueueSize.min capacity must be at least 1: got ${minCapacity}`);
        }
        this.asg = this.cluster.addCapacity('RCS Capacity', {
            vpcSubnets: (_f = props.vpcSubnets) !== null && _f !== void 0 ? _f : { subnetType: aws_ec2_1.SubnetType.PRIVATE },
            instanceType: (_g = props.instanceType) !== null && _g !== void 0 ? _g : new aws_ec2_1.InstanceType('c5.large'),
            minCapacity,
            desiredCapacity: (_h = props.renderQueueSize) === null || _h === void 0 ? void 0 : _h.desired,
            maxCapacity: 1,
            blockDevices: [{
                    deviceName: '/dev/xvda',
                    // See: https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-ami-storage-config.html
                    // We want the volume to be encrypted. The default AMI size is 30-GiB.
                    volume: aws_autoscaling_1.BlockDeviceVolume.ebs(30, { encrypted: true }),
                }],
            updateType: aws_autoscaling_1.UpdateType.ROLLING_UPDATE,
        });
        /**
         * The ECS-optimized AMI that is defaulted to when adding capacity to a cluster does not include the awscli or unzip
         * packages as is the case with the standard Amazon Linux AMI. These are required by RFDK scripts to configure the
         * direct connection on the host container instances.
         */
        this.asg.userData.addCommands('yum install -yq awscli unzip');
        const externalPortNumber = RenderQueue.RCS_PROTO_PORTS[externalProtocol];
        const internalPortNumber = RenderQueue.RCS_PROTO_PORTS[internalProtocol];
        this.logGroup = core_2.LogGroupFactory.createOrFetch(this, 'LogGroupWrapper', id, {
            logGroupPrefix: '/renderfarm/',
            ...props.logGroupProps,
        });
        this.logGroup.grantWrite(this.asg);
        const taskDefinition = this.createTaskDefinition({
            image: props.images.remoteConnectionServer,
            portNumber: internalPortNumber,
            protocol: internalProtocol,
            repository: props.repository,
        });
        this.taskDefinition = taskDefinition;
        // The fully-qualified domain name to use for the ALB
        let loadBalancerFQDN;
        if (props.hostname) {
            const label = (_j = props.hostname.hostname) !== null && _j !== void 0 ? _j : 'renderqueue';
            if (props.hostname.hostname && !RenderQueue.RE_VALID_HOSTNAME.test(label)) {
                throw new Error(`Invalid RenderQueue hostname: ${label}`);
            }
            loadBalancerFQDN = `${label}.${props.hostname.zone.zoneName}`;
        }
        const loadBalancer = new aws_elasticloadbalancingv2_1.ApplicationLoadBalancer(this, 'LB', {
            vpc: this.cluster.vpc,
            internetFacing: false,
            deletionProtection: (_k = props.deletionProtection) !== null && _k !== void 0 ? _k : true,
        });
        this.pattern = new aws_ecs_patterns_1.ApplicationLoadBalancedEc2Service(this, 'AlbEc2ServicePattern', {
            certificate: this.clientCert,
            cluster: this.cluster,
            desiredCount: (_l = props.renderQueueSize) === null || _l === void 0 ? void 0 : _l.desired,
            domainZone: (_m = props.hostname) === null || _m === void 0 ? void 0 : _m.zone,
            domainName: loadBalancerFQDN,
            listenerPort: externalPortNumber,
            loadBalancer,
            protocol: externalProtocol,
            taskDefinition,
            // This is required to right-size our host capacity and not have the ECS service block on updates. We set a memory
            // reservation, but no memory limit on the container. This allows the container's memory usage to grow unbounded.
            // We want 1:1 container to container instances to not over-spend, but this comes at the price of down-time during
            // cloudformation updates.
            minHealthyPercent: 0,
            maxHealthyPercent: 100,
            // This is required to ensure that the ALB listener's security group does not allow any ingress by default.
            openListener: false,
        });
        // An explicit dependency is required from the Service to the Client certificate
        // Otherwise cloud formation will try to remove the cert before the ALB using it is disposed.
        if (this.clientCert) {
            this.pattern.node.addDependency(this.clientCert);
        }
        // An explicit dependency is required from the service to the ASG providing its capacity.
        // See: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-attribute-dependson.html
        this.pattern.service.node.addDependency(this.asg);
        this.loadBalancer = this.pattern.loadBalancer;
        // Enabling dropping of invalid HTTP header fields on the load balancer to prevent http smuggling attacks.
        this.loadBalancer.setAttribute('routing.http.drop_invalid_header_fields.enabled', 'true');
        if (props.accessLogs) {
            const accessLogsBucket = props.accessLogs.destinationBucket;
            // Policies are applied according to
            // https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:PutObject'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [`${accessLogsBucket.bucketArn}/*`],
                conditions: {
                    StringEquals: {
                        's3:x-amz-acl': 'bucket-owner-full-control',
                    },
                },
            }));
            accessLogsBucket.addToResourcePolicy(new aws_iam_1.PolicyStatement({
                actions: ['s3:GetBucketAcl'],
                principals: [new aws_iam_1.ServicePrincipal('delivery.logs.amazonaws.com')],
                resources: [accessLogsBucket.bucketArn],
            }));
            this.loadBalancer.logAccessLogs(accessLogsBucket, props.accessLogs.prefix);
        }
        // Ensure tasks are run on separate container instances
        this.pattern.service.addPlacementConstraints(aws_ecs_1.PlacementConstraint.distinctInstances());
        /**
         * Uses an escape-hatch to set the target group protocol to HTTPS. We cannot configure server certificate
         * validation, but at least traffic is encrypted and terminated at the application layer.
         */
        const listener = this.loadBalancer.node.findChild('PublicListener');
        this.listener = listener;
        const targetGroup = listener.node.findChild('ECSGroup');
        const targetGroupResource = targetGroup.node.defaultChild;
        targetGroupResource.protocol = aws_elasticloadbalancingv2_1.ApplicationProtocol[internalProtocol];
        targetGroupResource.port = internalPortNumber;
        this.grantPrincipal = taskDefinition.taskRole;
        this.connections = new aws_ec2_1.Connections({
            defaultPort: aws_ec2_1.Port.tcp(externalPortNumber),
            securityGroups: this.pattern.loadBalancer.connections.securityGroups,
        });
        this.endpoint = new core_2.ConnectableApplicationEndpoint({
            address: this.pattern.loadBalancer.loadBalancerDnsName,
            port: externalPortNumber,
            connections: this.connections,
            protocol: externalProtocol,
        });
        if (externalProtocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP) {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttp({
                endpoint: this.endpoint,
            });
        }
        else {
            this.rqConnection = rq_connection_1.RenderQueueConnection.forHttps({
                endpoint: this.endpoint,
                caCert: this.certChain,
            });
        }
        this.node.defaultChild = taskDefinition;
    }
    /**
     * @inheritdoc
     */
    configureClientECS(param) {
        param.hosts.forEach(host => this.addChildDependency(host));
        return this.rqConnection.configureClientECS(param);
    }
    /**
     * @inheritdoc
     */
    configureClientInstance(param) {
        this.addChildDependency(param.host);
        this.rqConnection.configureClientInstance(param);
    }
    /**
     * Add an ordering dependency to another Construct.
     *
     * All constructs in the child's scope will be deployed after the RenderQueue has been deployed and is ready to recieve traffic.
     *
     * This can be used to ensure that the RenderQueue is fully up and serving queries before a client attempts to connect to it.
     *
     * @param child The child to make dependent upon this RenderQueue.
     */
    addChildDependency(child) {
        // Narrowly define the dependencies to reduce the probability of cycles
        // ex: cycles that involve the security group of the RenderQueue & child.
        child.node.addDependency(this.listener);
        child.node.addDependency(this.taskDefinition);
    }
    createTaskDefinition(props) {
        const { image, portNumber, protocol, repository } = props;
        const taskDefinition = new aws_ecs_1.Ec2TaskDefinition(this, 'RCSTask');
        // Mount the repo filesystem to RenderQueue.HOST_REPO_FS_MOUNT_PATH
        const connection = repository.configureClientECS({
            containerInstances: {
                hosts: [this.asg],
            },
            containers: {
                taskDefinition,
            },
        });
        const environment = connection.containerEnvironment;
        if (protocol === aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS) {
            // Generate a self-signed X509 certificate, private key and passphrase for use by the RCS containers.
            // Note: the Application Load Balancer does not validate the certificate in any way.
            const rcsCertPem = new core_2.X509CertificatePem(this, 'TlsCaCertPem', {
                subject: {
                    cn: 'renderfarm.local',
                },
            });
            const rcsCertPkcs = new core_2.X509CertificatePkcs12(this, 'TlsRcsCertBundle', {
                sourceCertificate: rcsCertPem,
            });
            [rcsCertPem.cert, rcsCertPkcs.cert, rcsCertPkcs.passphrase].forEach(secret => {
                secret.grantRead(taskDefinition.taskRole);
            });
            environment.RCS_TLS_CA_CERT_URI = rcsCertPem.cert.secretArn;
            environment.RCS_TLS_CERT_URI = rcsCertPkcs.cert.secretArn;
            environment.RCS_TLS_CERT_PASSPHRASE_URI = rcsCertPkcs.passphrase.secretArn;
            environment.RCS_TLS_REQUIRE_CLIENT_CERT = 'no';
        }
        const containerDefinition = taskDefinition.addContainer('ContainerDefinition', {
            image,
            memoryReservationMiB: 2048,
            environment,
            logging: aws_ecs_1.LogDriver.awsLogs({
                logGroup: this.logGroup,
                streamPrefix: 'RCS',
            }),
        });
        containerDefinition.addMountPoints(connection.readWriteMountPoint);
        // Increase ulimits
        containerDefinition.addUlimits({
            name: aws_ecs_1.UlimitName.NOFILE,
            softLimit: 200000,
            hardLimit: 200000,
        }, {
            name: aws_ecs_1.UlimitName.NPROC,
            softLimit: 64000,
            hardLimit: 64000,
        });
        containerDefinition.addPortMappings({
            containerPort: portNumber,
            hostPort: portNumber,
        });
        return taskDefinition;
    }
}
exports.RenderQueue = RenderQueue;
/**
 * Container listening ports for each protocol.
 */
RenderQueue.RCS_PROTO_PORTS = {
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP]: 8080,
    [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS]: 4433,
};
/**
 * Regular expression that validates a hostname (portion in front of the subdomain).
 */
RenderQueue.RE_VALID_HOSTNAME = /^[a-z](?:[a-z0-9-]{0,61}[a-z0-9])?$/i;
//# sourceMappingURL=data:application/json;base64,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