"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('Test StaticIpServer', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        const app = new core_1.App();
        stack = new core_1.Stack(app, 'StackName');
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
    });
    test('basic setup', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            Properties: {
                MinSize: '1',
                MaxSize: '1',
                LifecycleHookSpecificationList: assert_1.arrayWith(assert_1.objectLike({
                    DefaultResult: 'ABANDON',
                    HeartbeatTimeout: 120,
                    LifecycleHookName: 'NewStaticPrivateIpServer',
                    LifecycleTransition: 'autoscaling:EC2_INSTANCE_LAUNCHING',
                    NotificationMetadata: {
                        'Fn::Join': assert_1.arrayWith([
                            '{\"eniId\":\"',
                            {
                                Ref: 'InstanceEniA230F5FE',
                            },
                            '\"}',
                        ]),
                    },
                })),
                Tags: assert_1.arrayWith({
                    Key: 'RfdkStaticPrivateIpServerGrantConditionKey',
                    PropagateAtLaunch: true,
                    Value: 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
                }),
            },
            DependsOn: assert_1.arrayWith('AttachEniToInstance83a5dca5db544aa485d28d419cdf85ceAttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0695D3E7F', // The SNS Topic Subscription; this is key.
            'InstanceEniA230F5FE'),
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::NetworkInterface', {
            Description: 'Static ENI for StackName/Instance',
            GroupSet: assert_1.arrayWith({
                'Fn::GetAtt': [
                    'InstanceAsgInstanceSecurityGroup2DB1DA8B',
                    'GroupId',
                ],
            }),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'index.handler',
            Runtime: 'nodejs12.x',
            Description: 'Created by AWS-RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack \'StackName\'. This lambda attaches an ENI to newly launched instances.',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            UpdateReplacePolicy: 'Delete',
            DeletionPolicy: 'Delete',
        }, assert_1.ResourcePart.CompleteDefinition));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
            KeyPolicy: {
                Statement: [
                    {
                        Action: 'kms:*',
                        Effect: 'Allow',
                        Principal: {
                            AWS: {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        {
                                            Ref: 'AWS::Partition',
                                        },
                                        ':iam::',
                                        {
                                            Ref: 'AWS::AccountId',
                                        },
                                        ':root',
                                    ],
                                ],
                            },
                        },
                        Resource: '*',
                    },
                ],
            },
            EnableKeyRotation: true,
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
            DisplayName: 'For AWS-RFDK instance-launch notifications for stack \'StackName\'',
            KmsMasterKeyId: {
                Ref: 'SNSEncryptionKey255e9e52ad034ddf8ff8274bc10d63d1EDF79FFE',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
            Protocol: 'lambda',
            TopicArn: {
                Ref: 'AttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0F5D22665',
            },
            Endpoint: {
                'Fn::GetAtt': [
                    'AttachEniToInstance83a5dca5db544aa485d28d419cdf85ce70724E62',
                    'Arn',
                ],
            },
        }));
        // The Lambda's policy should allow ENI attachment & condition-limited CompleteLifecycle.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith(assert_1.objectLike({
                    Action: 'autoscaling:CompleteLifecycleAction',
                    Effect: 'Allow',
                    Condition: {
                        'ForAnyValue:StringEquals': {
                            'autoscaling:ResourceTag/RfdkStaticPrivateIpServerGrantConditionKey': 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
                        },
                    },
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':autoscaling:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':autoScalingGroup:*:autoScalingGroupName/*',
                            ],
                        ],
                    },
                }), assert_1.objectLike({
                    Effect: 'Allow',
                    Action: [
                        'ec2:DescribeNetworkInterfaces',
                        'ec2:AttachNetworkInterface',
                    ],
                    Resource: '*',
                })),
            },
        }));
        // Count singleton objects
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 2)); // Log retention & event handler.
        assert_1.expect(stack).to(assert_1.countResources('AWS::SNS::Topic', 1));
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: 'sts:AssumeRole',
                        Principal: {
                            Service: 'autoscaling.amazonaws.com',
                        },
                    },
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Policy', 1, {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'kms:Decrypt',
                            'kms:GenerateDataKey',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'SNSEncryptionKey255e9e52ad034ddf8ff8274bc10d63d1EDF79FFE',
                                'Arn',
                            ],
                        },
                    },
                    {
                        Action: 'sns:Publish',
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'AttachEniNotificationTopicc8b1e9a6783c4954b191204dd5e3b9e0F5D22665',
                        },
                    },
                ],
            },
        }));
    });
    test('creates singleton resources', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance1', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        new lib_1.StaticPrivateIpServer(stack, 'Instance2', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
        });
        // THEN
        // Make sure both ASGs are tagged to allow CompleteLifeCycle by the singleton lambda.
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::AutoScaling::AutoScalingGroup', 2, {
            Tags: assert_1.arrayWith({
                Key: 'RfdkStaticPrivateIpServerGrantConditionKey',
                PropagateAtLaunch: true,
                Value: 'StackNameAttachEniToInstance83a5dca5db544aa485d28d419cdf85ceF20CDF73',
            }),
        }));
        // Count singleton objects
        assert_1.expect(stack).to(assert_1.countResources('AWS::Lambda::Function', 2)); // Log retention & event handler.
        assert_1.expect(stack).to(assert_1.countResources('AWS::SNS::Topic', 1));
        assert_1.expect(stack).to(assert_1.countResourcesLike('AWS::IAM::Role', 1, {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Effect: 'Allow',
                        Action: 'sts:AssumeRole',
                        Principal: {
                            Service: 'autoscaling.amazonaws.com',
                        },
                    },
                ],
            },
        }));
    });
    test('throw exception when no available subnets', () => {
        // THEN
        expect(() => {
            new lib_1.StaticPrivateIpServer(stack, 'Instance', {
                vpc,
                instanceType: new aws_ec2_1.InstanceType('t3.small'),
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                vpcSubnets: {
                    subnetType: aws_ec2_1.SubnetType.PRIVATE,
                    availabilityZones: ['dummy zone'],
                },
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('resource signal count', () => {
        // WHEN
        new lib_1.StaticPrivateIpServer(stack, 'Instance', {
            vpc,
            instanceType: new aws_ec2_1.InstanceType('t3.small'),
            machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
            resourceSignalTimeout: core_1.Duration.hours(12),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::AutoScalingGroup', {
            CreationPolicy: {
                ResourceSignal: {
                    Count: 1,
                    Timeout: 'PT12H',
                },
            },
        }, assert_1.ResourcePart.CompleteDefinition));
        expect(() => {
            new lib_1.StaticPrivateIpServer(stack, 'InstanceFail', {
                vpc,
                instanceType: new aws_ec2_1.InstanceType('t3.small'),
                machineImage: aws_ec2_1.MachineImage.latestAmazonLinux({ generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2 }),
                resourceSignalTimeout: core_1.Duration.seconds(12 * 60 * 60 + 1),
            });
        }).toThrowError('Resource signal timeout cannot exceed 12 hours.');
    });
});
//# sourceMappingURL=data:application/json;base64,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