"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const script_assets_1 = require("../lib/script-assets");
const asset_constants_1 = require("./asset-constants");
const instanceType = aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T3, aws_ec2_1.InstanceSize.MICRO);
const linuxImage = new aws_ec2_1.AmazonLinuxImage();
const windowsImage = new aws_ec2_1.WindowsImage(aws_ec2_1.WindowsVersion.WINDOWS_SERVER_2019_ENGLISH_FULL_BASE);
describe('executeScriptAsset', () => {
    let stack;
    let vpc;
    beforeEach(() => {
        // Create a new stack, vpc, and instance for each test
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'vpc');
    });
    it.each([
        [linuxImage],
        [windowsImage],
    ])('grants read permissions', (machineImage) => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'inst', {
            vpc,
            instanceType,
            machineImage,
        });
        const asset = new script_assets_1.ScriptAsset(stack, 'asset', {
            path: path.join(__dirname, '../scripts/bash/configureCloudWatchAgent.sh'),
        });
        // WHEN
        asset.executeOn({ host: instance });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            's3:GetObject*',
                            's3:GetBucket*',
                            's3:List*',
                        ],
                        Effect: 'Allow',
                        Resource: [
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':s3:::',
                                        { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                                    ],
                                ],
                            },
                            {
                                'Fn::Join': [
                                    '',
                                    [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':s3:::',
                                        { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                                        '/*',
                                    ],
                                ],
                            },
                        ],
                    },
                ],
                Version: '2012-10-17',
            },
            PolicyName: 'instInstanceRoleDefaultPolicyCB9E402C',
            Roles: [{ Ref: 'instInstanceRoleFE783FB1' }],
        }));
    });
    test('downloads and executes script for linux', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'inst', {
            vpc,
            instanceType,
            machineImage: linuxImage,
        });
        const asset = new script_assets_1.ScriptAsset(stack, 'asset', {
            path: path.join(__dirname, '../scripts/bash/configureCloudWatchAgent.sh'),
        });
        // WHEN
        asset.executeOn({
            host: instance,
            args: ['arg1'],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            "#!/bin/bash\nmkdir -p $(dirname '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "')\naws s3 cp 's3://",
                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                            '/',
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "' '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "'\nset -e\nchmod +x '/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "'\n'/tmp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "' arg1",
                        ],
                    ],
                },
            },
        }));
    });
    test('downloads and executes script for windows', () => {
        // GIVEN
        const instance = new aws_ec2_1.Instance(stack, 'inst', {
            vpc,
            instanceType,
            machineImage: windowsImage,
        });
        const asset = new script_assets_1.ScriptAsset(stack, 'asset', {
            path: path.join(__dirname, '../scripts/bash/configureCloudWatchAgent.sh'),
        });
        // WHEN
        asset.executeOn({
            host: instance,
            args: ['arg1'],
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::Instance', {
            UserData: {
                'Fn::Base64': {
                    'Fn::Join': [
                        '',
                        [
                            "<powershell>mkdir (Split-Path -Path 'C:/temp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "' ) -ea 0\nRead-S3Object -BucketName '",
                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Bucket },
                            "' -key '",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "' -file 'C:/temp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "' -ErrorAction Stop\n&'C:/temp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "' arg1\nif (!$?) { Write-Error 'Failed to execute the file \"C:/temp/",
                            {
                                'Fn::Select': [
                                    0,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            {
                                'Fn::Select': [
                                    1,
                                    {
                                        'Fn::Split': [
                                            '||',
                                            { Ref: asset_constants_1.CWA_ASSET_LINUX.Key },
                                        ],
                                    },
                                ],
                            },
                            "\"' -ErrorAction Stop }</powershell>",
                        ],
                    ],
                },
            },
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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