"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
describe('MongoDbPostInstall', () => {
    let stack;
    let vpc;
    let mongoDb;
    let pwUser1Arn;
    let pwUser2Arn;
    let pwUser1;
    let pwUser2;
    let x509User1Arn;
    let x509User2Arn;
    let x509User1;
    let x509User2;
    beforeEach(() => {
        const hostname = 'mongodb';
        const zoneName = 'testZone.internal';
        const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
        const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        const dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        const caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        const serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
        mongoDb = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        pwUser1Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1';
        pwUser2Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2';
        pwUser1 = aws_secretsmanager_1.Secret.fromSecretArn(stack, 'PwUser1', pwUser1Arn);
        pwUser2 = aws_secretsmanager_1.Secret.fromSecretArn(stack, 'PwUser2', pwUser2Arn);
        x509User1Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/X509User1';
        x509User2Arn = 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/X509User2';
        x509User1 = {
            certificate: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'x509User1', x509User1Arn),
            roles: JSON.stringify([{ role: 'readWrite', db: 'testdb1' }]),
        };
        x509User2 = {
            certificate: aws_secretsmanager_1.Secret.fromSecretArn(stack, 'x509User2', x509User2Arn),
            roles: JSON.stringify([{ role: 'readWrite', db: 'testdb2' }]),
        };
    });
    test('created correctly: both user types', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'mongodb.configureMongo',
            Environment: {
                Variables: {
                    DEBUG: 'false',
                },
            },
            Runtime: 'nodejs12.x',
            VpcConfig: {
                SecurityGroupIds: [
                    {
                        'Fn::GetAtt': [
                            'MongoPostInstallLambdaSecurityGroup62729E3A',
                            'GroupId',
                        ],
                    },
                ],
                SubnetIds: [
                    {
                        Ref: 'VpcPrivateSubnet1Subnet536B997A',
                    },
                    {
                        Ref: 'VpcPrivateSubnet2Subnet3788AAA1',
                    },
                ],
            },
        }));
        // Lambda role can get the required secrets.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'MongoDbInstanceAdminUser54147F2B',
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1',
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2',
                    },
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_MongoDbPostInstallSetup', {
            Properties: {
                Connection: {
                    Hostname: 'mongodb.testZone.internal',
                    Port: '27017',
                    CaCertificate: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                    Credentials: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                },
                PasswordAuthUsers: [
                    pwUser1Arn,
                    pwUser2Arn,
                ],
                X509AuthUsers: [
                    {
                        Certificate: x509User1Arn,
                        Roles: x509User1.roles,
                    },
                    {
                        Certificate: x509User2Arn,
                        Roles: x509User2.roles,
                    },
                ],
            },
            DependsOn: [
                'MongoDbInstanceServerAsgASG2643AD1D',
                'MongoPostInstallLambdaServiceRoleDefaultPolicy8B1C1CE8',
                'MongoPostInstallLambdaServiceRoleCD03B9B9',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
    });
    test('created correctly: only password users', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        // Lambda role can get the required secrets.
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            'Fn::GetAtt': [
                                'ServerCert',
                                'CertChain',
                            ],
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: {
                            Ref: 'MongoDbInstanceAdminUser54147F2B',
                        },
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User1',
                    },
                    {
                        Action: [
                            'secretsmanager:GetSecretValue',
                            'secretsmanager:DescribeSecret',
                        ],
                        Effect: 'Allow',
                        Resource: 'arn:aws:secretsmanager:us-west-1:1234567890:secret:SecretPath/User2',
                    },
                ],
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_MongoDbPostInstallSetup', {
            Connection: {
                Hostname: 'mongodb.testZone.internal',
                Port: '27017',
                CaCertificate: {
                    'Fn::GetAtt': [
                        'ServerCert',
                        'CertChain',
                    ],
                },
                Credentials: {
                    Ref: 'MongoDbInstanceAdminUser54147F2B',
                },
            },
            PasswordAuthUsers: [
                pwUser1Arn,
                pwUser2Arn,
            ],
        }));
    });
    test('created correctly: only x509 users', () => {
        // GIVEN
        const users = {
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::RFDK_MongoDbPostInstallSetup', {
            Connection: {
                Hostname: 'mongodb.testZone.internal',
                Port: '27017',
                CaCertificate: {
                    'Fn::GetAtt': [
                        'ServerCert',
                        'CertChain',
                    ],
                },
                Credentials: {
                    Ref: 'MongoDbInstanceAdminUser54147F2B',
                },
            },
            X509AuthUsers: [
                {
                    Certificate: x509User1Arn,
                    Roles: x509User1.roles,
                },
                {
                    Certificate: x509User2Arn,
                    Roles: x509User2.roles,
                },
            ],
        }));
    });
    test('use selected subnets', () => {
        // GIVEN
        const users = {
            passwordAuthUsers: [pwUser1, pwUser2],
            x509AuthUsers: [x509User1, x509User2],
        };
        // WHEN
        new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
            vpc,
            vpcSubnets: { subnets: [vpc.privateSubnets[0]] },
            mongoDb,
            users,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', {
            Handler: 'mongodb.configureMongo',
            VpcConfig: {
                SubnetIds: [
                    {
                        Ref: 'VpcPrivateSubnet1Subnet536B997A',
                    },
                ],
            },
        }));
    });
    test('assert bad x509 role', () => {
        // GIVEN
        const users = {
            x509AuthUsers: [
                {
                    certificate: x509User1.certificate,
                    roles: '}{',
                },
            ],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbPostInstallSetup(stack, 'MongoPostInstall', {
                vpc,
                mongoDb,
                users,
            });
        }).toThrowError(/MongoDbPostInstallSetup: Could not parse JSON role for x509 user:/);
    });
});
//# sourceMappingURL=data:application/json;base64,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