"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_route53_1 = require("@aws-cdk/aws-route53");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const asset_constants_1 = require("./asset-constants");
const token_regex_helpers_1 = require("./token-regex-helpers");
describe('Test MongoDbInstance', () => {
    let stack;
    let vpc;
    let dnsZone;
    let caCert;
    let serverCert;
    const hostname = 'hostname';
    const zoneName = 'testZone';
    const version = lib_1.MongoDbVersion.COMMUNITY_3_6;
    const userSsplAcceptance = lib_1.MongoDbSsplLicenseAcceptance.USER_ACCEPTS_SSPL;
    beforeEach(() => {
        stack = new core_1.Stack();
        vpc = new aws_ec2_1.Vpc(stack, 'Vpc');
        dnsZone = new aws_route53_1.PrivateHostedZone(stack, 'PrivateHostedZone', {
            vpc,
            zoneName,
        });
        caCert = new lib_1.X509CertificatePem(stack, 'CaCert', {
            subject: {
                cn: 'DistinguishedName',
            },
        });
        serverCert = new lib_1.X509CertificatePem(stack, 'ServerCert', {
            subject: {
                cn: `${hostname}.${zoneName}`,
            },
            signingCertificate: caCert,
        });
    });
    test('default mongodb instance is created correctly', () => {
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::AutoScaling::AutoScalingGroup'));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: 'r5.large',
            BlockDeviceMappings: assert_1.arrayWith(assert_1.objectLike({
                Ebs: assert_1.objectLike({
                    Encrypted: true,
                }),
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Route53::RecordSet', {
            Name: hostname + '.' + zoneName + '.',
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SecretsManager::Secret', {
            Description: 'Admin credentials for the MongoDB database MongoDbInstance',
            GenerateSecretString: {
                ExcludeCharacters: '\"()$\'',
                ExcludePunctuation: true,
                GenerateStringKey: 'password',
                IncludeSpace: false,
                PasswordLength: 24,
                RequireEachIncludedType: true,
                SecretStringTemplate: '{\"username\":\"admin\"}',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            Tags: assert_1.arrayWith(assert_1.objectLike({
                Key: 'VolumeGrantAttach-785e71aba44fe1179b3900b2d6024184',
                Value: 'c66ef9c61ae07e231dff5ff9abf99ca1',
            })),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: assert_1.objectLike({
                Statement: assert_1.arrayWith({
                    Action: [
                        'logs:CreateLogStream',
                        'logs:PutLogEvents',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::Join': [
                            '',
                            [
                                'arn:',
                                {
                                    Ref: 'AWS::Partition',
                                },
                                ':logs:',
                                {
                                    Ref: 'AWS::Region',
                                },
                                ':',
                                {
                                    Ref: 'AWS::AccountId',
                                },
                                ':log-group:',
                                {
                                    'Fn::GetAtt': [
                                        'MongoDbInstanceMongoDbInstanceLogGroupWrapperEAF733BB',
                                        'LogGroupName',
                                    ],
                                },
                                ':*',
                            ],
                        ],
                    },
                }, {
                    Action: [
                        's3:GetObject*',
                        's3:GetBucket*',
                        's3:List*',
                    ],
                    Effect: 'Allow',
                    Resource: [
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.MONGODB_INSTANCE_3_6_SCRIPT.Bucket,
                                    },
                                ],
                            ],
                        },
                        {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':s3:::',
                                    {
                                        Ref: asset_constants_1.MONGODB_INSTANCE_3_6_SCRIPT.Bucket,
                                    },
                                    '/*',
                                ],
                            ],
                        },
                    ],
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Cert',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'CertChain',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        'Fn::GetAtt': [
                            'ServerCert',
                            'Key',
                        ],
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'ServerCertPassphraseE4C3CB38',
                    },
                }, {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Effect: 'Allow',
                    Resource: {
                        Ref: 'MongoDbInstanceAdminUser54147F2B',
                    },
                }),
            }),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::LogRetention', {
            LogGroupName: '/renderfarm/MongoDbInstance',
        }));
        const cloudInitLogPath = '/var/log/cloud-init-output.log';
        const cloudInitLogPrefix = 'cloud-init-output';
        const mongoLogPath = '/var/log/mongodb/mongod.log';
        const mongoLogPrefix = 'MongoDB';
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::SSM::Parameter', {
            Description: 'config file for Repository logs config',
            Value: assert_1.objectLike({
                'Fn::Join': assert_1.arrayWith(assert_1.arrayWith('\",\"log_stream_name\":\"' + cloudInitLogPrefix + '-{instance_id}\",\"file_path\":\"' + cloudInitLogPath + '\",' +
                    '\"timezone\":\"Local\"},{\"log_group_name\":\"', '\",\"log_stream_name\":\"' + mongoLogPrefix + '-{instance_id}\",\"file_path\":\"' + mongoLogPath + '\"' +
                    ',\"timezone\":\"Local\"}]}},\"log_stream_name\":\"DefaultLogStream-{instance_id}\",\"force_flush_interval\":15}}')),
            }),
        }));
        const userData = instance.userData.render();
        const token = '${Token[TOKEN.\\d+]}';
        // Make sure we add signal on exit
        const exitTrap = '#!/bin/bash\n' +
            'function exitTrap(){\n' +
            'exitCode=$?\n' +
            '/opt/aws/bin/cfn-signal --stack Stack --resource MongoDbInstanceServerAsgASG2643AD1D --region ' + token +
            ' -e $exitCode || echo \'Failed to send Cloudformation Signal\'\n' +
            'test \"${MONGO_SETUP_DIR} != \"\" && sudo umount \"${MONGO_SETUP_DIR}\n' +
            '}';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(exitTrap)));
        const callExitTrap = 'trap exitTrap EXIT';
        expect(userData).toMatch(new RegExp(callExitTrap));
        const settings = 'set -xefuo pipefail';
        expect(userData).toMatch(new RegExp(settings));
        const createTempDir = 'mkdir -p $\\(dirname \'/tmp/' + token + token + '\'\\)\n';
        const s3Copy = 'aws s3 cp \'s3://' + token + '/' + token + token + '\' \'/tmp/' + token + token + '\'\n';
        // CloudWatch Agent
        const setE = 'set -e\n';
        const setChmod = 'chmod \\+x \'/tmp/' + token + token + '\'\n';
        const execute = '\'/tmp/' + token + token + '\' ' + token + '\n';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + setE + setChmod + execute)));
        // Make sure we mount EBS volume
        const mount = 'TMPDIR=$\\(mktemp -d\\)\n' +
            'pushd \"$TMPDIR\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'bash ./mountEbsBlockVolume.sh ' + token + ' xfs /var/lib/mongo rw \"\"\n' +
            'popd\n' +
            'rm -f /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + mount)));
        // install mongodb
        const bashCmd = 'bash /tmp/' + token + token;
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + bashCmd)));
        // configureMongoDb
        const monogdbConfig = 'which mongod && test -f /etc/mongod.conf\n' +
            'sudo service mongod stop\n' +
            'MONGO_SETUP_DIR=$\\(mktemp -d\\)\n' +
            'mkdir -p \"${MONGO_SETUP_DIR}\"\n' +
            'sudo mount -t tmpfs -o size=50M tmpfs \"${MONGO_SETUP_DIR}\"\n' +
            'pushd \"${MONGO_SETUP_DIR}\"\n' +
            'unzip /tmp/' + token + token + '\n' +
            'cp /etc/mongod.conf .';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(createTempDir + s3Copy + monogdbConfig)));
        // Getting the server certificate
        const serverCertCmd = 'bash serverCertFromSecrets.sh \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\" \\"' + token + '\\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(serverCertCmd)));
        // set mongodb certificates and credentials
        const monogdbCredentials = 'sudo mkdir -p /etc/mongod_certs\n' +
            'sudo mv ./ca.crt ./key.pem /etc/mongod_certs\n' +
            'sudo chown root.mongod -R /etc/mongod_certs/\n' +
            'sudo chmod 640 -R /etc/mongod_certs/\n' +
            'sudo chmod 750 /etc/mongod_certs/\n' +
            'sudo chown mongod.mongod -R /var/lib/mongo\n' +
            'bash ./setMongoLimits.sh\n' +
            'bash ./setStoragePath.sh \"/var/lib/mongo\"\n' +
            'bash ./setMongoNoAuth.sh\n' +
            'sudo service mongod start\n' +
            'bash ./setAdminCredentials.sh \"' + token + '\"';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(monogdbCredentials)));
        // Setup for live deployment, and start mongod
        const startMongo = 'sudo service mongod stop\n' +
            'bash ./setLiveConfiguration.sh\n' +
            'sudo systemctl enable mongod\n' +
            'sudo service mongod start\n' +
            'popd';
        expect(userData).toMatch(new RegExp(token_regex_helpers_1.escapeTokenRegex(startMongo)));
        // Make sure all the required public members are set
        expect(instance.version).toBe(version);
        expect(instance.connections).toBeDefined();
        expect(instance.connections).toBe(instance.server.connections);
        expect(instance.grantPrincipal).toBeDefined();
        expect(instance.grantPrincipal).toBe(instance.server.grantPrincipal);
        expect(instance.port).toBeDefined();
        expect(instance.role).toBeDefined();
        expect(instance.role).toBe(instance.server.role);
        expect(instance.userData).toBeDefined();
        expect(instance.userData).toBe(instance.server.userData);
        expect(instance.fullHostname).toBeDefined();
    });
    test('throw exception when no available subnets', () => {
        // GIVEN
        const invalidSubnets = {
            subnetType: aws_ec2_1.SubnetType.PRIVATE,
            availabilityZones: ['dummy zone'],
        };
        // THEN
        expect(() => {
            new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
                mongoDb: {
                    version,
                    dnsZone,
                    hostname,
                    serverCertificate: serverCert,
                    userSsplAcceptance,
                },
                vpc,
                vpcSubnets: invalidSubnets,
            });
        }).toThrowError(/Did not find any subnets matching/);
    });
    test('changing instance type works correctly', () => {
        // GIVEN
        const expectedInstanceType = 'm4.micro';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            instanceType: new aws_ec2_1.InstanceType(expectedInstanceType),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            InstanceType: expectedInstanceType,
        }));
    });
    test('allowing ssh connection with key name', () => {
        // GIVEN
        const expectedKeyName = 'someKeyName';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            keyName: expectedKeyName,
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::AutoScaling::LaunchConfiguration', {
            KeyName: expectedKeyName,
        }));
    });
    test('using custom admin user works correctly', () => {
        // GIVEN
        const expectedAdminUser = new aws_secretsmanager_1.Secret(stack, 'AdminUser', {
            description: 'Custom admin credentials for the MongoDB database',
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
                generateStringKey: 'test_password',
                secretStringTemplate: JSON.stringify({ username: 'admin' }),
            },
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                adminUser: expectedAdminUser,
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
        });
        // THEN
        expect(instance.adminUser).toBe(expectedAdminUser);
    });
    test('setting security group works correctly', () => {
        // GIVEN
        const actualSecurityGroup = new aws_ec2_1.SecurityGroup(stack, 'SecurityGroup', {
            securityGroupName: 'CustomSecurityGroup',
            vpc,
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            securityGroup: actualSecurityGroup,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::SecurityGroup', 1));
    });
    test('setting role works correctly', () => {
        // GIVEN
        const expectedRole = new aws_iam_1.Role(stack, 'Role', {
            assumedBy: new aws_iam_1.ServicePrincipal('ec2.amazonaws.com'),
            roleName: 'CustomRole',
        });
        // WHEN
        const instance = new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            role: expectedRole,
        });
        // THEN
        expect(instance.server.role).toBe(expectedRole);
        expect(instance.role).toBe(expectedRole);
    });
    test('setting custom data volume works correctly', () => {
        // GIVEN
        const actualVolume = new aws_ec2_1.Volume(stack, 'Volume', {
            availabilityZone: 'us-east-1a',
            size: core_1.Size.gibibytes(50),
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volume: actualVolume,
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.countResources('AWS::EC2::Volume', 1));
    });
    test('setting custom encryption key for data volume works correctly', () => {
        // GIVEN
        // KmsKeyId is Key961B73FD
        const actualEncryptionKey = new aws_kms_1.Key(stack, 'Key', {
            description: 'Key for testing',
        });
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        encryptionKey: actualEncryptionKey,
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Encrypted: true,
            KmsKeyId: assert_1.objectLike({
                'Fn::GetAtt': assert_1.arrayWith('Key961B73FD'),
            }),
        }));
    });
    test('setting custom size for data volume works correctly', () => {
        // GIVEN
        const volumeSize = 123;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
                mongoDataVolume: {
                    volumeProps: {
                        size: core_1.Size.gibibytes(volumeSize),
                    },
                },
            },
            vpc,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::EC2::Volume', {
            Size: volumeSize,
        }));
    });
    test('setting LogGroup bucket name enables export to S3', () => {
        // GIVEN
        const bucketName = 'test-bucket';
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                bucketName,
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Events::Rule', {
            Targets: assert_1.arrayWith(assert_1.objectLike({
                Input: '{\"BucketName\":\"' + bucketName + '\",\"ExportFrequencyInHours\":1,\"LogGroupName\":\"/renderfarm/MongoDbInstance\",\"RetentionInHours\":72}',
            })),
        }));
    });
    test('is created with correct LogGroup prefix', () => {
        // GIVEN
        const logGroupPrefix = 'test-prefix/';
        const uniqueId = 'MongoDbInstance';
        // WHEN
        new lib_1.MongoDbInstance(stack, uniqueId, {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                logGroupPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: logGroupPrefix + uniqueId,
        }));
    });
    test('not using default LogGroup prefix if prefix is empty', () => {
        // GIVEN
        const logGroupPrefix = '';
        const uniqueId = 'MongoDbInstance';
        const expectedLogGroupName = logGroupPrefix + uniqueId;
        const defaultLogGroupName = '/renderfarm/' + uniqueId;
        // WHEN
        new lib_1.MongoDbInstance(stack, uniqueId, {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                logGroupPrefix,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: expectedLogGroupName,
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('Custom::LogRetention', {
            LogGroupName: defaultLogGroupName,
        }));
    });
    test('is created with correct LogGroup retention', () => {
        // GIVEN
        const retention = aws_logs_1.RetentionDays.ONE_DAY;
        // WHEN
        new lib_1.MongoDbInstance(stack, 'MongoDbInstance', {
            mongoDb: {
                version,
                dnsZone,
                hostname,
                serverCertificate: serverCert,
                userSsplAcceptance,
            },
            vpc,
            logGroupProps: {
                retention,
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('Custom::LogRetention', {
            RetentionInDays: retention,
        }));
    });
});
//# sourceMappingURL=data:application/json;base64,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