"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
/* eslint-disable no-console */
const AWS = require("aws-sdk-mock");
const sinon = require("sinon");
const lambdaCode = require("../../lambdas/nodejs/unhealthyFleetAction/index");
AWS.setSDK(require.resolve('aws-sdk'));
const sampleEvent = {
    Records: [{
            EventSource: 'aws:sns',
            EventVersion: '1.0',
            EventSubscriptionArn: 'arn-test',
            Sns: {
                Type: 'Notification',
                MessageId: '8b15e083-30a3-5a57-a17e-b7a721ad365d',
                TopicArn: 'arn-test',
                Subject: 'ALARM: "test" in US West (Oregon)',
                Timestamp: '2020-04-29T23:33:34.928Z',
                Message: '{"AlarmName":"testAlarm","AlarmDescription":null,"NewStateValue":"ALARM","NewStateReason":"Threshold Crossed: 5 out of the last 5 datapoints were less than the threshold (65.0). The most recent datapoints which crossed the threshold: [0.0 (29/04/20 23:32:00), 0.0 (29/04/20 23:31:00), 0.0 (29/04/20 23:30:00), 0.0 (29/04/20 23:29:00), 0.0 (29/04/20 23:28:00)] (minimum 5 datapoints for OK -> ALARM transition).","StateChangeTime":"2020-04-29T23:33:34.876+0000","Region":"US West (Oregon)","AlarmArn":"test-arn","OldStateValue":"INSUFFICIENT_DATA","Trigger":{"Period":60,"EvaluationPeriods":5,"ComparisonOperator":"LessThanThreshold","Threshold":65.0,"TreatMissingData":"- TreatMissingData:                    missing","EvaluateLowSampleCountPercentile":"","Metrics":[{"Expression":"100*(healthyHostCount/fleetCapacity)","Id":"expr_1","ReturnData":true},{"Id":"healthyHostCount","Label":"HealthyHostCount","MetricStat":{"Metric":{"Dimensions":[{"value":"testTargetGroup","name":"TargetGroup"},{"value":"testLoadBalancer","name":"LoadBalancer"}],"MetricName":"HealthyHostCount","Namespace":"AWS/NetworkELB"},"Period":60,"Stat":"Average"},"ReturnData":false},{"Id":"fleetCapacity","Label":"GroupDesiredCapacity","MetricStat":{"Metric":{"Dimensions":[{"value":"testFleetId","name":"AutoScalingGroupName"}],"MetricName":"GroupDesiredCapacity","Namespace":"AWS/AutoScaling"},"Period":60,"Stat":"Average"},"ReturnData":false}]}}',
                SignatureVersion: '1',
                Signature: 'testSignature',
                SigningCertUrl: 'testSigningCertUrl',
                UnsubscribeUrl: 'testUnsubscribeURL',
                MessageAttributes: {},
            },
        }],
};
const context = {
    functionName: 'provider',
};
beforeEach(() => {
    console.log = () => { };
    console.error = () => { };
    console.info = () => { };
});
afterEach(() => {
    AWS.restore();
});
test('success scenario single fleet', async () => {
    // WHEN
    const updateAutoScalingGroupFake = sinon.fake.resolves({});
    AWS.mock('AutoScaling', 'updateAutoScalingGroup', updateAutoScalingGroupFake);
    const result = (await lambdaCode.handler(sampleEvent, context));
    // THEN
    expect(result.status).toEqual('OK');
    sinon.assert.calledWith(updateAutoScalingGroupFake, {
        AutoScalingGroupName: 'testFleetId',
        MaxSize: 0,
        MinSize: 0,
        DesiredCapacity: 0,
    });
});
test('failure scenario, AWS api returns failure', async () => {
    // WHEN
    const error = new Error();
    error.code = 'AccessDeniedException';
    const updateAutoScalingGroupFake = sinon.fake.rejects(error);
    AWS.mock('AutoScaling', 'updateAutoScalingGroup', updateAutoScalingGroupFake);
    const result = (await lambdaCode.handler(sampleEvent, context));
    // THEN
    expect(result.status).toEqual('ERROR');
    expect(result.reason).toMatch(/Exception while suspending fleet/);
    sinon.assert.calledWith(updateAutoScalingGroupFake, {
        AutoScalingGroupName: 'testFleetId',
        MaxSize: 0,
        MinSize: 0,
        DesiredCapacity: 0,
    });
});
test('failure scenario, MetricStat not found', async () => {
    // WHEN
    const successEventSingle = JSON.parse(JSON.stringify(sampleEvent));
    successEventSingle.Records[0].Sns.Message = '{"AlarmName":"testAlarm","AlarmDescription":null,"NewStateValue":"ALARM","NewStateReason":"Threshold Crossed: 5 out of the last 5 datapoints were less than the threshold (65.0). The most recent datapoints which crossed the threshold: [0.0 (29/04/20 23:32:00), 0.0 (29/04/20 23:31:00), 0.0 (29/04/20 23:30:00), 0.0 (29/04/20 23:29:00), 0.0 (29/04/20 23:28:00)] (minimum 5 datapoints for OK -> ALARM transition).","StateChangeTime":"2020-04-29T23:33:34.876+0000","Region":"US West (Oregon)","AlarmArn":"test-arn","OldStateValue":"INSUFFICIENT_DATA","Trigger":{"Period":60,"EvaluationPeriods":5,"ComparisonOperator":"LessThanThreshold","Threshold":65.0,"TreatMissingData":"- TreatMissingData:                    missing","EvaluateLowSampleCountPercentile":"","Metrics":[{"Expression":"100*(healthyHostCount/fleetCapacity)","Id":"expr_1","ReturnData":true},{"Id":"healthyHostCount","Label":"HealthyHostCount","MetricStat":{"Metric":{"Dimensions":[{"value":"testTargetGroup","name":"TargetGroup"},{"value":"testLoadBalancer","name":"LoadBalancer"}],"MetricName":"HealthyHostCount","Namespace":"AWS/NetworkELB"},"Period":60,"Stat":"Average"},"ReturnData":false},{"Id":"fleetCapacity","Label":"GroupDesiredCapacity","M":{"Metric":{"Dimensions":[{"value":"testFleetId2","name":"AutoScalingGroupName"}],"MetricName":"GroupDesiredCapacity","Namespace":"AWS/AutoScaling"},"Period":60,"Stat":"Average"},"ReturnData":false}]}}';
    const updateAutoScalingGroupFake = sinon.fake.resolves({});
    AWS.mock('AutoScaling', 'updateAutoScalingGroup', updateAutoScalingGroupFake);
    const result = (await lambdaCode.handler(successEventSingle, context));
    // THEN
    expect(result.status).toEqual('ERROR');
    sinon.assert.notCalled(updateAutoScalingGroupFake);
});
test('Error if 2 records are found', async () => {
    // WHEN
    const successEventSingle = JSON.parse(JSON.stringify(sampleEvent));
    successEventSingle.Records.push(JSON.parse(JSON.stringify(successEventSingle.Records[0])));
    const updateAutoScalingGroupFake = sinon.fake.resolves({});
    AWS.mock('AutoScaling', 'updateAutoScalingGroup', updateAutoScalingGroupFake);
    const result = (await lambdaCode.handler(successEventSingle, context));
    // THEN
    expect(result.status).toEqual('ERROR');
    expect(result.reason).toMatch(/Expecting a single record in SNS Event/);
    sinon.assert.notCalled(updateAutoScalingGroupFake);
});
test('Error if exactly 3 metrics are not found', async () => {
    // WHEN
    const successEventSingle = JSON.parse(JSON.stringify(sampleEvent));
    successEventSingle.Records[0].Sns.Message = '{"AlarmName":"testAlarm","AlarmDescription":null,"NewStateValue":"ALARM","NewStateReason":"Threshold Crossed: 5 out of the last 5 datapoints were less than the threshold (65.0). The most recent datapoints which crossed the threshold: [0.0 (29/04/20 23:32:00), 0.0 (29/04/20 23:31:00), 0.0 (29/04/20 23:30:00), 0.0 (29/04/20 23:29:00), 0.0 (29/04/20 23:28:00)] (minimum 5 datapoints for OK -> ALARM transition).","StateChangeTime":"2020-04-29T23:33:34.876+0000","Region":"US West (Oregon)","AlarmArn":"test-arn","OldStateValue":"INSUFFICIENT_DATA","Trigger":{"Period":60,"EvaluationPeriods":5,"ComparisonOperator":"LessThanThreshold","Threshold":65.0,"TreatMissingData":"- TreatMissingData:                    missing","EvaluateLowSampleCountPercentile":"","Metrics":[{"Id":"healthyHostCount","Label":"HealthyHostCount","MetricStat":{"Metric":{"Dimensions":[{"value":"testTargetGroup","name":"TargetGroup"},{"value":"testLoadBalancer","name":"LoadBalancer"}],"MetricName":"HealthyHostCount","Namespace":"AWS/NetworkELB"},"Period":60,"Stat":"Average"},"ReturnData":false},{"Id":"fleetCapacity","Label":"GroupDesiredCapacity","MetricStat":{"Metric":{"Dimensions":[{"value":"testFleetId2","name":"AutoScalingGroupName"}],"MetricName":"GroupDesiredCapacity","Namespace":"AWS/AutoScaling"},"Period":60,"Stat":"Average"},"ReturnData":false}]}}';
    const updateAutoScalingGroupFake = sinon.fake.resolves({});
    AWS.mock('AutoScaling', 'updateAutoScalingGroup', updateAutoScalingGroupFake);
    const result = (await lambdaCode.handler(successEventSingle, context));
    // THEN
    expect(result.status).toEqual('ERROR');
    expect(result.reason).toMatch(/Exactly 3 metrics should be present in the alarm message/);
    sinon.assert.notCalled(updateAutoScalingGroupFake);
});
test('failure scenario, incorrect dimension, metrics and message', async () => {
    // WHEN
    const successEventSingle = JSON.parse(JSON.stringify(sampleEvent));
    const updateAutoScalingGroupFake = sinon.fake.resolves({});
    AWS.mock('AutoScaling', 'updateAutoScalingGroup', updateAutoScalingGroupFake);
    successEventSingle.Records[0].Sns.Message = '{"AlarmName":"testAlarm","AlarmDescription":null,"NewStateValue":"ALARM","NewStateReason":"Threshold Crossed: 5 out of the last 5 datapoints were less than the threshold (65.0). The most recent datapoints which crossed the threshold: [0.0 (29/04/20 23:32:00), 0.0 (29/04/20 23:31:00), 0.0 (29/04/20 23:30:00), 0.0 (29/04/20 23:29:00), 0.0 (29/04/20 23:28:00)] (minimum 5 datapoints for OK -> ALARM transition).","StateChangeTime":"2020-04-29T23:33:34.876+0000","Region":"US West (Oregon)","AlarmArn":"test-arn","OldStateValue":"INSUFFICIENT_DATA","Trigger":{"Period":60,"EvaluationPeriods":5,"ComparisonOperator":"LessThanThreshold","Threshold":65.0,"TreatMissingData":"- TreatMissingData:                    missing","EvaluateLowSampleCountPercentile":"","Metrics":[{"Expression":"100*(healthyHostCount/fleetCapacity)","Id":"expr_1","ReturnData":true},{"Id":"healthyHostCount","Label":"HealthyHostCount","MetricStat":{"Metric":{"Dimensions":[{"value":"testTargetGroup","name":"TargetGroup"},{"value":"testLoadBalancer","name":"LoadBalancer"}],"MetricName":"HealthyHostCount","Namespace":"AWS/NetworkELB"},"Period":60,"Stat":"Average"},"ReturnData":false},{"Id":"fleetCapacity","Label":"GroupDesiredCapacity","MetricStat":{"Metric":{"Dimensions":[{"value":"testFleetId","name":"AutoScalingGroup"}],"MetricName":"GroupDesiredCapacity","Namespace":"AWS/AutoScaling"},"Period":60,"Stat":"Average"},"ReturnData":false}]}}';
    (await lambdaCode.handler(successEventSingle, context));
    // THEN
    sinon.assert.notCalled(updateAutoScalingGroupFake);
    // WHEN
    successEventSingle.Records[0].Sns.Message = '{"AlarmName":"testAlarm","AlarmDescription":null,"NewStateValue":"ALARM","NewStateReason":"Threshold Crossed: 5 out of the last 5 datapoints were less than the threshold (65.0). The most recent datapoints which crossed the threshold: [0.0 (29/04/20 23:32:00), 0.0 (29/04/20 23:31:00), 0.0 (29/04/20 23:30:00), 0.0 (29/04/20 23:29:00), 0.0 (29/04/20 23:28:00)] (minimum 5 datapoints for OK -> ALARM transition).","StateChangeTime":"2020-04-29T23:33:34.876+0000","Region":"US West (Oregon)","AlarmArn":"test-arn","OldStateValue":"INSUFFICIENT_DATA","Trigger":{"Period":60,"EvaluationPeriods":5,"ComparisonOperator":"LessThanThreshold","Threshold":65.0,"TreatMissingData":"- TreatMissingData:                    missing","EvaluateLowSampleCountPercentile":"","Metrics":[{"Expression":"100*(healthyHostCount/fleetCapacity)","Id":"expr_1","ReturnData":true},{"Id":"healthyHostCount","Label":"HealthyHostCount","MetricStat":{"Metric":{"Dimensions":[{"value":"testTargetGroup","name":"TargetGroup"},{"value":"testLoadBalancer","name":"LoadBalancer"}],"MetricName":"HealthyHostCount","Namespace":"AWS/NetworkELB"},"Period":60,"Stat":"Average"},"ReturnData":false},{"Id":"fleetCapacity","Label":"GroupDesiredCapacity","MetricStat":{"Metric":{"Dimen":[{"value":"testFleetId2","name":"AutoScalingGroupName"}],"MetricName":"GroupDesiredCapacity","Namespace":"AWS/AutoScaling"},"Period":60,"Stat":"Average"},"ReturnData":false}]}}';
    (await lambdaCode.handler(successEventSingle, context));
    // THEN
    sinon.assert.notCalled(updateAutoScalingGroupFake);
    // WHEN
    successEventSingle.Records[0].Sns.Message = '{"AlarmName":"testAlarm","AlarmDescription":null,"NewStateValue":"ALARM","NewStateReason":"Threshold Crossed: 5 out of the last 5 datapoints were less than the threshold (65.0). The most recent datapoints which crossed the threshold: [0.0 (29/04/20 23:32:00), 0.0 (29/04/20 23:31:00), 0.0 (29/04/20 23:30:00), 0.0 (29/04/20 23:29:00), 0.0 (29/04/20 23:28:00)] (minimum 5 datapoints for OK -> ALARM transition).","StateChangeTime":"2020-04-29T23:33:34.876+0000","Region":"US West (Oregon)","AlarmArn":"test-arn","OldStateValue":"INSUFFICIENT_DATA","Trigger":{"Period":60,"EvaluationPeriods":5,"ComparisonOperator":"LessThanThreshold","Threshold":65.0,"TreatMissingData":"- TreatMissingData:                    missing","EvaluateLowSampleCountPercentile":"","M":[{"Expression":"100*(healthyHostCount/fleetCapacity)","Id":"expr_1","ReturnData":true},{"Id":"healthyHostCount","Label":"HealthyHostCount","MetricStat":{"Metric":{"Dimensions":[{"value":"testTargetGroup","name":"TargetGroup"},{"value":"testLoadBalancer","name":"LoadBalancer"}],"MetricName":"HealthyHostCount","Namespace":"AWS/NetworkELB"},"Period":60,"Stat":"Average"},"ReturnData":false},{"Id":"fleetCapacity","Label":"GroupDesiredCapacity","MetricStat":{"Metric":{"Dimen":[{"value":"testFleetId2","name":"AutoScalingGroupName"}],"MetricName":"GroupDesiredCapacity","Namespace":"AWS/AutoScaling"},"Period":60,"Stat":"Average"},"ReturnData":false}]}}';
    (await lambdaCode.handler(successEventSingle, context));
    // THEN
    sinon.assert.notCalled(updateAutoScalingGroupFake);
    // WHEN
    delete successEventSingle.Records[0].Sns.Message;
    (await lambdaCode.handler(successEventSingle, context));
    // THEN
    sinon.assert.notCalled(updateAutoScalingGroupFake);
    // WHEN
    successEventSingle.Records[0].Sns.Message = '{"AlarmName":"testAlarm","AlarmDescription":null,"NewStateValue":"ALARM","NewStateReason":"Threshold Crossed: 5 out of the last 5 datapoints were less than the threshold (65.0). The most recent datapoints which crossed the threshold: [0.0 (29/04/20 23:32:00), 0.0 (29/04/20 23:31:00), 0.0 (29/04/20 23:30:00), 0.0 (29/04/20 23:29:00), 0.0 (29/04/20 23:28:00)] (minimum 5 datapoints for OK -> ALARM transition).","StateChangeTime":"2020-04-29T23:33:34.876+0000","Region":"US West (Oregon)","AlarmArn":"test-arn","OldStateValue":"INSUFFICIENT_DATA","Trigger":{"Period":60,"EvaluationPeriods":5,"ComparisonOperator":"LessThanThreshold","Threshold":65.0,"TreatMissingData":"- TreatMissingData:                    missing","EvaluateLowSampleCountPercentile":"","Metrics":[{"Expression":"100*(healthyHostCount/fleetCapacity)","Id":"expr_1","ReturnData":true},{"Id":"healthyHostCount","Label":"HealthyHostCount","MetricStat":{"Metric":{"Dimensions":[{"value":"testTargetGroup","name":"TargetGroup"},{"value":"testLoadBalancer","name":"LoadBalancer"}],"MetricName":"HealthyHostCount","Namespace":"AWS/NetworkELB"},"Period":60,"Stat":"Average"},"ReturnData":false},{"Id":"eetCapacity","Label":"GroupDesiredCapacity","MetricStat":{"Metric":{"Dimensions":[{"value":"testFleetId","name":"AutoScalingGroupName"}],"MetricName":"GroupDesiredCapacity","Namespace":"AWS/AutoScaling"},"Period":60,"Stat":"Average"},"ReturnData":false}]}}';
    (await lambdaCode.handler(successEventSingle, context));
    // THEN
    sinon.assert.notCalled(updateAutoScalingGroupFake);
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidW5oZWFsdGh5RmxlZXRBY3Rpb24udGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVuaGVhbHRoeUZsZWV0QWN0aW9uLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7R0FHRzs7QUFFSCwrQkFBK0I7QUFFL0Isb0NBQW9DO0FBQ3BDLCtCQUErQjtBQUMvQiw4RUFBOEU7QUFFOUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUM7QUFFdkMsTUFBTSxXQUFXLEdBQUc7SUFDbEIsT0FBTyxFQUFFLENBQUM7WUFDUixXQUFXLEVBQUUsU0FBUztZQUN0QixZQUFZLEVBQUUsS0FBSztZQUNuQixvQkFBb0IsRUFBRSxVQUFVO1lBQ2hDLEdBQUcsRUFBRTtnQkFDSCxJQUFJLEVBQUUsY0FBYztnQkFDcEIsU0FBUyxFQUFFLHNDQUFzQztnQkFDakQsUUFBUSxFQUFFLFVBQVU7Z0JBQ3BCLE9BQU8sRUFBRSxtQ0FBbUM7Z0JBQzVDLFNBQVMsRUFBRSwwQkFBMEI7Z0JBQ3JDLE9BQU8sRUFBRSwrNENBQSs0QztnQkFDeDVDLGdCQUFnQixFQUFFLEdBQUc7Z0JBQ3JCLFNBQVMsRUFBRSxlQUFlO2dCQUMxQixjQUFjLEVBQUUsb0JBQW9CO2dCQUNwQyxjQUFjLEVBQUUsb0JBQW9CO2dCQUNwQyxpQkFBaUIsRUFBRSxFQUFFO2FBQ3RCO1NBQ0YsQ0FBQztDQUNILENBQUM7QUFFRixNQUFNLE9BQU8sR0FBRztJQUNkLFlBQVksRUFBRSxVQUFVO0NBQ0osQ0FBQztBQUV2QixVQUFVLENBQUMsR0FBRyxFQUFFO0lBQ2QsT0FBTyxDQUFDLEdBQUcsR0FBRyxHQUFHLEVBQUUsR0FBRSxDQUFDLENBQUM7SUFDdkIsT0FBTyxDQUFDLEtBQUssR0FBRyxHQUFHLEVBQUUsR0FBRSxDQUFDLENBQUM7SUFDekIsT0FBTyxDQUFDLElBQUksR0FBRyxHQUFHLEVBQUUsR0FBRSxDQUFDLENBQUM7QUFDMUIsQ0FBQyxDQUFDLENBQUM7QUFFSCxTQUFTLENBQUMsR0FBRyxFQUFFO0lBQ2IsR0FBRyxDQUFDLE9BQU8sRUFBRSxDQUFDO0FBQ2hCLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLCtCQUErQixFQUFFLEtBQUssSUFBSSxFQUFFO0lBQy9DLE9BQU87SUFDUCxNQUFNLDBCQUEwQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzNELEdBQUcsQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLHdCQUF3QixFQUFFLDBCQUEwQixDQUFDLENBQUM7SUFFOUUsTUFBTSxNQUFNLEdBQUcsQ0FBQyxNQUFNLFVBQVUsQ0FBQyxPQUFPLENBQUMsV0FBVyxFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFFaEUsT0FBTztJQUNQLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBRXBDLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLDBCQUEwQixFQUFFO1FBQ2xELG9CQUFvQixFQUFFLGFBQWE7UUFDbkMsT0FBTyxFQUFFLENBQUM7UUFDVixPQUFPLEVBQUUsQ0FBQztRQUNWLGVBQWUsRUFBRSxDQUFDO0tBQ25CLENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDJDQUEyQyxFQUFFLEtBQUssSUFBSSxFQUFFO0lBQzNELE9BQU87SUFDUCxNQUFNLEtBQUssR0FBRyxJQUFJLEtBQUssRUFBMkIsQ0FBQztJQUNuRCxLQUFLLENBQUMsSUFBSSxHQUFHLHVCQUF1QixDQUFDO0lBRXJDLE1BQU0sMEJBQTBCLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUM7SUFDN0QsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsd0JBQXdCLEVBQUUsMEJBQTBCLENBQUMsQ0FBQztJQUU5RSxNQUFNLE1BQU0sR0FBRyxDQUFDLE1BQU0sVUFBVSxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztJQUVoRSxPQUFPO0lBQ1AsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDdkMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUMsa0NBQWtDLENBQUMsQ0FBQztJQUVsRSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQywwQkFBMEIsRUFBRTtRQUNsRCxvQkFBb0IsRUFBRSxhQUFhO1FBQ25DLE9BQU8sRUFBRSxDQUFDO1FBQ1YsT0FBTyxFQUFFLENBQUM7UUFDVixlQUFlLEVBQUUsQ0FBQztLQUNuQixDQUFDLENBQUM7QUFDTCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyx3Q0FBd0MsRUFBRSxLQUFLLElBQUksRUFBRTtJQUN4RCxPQUFPO0lBQ1AsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztJQUNuRSxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLE9BQU8sR0FBRyx1NENBQXU0QyxDQUFDO0lBRXA3QyxNQUFNLDBCQUEwQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzNELEdBQUcsQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLHdCQUF3QixFQUFFLDBCQUEwQixDQUFDLENBQUM7SUFFOUUsTUFBTSxNQUFNLEdBQUcsQ0FBQyxNQUFNLFVBQVUsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztJQUV2RSxPQUFPO0lBQ1AsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7SUFFdkMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsMEJBQTBCLENBQUMsQ0FBQztBQUNyRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQyw4QkFBOEIsRUFBRSxLQUFLLElBQUksRUFBRTtJQUM5QyxPQUFPO0lBQ1AsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztJQUNuRSxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFFM0YsTUFBTSwwQkFBMEIsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUMzRCxHQUFHLENBQUMsSUFBSSxDQUFDLGFBQWEsRUFBRSx3QkFBd0IsRUFBRSwwQkFBMEIsQ0FBQyxDQUFDO0lBRTlFLE1BQU0sTUFBTSxHQUFHLENBQUMsTUFBTSxVQUFVLENBQUMsT0FBTyxDQUFDLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFFdkUsT0FBTztJQUNQLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDO0lBQ3ZDLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsT0FBTyxDQUFDLHdDQUF3QyxDQUFDLENBQUM7SUFFeEUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsMEJBQTBCLENBQUMsQ0FBQztBQUNyRCxDQUFDLENBQUMsQ0FBQztBQUVILElBQUksQ0FBQywwQ0FBMEMsRUFBRSxLQUFLLElBQUksRUFBRTtJQUMxRCxPQUFPO0lBQ1AsTUFBTSxrQkFBa0IsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQztJQUNuRSxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLE9BQU8sR0FBRywwekNBQTB6QyxDQUFDO0lBRXYyQyxNQUFNLDBCQUEwQixHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQzNELEdBQUcsQ0FBQyxJQUFJLENBQUMsYUFBYSxFQUFFLHdCQUF3QixFQUFFLDBCQUEwQixDQUFDLENBQUM7SUFFOUUsTUFBTSxNQUFNLEdBQUcsQ0FBQyxNQUFNLFVBQVUsQ0FBQyxPQUFPLENBQUMsa0JBQWtCLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztJQUV2RSxPQUFPO0lBQ1AsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUM7SUFDdkMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxPQUFPLENBQUMsMERBQTBELENBQUMsQ0FBQztJQUUxRixLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0FBQ3JELENBQUMsQ0FBQyxDQUFDO0FBRUgsSUFBSSxDQUFDLDREQUE0RCxFQUFFLEtBQUssSUFBSSxFQUFFO0lBQzVFLE9BQU87SUFDUCxNQUFNLGtCQUFrQixHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFNBQVMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDO0lBRW5FLE1BQU0sMEJBQTBCLEdBQUcsS0FBSyxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDM0QsR0FBRyxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUUsd0JBQXdCLEVBQUUsMEJBQTBCLENBQUMsQ0FBQztJQUU5RSxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLE9BQU8sR0FBRywyNENBQTI0QyxDQUFDO0lBQ3g3QyxDQUFDLE1BQU0sVUFBVSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBRXhELE9BQU87SUFDUCxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBRW5ELE9BQU87SUFDUCxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLE9BQU8sR0FBRywyNENBQTI0QyxDQUFDO0lBRXg3QyxDQUFDLE1BQU0sVUFBVSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBRXhELE9BQU87SUFDUCxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBRW5ELE9BQU87SUFDUCxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLE9BQU8sR0FBRyxxNENBQXE0QyxDQUFDO0lBQ2w3QyxDQUFDLE1BQU0sVUFBVSxDQUFDLE9BQU8sQ0FBQyxrQkFBa0IsRUFBRSxPQUFPLENBQUMsQ0FBQyxDQUFDO0lBRXhELE9BQU87SUFDUCxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQywwQkFBMEIsQ0FBQyxDQUFDO0lBRW5ELE9BQU87SUFDUCxPQUFPLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDO0lBQ2pELENBQUMsTUFBTSxVQUFVLENBQUMsT0FBTyxDQUFDLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFFeEQsT0FBTztJQUNQLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLDBCQUEwQixDQUFDLENBQUM7SUFFbkQsT0FBTztJQUNQLGtCQUFrQixDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsT0FBTyxHQUFHLDY0Q0FBNjRDLENBQUM7SUFFMTdDLENBQUMsTUFBTSxVQUFVLENBQUMsT0FBTyxDQUFDLGtCQUFrQixFQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUM7SUFFeEQsT0FBTztJQUNQLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLDBCQUEwQixDQUFDLENBQUM7QUFDckQsQ0FBQyxDQUFDLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICogU1BEWC1MaWNlbnNlLUlkZW50aWZpZXI6IEFwYWNoZS0yLjBcbiAqL1xuXG4vKiBlc2xpbnQtZGlzYWJsZSBuby1jb25zb2xlICovXG5cbmltcG9ydCAqIGFzIEFXUyBmcm9tICdhd3Mtc2RrLW1vY2snO1xuaW1wb3J0ICogYXMgc2lub24gZnJvbSAnc2lub24nO1xuaW1wb3J0ICogYXMgbGFtYmRhQ29kZSBmcm9tICcuLi8uLi9sYW1iZGFzL25vZGVqcy91bmhlYWx0aHlGbGVldEFjdGlvbi9pbmRleCc7XG5cbkFXUy5zZXRTREsocmVxdWlyZS5yZXNvbHZlKCdhd3Mtc2RrJykpO1xuXG5jb25zdCBzYW1wbGVFdmVudCA9IHtcbiAgUmVjb3JkczogW3tcbiAgICBFdmVudFNvdXJjZTogJ2F3czpzbnMnLFxuICAgIEV2ZW50VmVyc2lvbjogJzEuMCcsXG4gICAgRXZlbnRTdWJzY3JpcHRpb25Bcm46ICdhcm4tdGVzdCcsXG4gICAgU25zOiB7XG4gICAgICBUeXBlOiAnTm90aWZpY2F0aW9uJyxcbiAgICAgIE1lc3NhZ2VJZDogJzhiMTVlMDgzLTMwYTMtNWE1Ny1hMTdlLWI3YTcyMWFkMzY1ZCcsXG4gICAgICBUb3BpY0FybjogJ2Fybi10ZXN0JyxcbiAgICAgIFN1YmplY3Q6ICdBTEFSTTogXCJ0ZXN0XCIgaW4gVVMgV2VzdCAoT3JlZ29uKScsXG4gICAgICBUaW1lc3RhbXA6ICcyMDIwLTA0LTI5VDIzOjMzOjM0LjkyOFonLFxuICAgICAgTWVzc2FnZTogJ3tcIkFsYXJtTmFtZVwiOlwidGVzdEFsYXJtXCIsXCJBbGFybURlc2NyaXB0aW9uXCI6bnVsbCxcIk5ld1N0YXRlVmFsdWVcIjpcIkFMQVJNXCIsXCJOZXdTdGF0ZVJlYXNvblwiOlwiVGhyZXNob2xkIENyb3NzZWQ6IDUgb3V0IG9mIHRoZSBsYXN0IDUgZGF0YXBvaW50cyB3ZXJlIGxlc3MgdGhhbiB0aGUgdGhyZXNob2xkICg2NS4wKS4gVGhlIG1vc3QgcmVjZW50IGRhdGFwb2ludHMgd2hpY2ggY3Jvc3NlZCB0aGUgdGhyZXNob2xkOiBbMC4wICgyOS8wNC8yMCAyMzozMjowMCksIDAuMCAoMjkvMDQvMjAgMjM6MzE6MDApLCAwLjAgKDI5LzA0LzIwIDIzOjMwOjAwKSwgMC4wICgyOS8wNC8yMCAyMzoyOTowMCksIDAuMCAoMjkvMDQvMjAgMjM6Mjg6MDApXSAobWluaW11bSA1IGRhdGFwb2ludHMgZm9yIE9LIC0+IEFMQVJNIHRyYW5zaXRpb24pLlwiLFwiU3RhdGVDaGFuZ2VUaW1lXCI6XCIyMDIwLTA0LTI5VDIzOjMzOjM0Ljg3NiswMDAwXCIsXCJSZWdpb25cIjpcIlVTIFdlc3QgKE9yZWdvbilcIixcIkFsYXJtQXJuXCI6XCJ0ZXN0LWFyblwiLFwiT2xkU3RhdGVWYWx1ZVwiOlwiSU5TVUZGSUNJRU5UX0RBVEFcIixcIlRyaWdnZXJcIjp7XCJQZXJpb2RcIjo2MCxcIkV2YWx1YXRpb25QZXJpb2RzXCI6NSxcIkNvbXBhcmlzb25PcGVyYXRvclwiOlwiTGVzc1RoYW5UaHJlc2hvbGRcIixcIlRocmVzaG9sZFwiOjY1LjAsXCJUcmVhdE1pc3NpbmdEYXRhXCI6XCItIFRyZWF0TWlzc2luZ0RhdGE6ICAgICAgICAgICAgICAgICAgICBtaXNzaW5nXCIsXCJFdmFsdWF0ZUxvd1NhbXBsZUNvdW50UGVyY2VudGlsZVwiOlwiXCIsXCJNZXRyaWNzXCI6W3tcIkV4cHJlc3Npb25cIjpcIjEwMCooaGVhbHRoeUhvc3RDb3VudC9mbGVldENhcGFjaXR5KVwiLFwiSWRcIjpcImV4cHJfMVwiLFwiUmV0dXJuRGF0YVwiOnRydWV9LHtcIklkXCI6XCJoZWFsdGh5SG9zdENvdW50XCIsXCJMYWJlbFwiOlwiSGVhbHRoeUhvc3RDb3VudFwiLFwiTWV0cmljU3RhdFwiOntcIk1ldHJpY1wiOntcIkRpbWVuc2lvbnNcIjpbe1widmFsdWVcIjpcInRlc3RUYXJnZXRHcm91cFwiLFwibmFtZVwiOlwiVGFyZ2V0R3JvdXBcIn0se1widmFsdWVcIjpcInRlc3RMb2FkQmFsYW5jZXJcIixcIm5hbWVcIjpcIkxvYWRCYWxhbmNlclwifV0sXCJNZXRyaWNOYW1lXCI6XCJIZWFsdGh5SG9zdENvdW50XCIsXCJOYW1lc3BhY2VcIjpcIkFXUy9OZXR3b3JrRUxCXCJ9LFwiUGVyaW9kXCI6NjAsXCJTdGF0XCI6XCJBdmVyYWdlXCJ9LFwiUmV0dXJuRGF0YVwiOmZhbHNlfSx7XCJJZFwiOlwiZmxlZXRDYXBhY2l0eVwiLFwiTGFiZWxcIjpcIkdyb3VwRGVzaXJlZENhcGFjaXR5XCIsXCJNZXRyaWNTdGF0XCI6e1wiTWV0cmljXCI6e1wiRGltZW5zaW9uc1wiOlt7XCJ2YWx1ZVwiOlwidGVzdEZsZWV0SWRcIixcIm5hbWVcIjpcIkF1dG9TY2FsaW5nR3JvdXBOYW1lXCJ9XSxcIk1ldHJpY05hbWVcIjpcIkdyb3VwRGVzaXJlZENhcGFjaXR5XCIsXCJOYW1lc3BhY2VcIjpcIkFXUy9BdXRvU2NhbGluZ1wifSxcIlBlcmlvZFwiOjYwLFwiU3RhdFwiOlwiQXZlcmFnZVwifSxcIlJldHVybkRhdGFcIjpmYWxzZX1dfX0nLFxuICAgICAgU2lnbmF0dXJlVmVyc2lvbjogJzEnLFxuICAgICAgU2lnbmF0dXJlOiAndGVzdFNpZ25hdHVyZScsXG4gICAgICBTaWduaW5nQ2VydFVybDogJ3Rlc3RTaWduaW5nQ2VydFVybCcsXG4gICAgICBVbnN1YnNjcmliZVVybDogJ3Rlc3RVbnN1YnNjcmliZVVSTCcsXG4gICAgICBNZXNzYWdlQXR0cmlidXRlczoge30sXG4gICAgfSxcbiAgfV0sXG59O1xuXG5jb25zdCBjb250ZXh0ID0ge1xuICBmdW5jdGlvbk5hbWU6ICdwcm92aWRlcicsXG59IGFzIEFXU0xhbWJkYS5Db250ZXh0O1xuXG5iZWZvcmVFYWNoKCgpID0+IHtcbiAgY29uc29sZS5sb2cgPSAoKSA9PiB7fTtcbiAgY29uc29sZS5lcnJvciA9ICgpID0+IHt9O1xuICBjb25zb2xlLmluZm8gPSAoKSA9PiB7fTtcbn0pO1xuXG5hZnRlckVhY2goKCkgPT4ge1xuICBBV1MucmVzdG9yZSgpO1xufSk7XG5cbnRlc3QoJ3N1Y2Nlc3Mgc2NlbmFyaW8gc2luZ2xlIGZsZWV0JywgYXN5bmMgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHVwZGF0ZUF1dG9TY2FsaW5nR3JvdXBGYWtlID0gc2lub24uZmFrZS5yZXNvbHZlcyh7fSk7XG4gIEFXUy5tb2NrKCdBdXRvU2NhbGluZycsICd1cGRhdGVBdXRvU2NhbGluZ0dyb3VwJywgdXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xuXG4gIGNvbnN0IHJlc3VsdCA9IChhd2FpdCBsYW1iZGFDb2RlLmhhbmRsZXIoc2FtcGxlRXZlbnQsIGNvbnRleHQpKTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChyZXN1bHQuc3RhdHVzKS50b0VxdWFsKCdPSycpO1xuXG4gIHNpbm9uLmFzc2VydC5jYWxsZWRXaXRoKHVwZGF0ZUF1dG9TY2FsaW5nR3JvdXBGYWtlLCB7XG4gICAgQXV0b1NjYWxpbmdHcm91cE5hbWU6ICd0ZXN0RmxlZXRJZCcsXG4gICAgTWF4U2l6ZTogMCxcbiAgICBNaW5TaXplOiAwLFxuICAgIERlc2lyZWRDYXBhY2l0eTogMCxcbiAgfSk7XG59KTtcblxudGVzdCgnZmFpbHVyZSBzY2VuYXJpbywgQVdTIGFwaSByZXR1cm5zIGZhaWx1cmUnLCBhc3luYyAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3QgZXJyb3IgPSBuZXcgRXJyb3IoKSBhcyBOb2RlSlMuRXJybm9FeGNlcHRpb247XG4gIGVycm9yLmNvZGUgPSAnQWNjZXNzRGVuaWVkRXhjZXB0aW9uJztcblxuICBjb25zdCB1cGRhdGVBdXRvU2NhbGluZ0dyb3VwRmFrZSA9IHNpbm9uLmZha2UucmVqZWN0cyhlcnJvcik7XG4gIEFXUy5tb2NrKCdBdXRvU2NhbGluZycsICd1cGRhdGVBdXRvU2NhbGluZ0dyb3VwJywgdXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xuXG4gIGNvbnN0IHJlc3VsdCA9IChhd2FpdCBsYW1iZGFDb2RlLmhhbmRsZXIoc2FtcGxlRXZlbnQsIGNvbnRleHQpKTtcblxuICAvLyBUSEVOXG4gIGV4cGVjdChyZXN1bHQuc3RhdHVzKS50b0VxdWFsKCdFUlJPUicpO1xuICBleHBlY3QocmVzdWx0LnJlYXNvbikudG9NYXRjaCgvRXhjZXB0aW9uIHdoaWxlIHN1c3BlbmRpbmcgZmxlZXQvKTtcblxuICBzaW5vbi5hc3NlcnQuY2FsbGVkV2l0aCh1cGRhdGVBdXRvU2NhbGluZ0dyb3VwRmFrZSwge1xuICAgIEF1dG9TY2FsaW5nR3JvdXBOYW1lOiAndGVzdEZsZWV0SWQnLFxuICAgIE1heFNpemU6IDAsXG4gICAgTWluU2l6ZTogMCxcbiAgICBEZXNpcmVkQ2FwYWNpdHk6IDAsXG4gIH0pO1xufSk7XG5cbnRlc3QoJ2ZhaWx1cmUgc2NlbmFyaW8sIE1ldHJpY1N0YXQgbm90IGZvdW5kJywgYXN5bmMgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHN1Y2Nlc3NFdmVudFNpbmdsZSA9IEpTT04ucGFyc2UoSlNPTi5zdHJpbmdpZnkoc2FtcGxlRXZlbnQpKTtcbiAgc3VjY2Vzc0V2ZW50U2luZ2xlLlJlY29yZHNbMF0uU25zLk1lc3NhZ2UgPSAne1wiQWxhcm1OYW1lXCI6XCJ0ZXN0QWxhcm1cIixcIkFsYXJtRGVzY3JpcHRpb25cIjpudWxsLFwiTmV3U3RhdGVWYWx1ZVwiOlwiQUxBUk1cIixcIk5ld1N0YXRlUmVhc29uXCI6XCJUaHJlc2hvbGQgQ3Jvc3NlZDogNSBvdXQgb2YgdGhlIGxhc3QgNSBkYXRhcG9pbnRzIHdlcmUgbGVzcyB0aGFuIHRoZSB0aHJlc2hvbGQgKDY1LjApLiBUaGUgbW9zdCByZWNlbnQgZGF0YXBvaW50cyB3aGljaCBjcm9zc2VkIHRoZSB0aHJlc2hvbGQ6IFswLjAgKDI5LzA0LzIwIDIzOjMyOjAwKSwgMC4wICgyOS8wNC8yMCAyMzozMTowMCksIDAuMCAoMjkvMDQvMjAgMjM6MzA6MDApLCAwLjAgKDI5LzA0LzIwIDIzOjI5OjAwKSwgMC4wICgyOS8wNC8yMCAyMzoyODowMCldIChtaW5pbXVtIDUgZGF0YXBvaW50cyBmb3IgT0sgLT4gQUxBUk0gdHJhbnNpdGlvbikuXCIsXCJTdGF0ZUNoYW5nZVRpbWVcIjpcIjIwMjAtMDQtMjlUMjM6MzM6MzQuODc2KzAwMDBcIixcIlJlZ2lvblwiOlwiVVMgV2VzdCAoT3JlZ29uKVwiLFwiQWxhcm1Bcm5cIjpcInRlc3QtYXJuXCIsXCJPbGRTdGF0ZVZhbHVlXCI6XCJJTlNVRkZJQ0lFTlRfREFUQVwiLFwiVHJpZ2dlclwiOntcIlBlcmlvZFwiOjYwLFwiRXZhbHVhdGlvblBlcmlvZHNcIjo1LFwiQ29tcGFyaXNvbk9wZXJhdG9yXCI6XCJMZXNzVGhhblRocmVzaG9sZFwiLFwiVGhyZXNob2xkXCI6NjUuMCxcIlRyZWF0TWlzc2luZ0RhdGFcIjpcIi0gVHJlYXRNaXNzaW5nRGF0YTogICAgICAgICAgICAgICAgICAgIG1pc3NpbmdcIixcIkV2YWx1YXRlTG93U2FtcGxlQ291bnRQZXJjZW50aWxlXCI6XCJcIixcIk1ldHJpY3NcIjpbe1wiRXhwcmVzc2lvblwiOlwiMTAwKihoZWFsdGh5SG9zdENvdW50L2ZsZWV0Q2FwYWNpdHkpXCIsXCJJZFwiOlwiZXhwcl8xXCIsXCJSZXR1cm5EYXRhXCI6dHJ1ZX0se1wiSWRcIjpcImhlYWx0aHlIb3N0Q291bnRcIixcIkxhYmVsXCI6XCJIZWFsdGh5SG9zdENvdW50XCIsXCJNZXRyaWNTdGF0XCI6e1wiTWV0cmljXCI6e1wiRGltZW5zaW9uc1wiOlt7XCJ2YWx1ZVwiOlwidGVzdFRhcmdldEdyb3VwXCIsXCJuYW1lXCI6XCJUYXJnZXRHcm91cFwifSx7XCJ2YWx1ZVwiOlwidGVzdExvYWRCYWxhbmNlclwiLFwibmFtZVwiOlwiTG9hZEJhbGFuY2VyXCJ9XSxcIk1ldHJpY05hbWVcIjpcIkhlYWx0aHlIb3N0Q291bnRcIixcIk5hbWVzcGFjZVwiOlwiQVdTL05ldHdvcmtFTEJcIn0sXCJQZXJpb2RcIjo2MCxcIlN0YXRcIjpcIkF2ZXJhZ2VcIn0sXCJSZXR1cm5EYXRhXCI6ZmFsc2V9LHtcIklkXCI6XCJmbGVldENhcGFjaXR5XCIsXCJMYWJlbFwiOlwiR3JvdXBEZXNpcmVkQ2FwYWNpdHlcIixcIk1cIjp7XCJNZXRyaWNcIjp7XCJEaW1lbnNpb25zXCI6W3tcInZhbHVlXCI6XCJ0ZXN0RmxlZXRJZDJcIixcIm5hbWVcIjpcIkF1dG9TY2FsaW5nR3JvdXBOYW1lXCJ9XSxcIk1ldHJpY05hbWVcIjpcIkdyb3VwRGVzaXJlZENhcGFjaXR5XCIsXCJOYW1lc3BhY2VcIjpcIkFXUy9BdXRvU2NhbGluZ1wifSxcIlBlcmlvZFwiOjYwLFwiU3RhdFwiOlwiQXZlcmFnZVwifSxcIlJldHVybkRhdGFcIjpmYWxzZX1dfX0nO1xuXG4gIGNvbnN0IHVwZGF0ZUF1dG9TY2FsaW5nR3JvdXBGYWtlID0gc2lub24uZmFrZS5yZXNvbHZlcyh7fSk7XG4gIEFXUy5tb2NrKCdBdXRvU2NhbGluZycsICd1cGRhdGVBdXRvU2NhbGluZ0dyb3VwJywgdXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xuXG4gIGNvbnN0IHJlc3VsdCA9IChhd2FpdCBsYW1iZGFDb2RlLmhhbmRsZXIoc3VjY2Vzc0V2ZW50U2luZ2xlLCBjb250ZXh0KSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QocmVzdWx0LnN0YXR1cykudG9FcXVhbCgnRVJST1InKTtcblxuICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKHVwZGF0ZUF1dG9TY2FsaW5nR3JvdXBGYWtlKTtcbn0pO1xuXG50ZXN0KCdFcnJvciBpZiAyIHJlY29yZHMgYXJlIGZvdW5kJywgYXN5bmMgKCkgPT4ge1xuICAvLyBXSEVOXG4gIGNvbnN0IHN1Y2Nlc3NFdmVudFNpbmdsZSA9IEpTT04ucGFyc2UoSlNPTi5zdHJpbmdpZnkoc2FtcGxlRXZlbnQpKTtcbiAgc3VjY2Vzc0V2ZW50U2luZ2xlLlJlY29yZHMucHVzaChKU09OLnBhcnNlKEpTT04uc3RyaW5naWZ5KHN1Y2Nlc3NFdmVudFNpbmdsZS5SZWNvcmRzWzBdKSkpO1xuXG4gIGNvbnN0IHVwZGF0ZUF1dG9TY2FsaW5nR3JvdXBGYWtlID0gc2lub24uZmFrZS5yZXNvbHZlcyh7fSk7XG4gIEFXUy5tb2NrKCdBdXRvU2NhbGluZycsICd1cGRhdGVBdXRvU2NhbGluZ0dyb3VwJywgdXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xuXG4gIGNvbnN0IHJlc3VsdCA9IChhd2FpdCBsYW1iZGFDb2RlLmhhbmRsZXIoc3VjY2Vzc0V2ZW50U2luZ2xlLCBjb250ZXh0KSk7XG5cbiAgLy8gVEhFTlxuICBleHBlY3QocmVzdWx0LnN0YXR1cykudG9FcXVhbCgnRVJST1InKTtcbiAgZXhwZWN0KHJlc3VsdC5yZWFzb24pLnRvTWF0Y2goL0V4cGVjdGluZyBhIHNpbmdsZSByZWNvcmQgaW4gU05TIEV2ZW50Lyk7XG5cbiAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZCh1cGRhdGVBdXRvU2NhbGluZ0dyb3VwRmFrZSk7XG59KTtcblxudGVzdCgnRXJyb3IgaWYgZXhhY3RseSAzIG1ldHJpY3MgYXJlIG5vdCBmb3VuZCcsIGFzeW5jICgpID0+IHtcbiAgLy8gV0hFTlxuICBjb25zdCBzdWNjZXNzRXZlbnRTaW5nbGUgPSBKU09OLnBhcnNlKEpTT04uc3RyaW5naWZ5KHNhbXBsZUV2ZW50KSk7XG4gIHN1Y2Nlc3NFdmVudFNpbmdsZS5SZWNvcmRzWzBdLlNucy5NZXNzYWdlID0gJ3tcIkFsYXJtTmFtZVwiOlwidGVzdEFsYXJtXCIsXCJBbGFybURlc2NyaXB0aW9uXCI6bnVsbCxcIk5ld1N0YXRlVmFsdWVcIjpcIkFMQVJNXCIsXCJOZXdTdGF0ZVJlYXNvblwiOlwiVGhyZXNob2xkIENyb3NzZWQ6IDUgb3V0IG9mIHRoZSBsYXN0IDUgZGF0YXBvaW50cyB3ZXJlIGxlc3MgdGhhbiB0aGUgdGhyZXNob2xkICg2NS4wKS4gVGhlIG1vc3QgcmVjZW50IGRhdGFwb2ludHMgd2hpY2ggY3Jvc3NlZCB0aGUgdGhyZXNob2xkOiBbMC4wICgyOS8wNC8yMCAyMzozMjowMCksIDAuMCAoMjkvMDQvMjAgMjM6MzE6MDApLCAwLjAgKDI5LzA0LzIwIDIzOjMwOjAwKSwgMC4wICgyOS8wNC8yMCAyMzoyOTowMCksIDAuMCAoMjkvMDQvMjAgMjM6Mjg6MDApXSAobWluaW11bSA1IGRhdGFwb2ludHMgZm9yIE9LIC0+IEFMQVJNIHRyYW5zaXRpb24pLlwiLFwiU3RhdGVDaGFuZ2VUaW1lXCI6XCIyMDIwLTA0LTI5VDIzOjMzOjM0Ljg3NiswMDAwXCIsXCJSZWdpb25cIjpcIlVTIFdlc3QgKE9yZWdvbilcIixcIkFsYXJtQXJuXCI6XCJ0ZXN0LWFyblwiLFwiT2xkU3RhdGVWYWx1ZVwiOlwiSU5TVUZGSUNJRU5UX0RBVEFcIixcIlRyaWdnZXJcIjp7XCJQZXJpb2RcIjo2MCxcIkV2YWx1YXRpb25QZXJpb2RzXCI6NSxcIkNvbXBhcmlzb25PcGVyYXRvclwiOlwiTGVzc1RoYW5UaHJlc2hvbGRcIixcIlRocmVzaG9sZFwiOjY1LjAsXCJUcmVhdE1pc3NpbmdEYXRhXCI6XCItIFRyZWF0TWlzc2luZ0RhdGE6ICAgICAgICAgICAgICAgICAgICBtaXNzaW5nXCIsXCJFdmFsdWF0ZUxvd1NhbXBsZUNvdW50UGVyY2VudGlsZVwiOlwiXCIsXCJNZXRyaWNzXCI6W3tcIklkXCI6XCJoZWFsdGh5SG9zdENvdW50XCIsXCJMYWJlbFwiOlwiSGVhbHRoeUhvc3RDb3VudFwiLFwiTWV0cmljU3RhdFwiOntcIk1ldHJpY1wiOntcIkRpbWVuc2lvbnNcIjpbe1widmFsdWVcIjpcInRlc3RUYXJnZXRHcm91cFwiLFwibmFtZVwiOlwiVGFyZ2V0R3JvdXBcIn0se1widmFsdWVcIjpcInRlc3RMb2FkQmFsYW5jZXJcIixcIm5hbWVcIjpcIkxvYWRCYWxhbmNlclwifV0sXCJNZXRyaWNOYW1lXCI6XCJIZWFsdGh5SG9zdENvdW50XCIsXCJOYW1lc3BhY2VcIjpcIkFXUy9OZXR3b3JrRUxCXCJ9LFwiUGVyaW9kXCI6NjAsXCJTdGF0XCI6XCJBdmVyYWdlXCJ9LFwiUmV0dXJuRGF0YVwiOmZhbHNlfSx7XCJJZFwiOlwiZmxlZXRDYXBhY2l0eVwiLFwiTGFiZWxcIjpcIkdyb3VwRGVzaXJlZENhcGFjaXR5XCIsXCJNZXRyaWNTdGF0XCI6e1wiTWV0cmljXCI6e1wiRGltZW5zaW9uc1wiOlt7XCJ2YWx1ZVwiOlwidGVzdEZsZWV0SWQyXCIsXCJuYW1lXCI6XCJBdXRvU2NhbGluZ0dyb3VwTmFtZVwifV0sXCJNZXRyaWNOYW1lXCI6XCJHcm91cERlc2lyZWRDYXBhY2l0eVwiLFwiTmFtZXNwYWNlXCI6XCJBV1MvQXV0b1NjYWxpbmdcIn0sXCJQZXJpb2RcIjo2MCxcIlN0YXRcIjpcIkF2ZXJhZ2VcIn0sXCJSZXR1cm5EYXRhXCI6ZmFsc2V9XX19JztcblxuICBjb25zdCB1cGRhdGVBdXRvU2NhbGluZ0dyb3VwRmFrZSA9IHNpbm9uLmZha2UucmVzb2x2ZXMoe30pO1xuICBBV1MubW9jaygnQXV0b1NjYWxpbmcnLCAndXBkYXRlQXV0b1NjYWxpbmdHcm91cCcsIHVwZGF0ZUF1dG9TY2FsaW5nR3JvdXBGYWtlKTtcblxuICBjb25zdCByZXN1bHQgPSAoYXdhaXQgbGFtYmRhQ29kZS5oYW5kbGVyKHN1Y2Nlc3NFdmVudFNpbmdsZSwgY29udGV4dCkpO1xuXG4gIC8vIFRIRU5cbiAgZXhwZWN0KHJlc3VsdC5zdGF0dXMpLnRvRXF1YWwoJ0VSUk9SJyk7XG4gIGV4cGVjdChyZXN1bHQucmVhc29uKS50b01hdGNoKC9FeGFjdGx5IDMgbWV0cmljcyBzaG91bGQgYmUgcHJlc2VudCBpbiB0aGUgYWxhcm0gbWVzc2FnZS8pO1xuXG4gIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQodXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xufSk7XG5cbnRlc3QoJ2ZhaWx1cmUgc2NlbmFyaW8sIGluY29ycmVjdCBkaW1lbnNpb24sIG1ldHJpY3MgYW5kIG1lc3NhZ2UnLCBhc3luYyAoKSA9PiB7XG4gIC8vIFdIRU5cbiAgY29uc3Qgc3VjY2Vzc0V2ZW50U2luZ2xlID0gSlNPTi5wYXJzZShKU09OLnN0cmluZ2lmeShzYW1wbGVFdmVudCkpO1xuXG4gIGNvbnN0IHVwZGF0ZUF1dG9TY2FsaW5nR3JvdXBGYWtlID0gc2lub24uZmFrZS5yZXNvbHZlcyh7fSk7XG4gIEFXUy5tb2NrKCdBdXRvU2NhbGluZycsICd1cGRhdGVBdXRvU2NhbGluZ0dyb3VwJywgdXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xuXG4gIHN1Y2Nlc3NFdmVudFNpbmdsZS5SZWNvcmRzWzBdLlNucy5NZXNzYWdlID0gJ3tcIkFsYXJtTmFtZVwiOlwidGVzdEFsYXJtXCIsXCJBbGFybURlc2NyaXB0aW9uXCI6bnVsbCxcIk5ld1N0YXRlVmFsdWVcIjpcIkFMQVJNXCIsXCJOZXdTdGF0ZVJlYXNvblwiOlwiVGhyZXNob2xkIENyb3NzZWQ6IDUgb3V0IG9mIHRoZSBsYXN0IDUgZGF0YXBvaW50cyB3ZXJlIGxlc3MgdGhhbiB0aGUgdGhyZXNob2xkICg2NS4wKS4gVGhlIG1vc3QgcmVjZW50IGRhdGFwb2ludHMgd2hpY2ggY3Jvc3NlZCB0aGUgdGhyZXNob2xkOiBbMC4wICgyOS8wNC8yMCAyMzozMjowMCksIDAuMCAoMjkvMDQvMjAgMjM6MzE6MDApLCAwLjAgKDI5LzA0LzIwIDIzOjMwOjAwKSwgMC4wICgyOS8wNC8yMCAyMzoyOTowMCksIDAuMCAoMjkvMDQvMjAgMjM6Mjg6MDApXSAobWluaW11bSA1IGRhdGFwb2ludHMgZm9yIE9LIC0+IEFMQVJNIHRyYW5zaXRpb24pLlwiLFwiU3RhdGVDaGFuZ2VUaW1lXCI6XCIyMDIwLTA0LTI5VDIzOjMzOjM0Ljg3NiswMDAwXCIsXCJSZWdpb25cIjpcIlVTIFdlc3QgKE9yZWdvbilcIixcIkFsYXJtQXJuXCI6XCJ0ZXN0LWFyblwiLFwiT2xkU3RhdGVWYWx1ZVwiOlwiSU5TVUZGSUNJRU5UX0RBVEFcIixcIlRyaWdnZXJcIjp7XCJQZXJpb2RcIjo2MCxcIkV2YWx1YXRpb25QZXJpb2RzXCI6NSxcIkNvbXBhcmlzb25PcGVyYXRvclwiOlwiTGVzc1RoYW5UaHJlc2hvbGRcIixcIlRocmVzaG9sZFwiOjY1LjAsXCJUcmVhdE1pc3NpbmdEYXRhXCI6XCItIFRyZWF0TWlzc2luZ0RhdGE6ICAgICAgICAgICAgICAgICAgICBtaXNzaW5nXCIsXCJFdmFsdWF0ZUxvd1NhbXBsZUNvdW50UGVyY2VudGlsZVwiOlwiXCIsXCJNZXRyaWNzXCI6W3tcIkV4cHJlc3Npb25cIjpcIjEwMCooaGVhbHRoeUhvc3RDb3VudC9mbGVldENhcGFjaXR5KVwiLFwiSWRcIjpcImV4cHJfMVwiLFwiUmV0dXJuRGF0YVwiOnRydWV9LHtcIklkXCI6XCJoZWFsdGh5SG9zdENvdW50XCIsXCJMYWJlbFwiOlwiSGVhbHRoeUhvc3RDb3VudFwiLFwiTWV0cmljU3RhdFwiOntcIk1ldHJpY1wiOntcIkRpbWVuc2lvbnNcIjpbe1widmFsdWVcIjpcInRlc3RUYXJnZXRHcm91cFwiLFwibmFtZVwiOlwiVGFyZ2V0R3JvdXBcIn0se1widmFsdWVcIjpcInRlc3RMb2FkQmFsYW5jZXJcIixcIm5hbWVcIjpcIkxvYWRCYWxhbmNlclwifV0sXCJNZXRyaWNOYW1lXCI6XCJIZWFsdGh5SG9zdENvdW50XCIsXCJOYW1lc3BhY2VcIjpcIkFXUy9OZXR3b3JrRUxCXCJ9LFwiUGVyaW9kXCI6NjAsXCJTdGF0XCI6XCJBdmVyYWdlXCJ9LFwiUmV0dXJuRGF0YVwiOmZhbHNlfSx7XCJJZFwiOlwiZmxlZXRDYXBhY2l0eVwiLFwiTGFiZWxcIjpcIkdyb3VwRGVzaXJlZENhcGFjaXR5XCIsXCJNZXRyaWNTdGF0XCI6e1wiTWV0cmljXCI6e1wiRGltZW5zaW9uc1wiOlt7XCJ2YWx1ZVwiOlwidGVzdEZsZWV0SWRcIixcIm5hbWVcIjpcIkF1dG9TY2FsaW5nR3JvdXBcIn1dLFwiTWV0cmljTmFtZVwiOlwiR3JvdXBEZXNpcmVkQ2FwYWNpdHlcIixcIk5hbWVzcGFjZVwiOlwiQVdTL0F1dG9TY2FsaW5nXCJ9LFwiUGVyaW9kXCI6NjAsXCJTdGF0XCI6XCJBdmVyYWdlXCJ9LFwiUmV0dXJuRGF0YVwiOmZhbHNlfV19fSc7XG4gIChhd2FpdCBsYW1iZGFDb2RlLmhhbmRsZXIoc3VjY2Vzc0V2ZW50U2luZ2xlLCBjb250ZXh0KSk7XG5cbiAgLy8gVEhFTlxuICBzaW5vbi5hc3NlcnQubm90Q2FsbGVkKHVwZGF0ZUF1dG9TY2FsaW5nR3JvdXBGYWtlKTtcblxuICAvLyBXSEVOXG4gIHN1Y2Nlc3NFdmVudFNpbmdsZS5SZWNvcmRzWzBdLlNucy5NZXNzYWdlID0gJ3tcIkFsYXJtTmFtZVwiOlwidGVzdEFsYXJtXCIsXCJBbGFybURlc2NyaXB0aW9uXCI6bnVsbCxcIk5ld1N0YXRlVmFsdWVcIjpcIkFMQVJNXCIsXCJOZXdTdGF0ZVJlYXNvblwiOlwiVGhyZXNob2xkIENyb3NzZWQ6IDUgb3V0IG9mIHRoZSBsYXN0IDUgZGF0YXBvaW50cyB3ZXJlIGxlc3MgdGhhbiB0aGUgdGhyZXNob2xkICg2NS4wKS4gVGhlIG1vc3QgcmVjZW50IGRhdGFwb2ludHMgd2hpY2ggY3Jvc3NlZCB0aGUgdGhyZXNob2xkOiBbMC4wICgyOS8wNC8yMCAyMzozMjowMCksIDAuMCAoMjkvMDQvMjAgMjM6MzE6MDApLCAwLjAgKDI5LzA0LzIwIDIzOjMwOjAwKSwgMC4wICgyOS8wNC8yMCAyMzoyOTowMCksIDAuMCAoMjkvMDQvMjAgMjM6Mjg6MDApXSAobWluaW11bSA1IGRhdGFwb2ludHMgZm9yIE9LIC0+IEFMQVJNIHRyYW5zaXRpb24pLlwiLFwiU3RhdGVDaGFuZ2VUaW1lXCI6XCIyMDIwLTA0LTI5VDIzOjMzOjM0Ljg3NiswMDAwXCIsXCJSZWdpb25cIjpcIlVTIFdlc3QgKE9yZWdvbilcIixcIkFsYXJtQXJuXCI6XCJ0ZXN0LWFyblwiLFwiT2xkU3RhdGVWYWx1ZVwiOlwiSU5TVUZGSUNJRU5UX0RBVEFcIixcIlRyaWdnZXJcIjp7XCJQZXJpb2RcIjo2MCxcIkV2YWx1YXRpb25QZXJpb2RzXCI6NSxcIkNvbXBhcmlzb25PcGVyYXRvclwiOlwiTGVzc1RoYW5UaHJlc2hvbGRcIixcIlRocmVzaG9sZFwiOjY1LjAsXCJUcmVhdE1pc3NpbmdEYXRhXCI6XCItIFRyZWF0TWlzc2luZ0RhdGE6ICAgICAgICAgICAgICAgICAgICBtaXNzaW5nXCIsXCJFdmFsdWF0ZUxvd1NhbXBsZUNvdW50UGVyY2VudGlsZVwiOlwiXCIsXCJNZXRyaWNzXCI6W3tcIkV4cHJlc3Npb25cIjpcIjEwMCooaGVhbHRoeUhvc3RDb3VudC9mbGVldENhcGFjaXR5KVwiLFwiSWRcIjpcImV4cHJfMVwiLFwiUmV0dXJuRGF0YVwiOnRydWV9LHtcIklkXCI6XCJoZWFsdGh5SG9zdENvdW50XCIsXCJMYWJlbFwiOlwiSGVhbHRoeUhvc3RDb3VudFwiLFwiTWV0cmljU3RhdFwiOntcIk1ldHJpY1wiOntcIkRpbWVuc2lvbnNcIjpbe1widmFsdWVcIjpcInRlc3RUYXJnZXRHcm91cFwiLFwibmFtZVwiOlwiVGFyZ2V0R3JvdXBcIn0se1widmFsdWVcIjpcInRlc3RMb2FkQmFsYW5jZXJcIixcIm5hbWVcIjpcIkxvYWRCYWxhbmNlclwifV0sXCJNZXRyaWNOYW1lXCI6XCJIZWFsdGh5SG9zdENvdW50XCIsXCJOYW1lc3BhY2VcIjpcIkFXUy9OZXR3b3JrRUxCXCJ9LFwiUGVyaW9kXCI6NjAsXCJTdGF0XCI6XCJBdmVyYWdlXCJ9LFwiUmV0dXJuRGF0YVwiOmZhbHNlfSx7XCJJZFwiOlwiZmxlZXRDYXBhY2l0eVwiLFwiTGFiZWxcIjpcIkdyb3VwRGVzaXJlZENhcGFjaXR5XCIsXCJNZXRyaWNTdGF0XCI6e1wiTWV0cmljXCI6e1wiRGltZW5cIjpbe1widmFsdWVcIjpcInRlc3RGbGVldElkMlwiLFwibmFtZVwiOlwiQXV0b1NjYWxpbmdHcm91cE5hbWVcIn1dLFwiTWV0cmljTmFtZVwiOlwiR3JvdXBEZXNpcmVkQ2FwYWNpdHlcIixcIk5hbWVzcGFjZVwiOlwiQVdTL0F1dG9TY2FsaW5nXCJ9LFwiUGVyaW9kXCI6NjAsXCJTdGF0XCI6XCJBdmVyYWdlXCJ9LFwiUmV0dXJuRGF0YVwiOmZhbHNlfV19fSc7XG5cbiAgKGF3YWl0IGxhbWJkYUNvZGUuaGFuZGxlcihzdWNjZXNzRXZlbnRTaW5nbGUsIGNvbnRleHQpKTtcblxuICAvLyBUSEVOXG4gIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQodXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xuXG4gIC8vIFdIRU5cbiAgc3VjY2Vzc0V2ZW50U2luZ2xlLlJlY29yZHNbMF0uU25zLk1lc3NhZ2UgPSAne1wiQWxhcm1OYW1lXCI6XCJ0ZXN0QWxhcm1cIixcIkFsYXJtRGVzY3JpcHRpb25cIjpudWxsLFwiTmV3U3RhdGVWYWx1ZVwiOlwiQUxBUk1cIixcIk5ld1N0YXRlUmVhc29uXCI6XCJUaHJlc2hvbGQgQ3Jvc3NlZDogNSBvdXQgb2YgdGhlIGxhc3QgNSBkYXRhcG9pbnRzIHdlcmUgbGVzcyB0aGFuIHRoZSB0aHJlc2hvbGQgKDY1LjApLiBUaGUgbW9zdCByZWNlbnQgZGF0YXBvaW50cyB3aGljaCBjcm9zc2VkIHRoZSB0aHJlc2hvbGQ6IFswLjAgKDI5LzA0LzIwIDIzOjMyOjAwKSwgMC4wICgyOS8wNC8yMCAyMzozMTowMCksIDAuMCAoMjkvMDQvMjAgMjM6MzA6MDApLCAwLjAgKDI5LzA0LzIwIDIzOjI5OjAwKSwgMC4wICgyOS8wNC8yMCAyMzoyODowMCldIChtaW5pbXVtIDUgZGF0YXBvaW50cyBmb3IgT0sgLT4gQUxBUk0gdHJhbnNpdGlvbikuXCIsXCJTdGF0ZUNoYW5nZVRpbWVcIjpcIjIwMjAtMDQtMjlUMjM6MzM6MzQuODc2KzAwMDBcIixcIlJlZ2lvblwiOlwiVVMgV2VzdCAoT3JlZ29uKVwiLFwiQWxhcm1Bcm5cIjpcInRlc3QtYXJuXCIsXCJPbGRTdGF0ZVZhbHVlXCI6XCJJTlNVRkZJQ0lFTlRfREFUQVwiLFwiVHJpZ2dlclwiOntcIlBlcmlvZFwiOjYwLFwiRXZhbHVhdGlvblBlcmlvZHNcIjo1LFwiQ29tcGFyaXNvbk9wZXJhdG9yXCI6XCJMZXNzVGhhblRocmVzaG9sZFwiLFwiVGhyZXNob2xkXCI6NjUuMCxcIlRyZWF0TWlzc2luZ0RhdGFcIjpcIi0gVHJlYXRNaXNzaW5nRGF0YTogICAgICAgICAgICAgICAgICAgIG1pc3NpbmdcIixcIkV2YWx1YXRlTG93U2FtcGxlQ291bnRQZXJjZW50aWxlXCI6XCJcIixcIk1cIjpbe1wiRXhwcmVzc2lvblwiOlwiMTAwKihoZWFsdGh5SG9zdENvdW50L2ZsZWV0Q2FwYWNpdHkpXCIsXCJJZFwiOlwiZXhwcl8xXCIsXCJSZXR1cm5EYXRhXCI6dHJ1ZX0se1wiSWRcIjpcImhlYWx0aHlIb3N0Q291bnRcIixcIkxhYmVsXCI6XCJIZWFsdGh5SG9zdENvdW50XCIsXCJNZXRyaWNTdGF0XCI6e1wiTWV0cmljXCI6e1wiRGltZW5zaW9uc1wiOlt7XCJ2YWx1ZVwiOlwidGVzdFRhcmdldEdyb3VwXCIsXCJuYW1lXCI6XCJUYXJnZXRHcm91cFwifSx7XCJ2YWx1ZVwiOlwidGVzdExvYWRCYWxhbmNlclwiLFwibmFtZVwiOlwiTG9hZEJhbGFuY2VyXCJ9XSxcIk1ldHJpY05hbWVcIjpcIkhlYWx0aHlIb3N0Q291bnRcIixcIk5hbWVzcGFjZVwiOlwiQVdTL05ldHdvcmtFTEJcIn0sXCJQZXJpb2RcIjo2MCxcIlN0YXRcIjpcIkF2ZXJhZ2VcIn0sXCJSZXR1cm5EYXRhXCI6ZmFsc2V9LHtcIklkXCI6XCJmbGVldENhcGFjaXR5XCIsXCJMYWJlbFwiOlwiR3JvdXBEZXNpcmVkQ2FwYWNpdHlcIixcIk1ldHJpY1N0YXRcIjp7XCJNZXRyaWNcIjp7XCJEaW1lblwiOlt7XCJ2YWx1ZVwiOlwidGVzdEZsZWV0SWQyXCIsXCJuYW1lXCI6XCJBdXRvU2NhbGluZ0dyb3VwTmFtZVwifV0sXCJNZXRyaWNOYW1lXCI6XCJHcm91cERlc2lyZWRDYXBhY2l0eVwiLFwiTmFtZXNwYWNlXCI6XCJBV1MvQXV0b1NjYWxpbmdcIn0sXCJQZXJpb2RcIjo2MCxcIlN0YXRcIjpcIkF2ZXJhZ2VcIn0sXCJSZXR1cm5EYXRhXCI6ZmFsc2V9XX19JztcbiAgKGF3YWl0IGxhbWJkYUNvZGUuaGFuZGxlcihzdWNjZXNzRXZlbnRTaW5nbGUsIGNvbnRleHQpKTtcblxuICAvLyBUSEVOXG4gIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQodXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xuXG4gIC8vIFdIRU5cbiAgZGVsZXRlIHN1Y2Nlc3NFdmVudFNpbmdsZS5SZWNvcmRzWzBdLlNucy5NZXNzYWdlO1xuICAoYXdhaXQgbGFtYmRhQ29kZS5oYW5kbGVyKHN1Y2Nlc3NFdmVudFNpbmdsZSwgY29udGV4dCkpO1xuXG4gIC8vIFRIRU5cbiAgc2lub24uYXNzZXJ0Lm5vdENhbGxlZCh1cGRhdGVBdXRvU2NhbGluZ0dyb3VwRmFrZSk7XG5cbiAgLy8gV0hFTlxuICBzdWNjZXNzRXZlbnRTaW5nbGUuUmVjb3Jkc1swXS5TbnMuTWVzc2FnZSA9ICd7XCJBbGFybU5hbWVcIjpcInRlc3RBbGFybVwiLFwiQWxhcm1EZXNjcmlwdGlvblwiOm51bGwsXCJOZXdTdGF0ZVZhbHVlXCI6XCJBTEFSTVwiLFwiTmV3U3RhdGVSZWFzb25cIjpcIlRocmVzaG9sZCBDcm9zc2VkOiA1IG91dCBvZiB0aGUgbGFzdCA1IGRhdGFwb2ludHMgd2VyZSBsZXNzIHRoYW4gdGhlIHRocmVzaG9sZCAoNjUuMCkuIFRoZSBtb3N0IHJlY2VudCBkYXRhcG9pbnRzIHdoaWNoIGNyb3NzZWQgdGhlIHRocmVzaG9sZDogWzAuMCAoMjkvMDQvMjAgMjM6MzI6MDApLCAwLjAgKDI5LzA0LzIwIDIzOjMxOjAwKSwgMC4wICgyOS8wNC8yMCAyMzozMDowMCksIDAuMCAoMjkvMDQvMjAgMjM6Mjk6MDApLCAwLjAgKDI5LzA0LzIwIDIzOjI4OjAwKV0gKG1pbmltdW0gNSBkYXRhcG9pbnRzIGZvciBPSyAtPiBBTEFSTSB0cmFuc2l0aW9uKS5cIixcIlN0YXRlQ2hhbmdlVGltZVwiOlwiMjAyMC0wNC0yOVQyMzozMzozNC44NzYrMDAwMFwiLFwiUmVnaW9uXCI6XCJVUyBXZXN0IChPcmVnb24pXCIsXCJBbGFybUFyblwiOlwidGVzdC1hcm5cIixcIk9sZFN0YXRlVmFsdWVcIjpcIklOU1VGRklDSUVOVF9EQVRBXCIsXCJUcmlnZ2VyXCI6e1wiUGVyaW9kXCI6NjAsXCJFdmFsdWF0aW9uUGVyaW9kc1wiOjUsXCJDb21wYXJpc29uT3BlcmF0b3JcIjpcIkxlc3NUaGFuVGhyZXNob2xkXCIsXCJUaHJlc2hvbGRcIjo2NS4wLFwiVHJlYXRNaXNzaW5nRGF0YVwiOlwiLSBUcmVhdE1pc3NpbmdEYXRhOiAgICAgICAgICAgICAgICAgICAgbWlzc2luZ1wiLFwiRXZhbHVhdGVMb3dTYW1wbGVDb3VudFBlcmNlbnRpbGVcIjpcIlwiLFwiTWV0cmljc1wiOlt7XCJFeHByZXNzaW9uXCI6XCIxMDAqKGhlYWx0aHlIb3N0Q291bnQvZmxlZXRDYXBhY2l0eSlcIixcIklkXCI6XCJleHByXzFcIixcIlJldHVybkRhdGFcIjp0cnVlfSx7XCJJZFwiOlwiaGVhbHRoeUhvc3RDb3VudFwiLFwiTGFiZWxcIjpcIkhlYWx0aHlIb3N0Q291bnRcIixcIk1ldHJpY1N0YXRcIjp7XCJNZXRyaWNcIjp7XCJEaW1lbnNpb25zXCI6W3tcInZhbHVlXCI6XCJ0ZXN0VGFyZ2V0R3JvdXBcIixcIm5hbWVcIjpcIlRhcmdldEdyb3VwXCJ9LHtcInZhbHVlXCI6XCJ0ZXN0TG9hZEJhbGFuY2VyXCIsXCJuYW1lXCI6XCJMb2FkQmFsYW5jZXJcIn1dLFwiTWV0cmljTmFtZVwiOlwiSGVhbHRoeUhvc3RDb3VudFwiLFwiTmFtZXNwYWNlXCI6XCJBV1MvTmV0d29ya0VMQlwifSxcIlBlcmlvZFwiOjYwLFwiU3RhdFwiOlwiQXZlcmFnZVwifSxcIlJldHVybkRhdGFcIjpmYWxzZX0se1wiSWRcIjpcImVldENhcGFjaXR5XCIsXCJMYWJlbFwiOlwiR3JvdXBEZXNpcmVkQ2FwYWNpdHlcIixcIk1ldHJpY1N0YXRcIjp7XCJNZXRyaWNcIjp7XCJEaW1lbnNpb25zXCI6W3tcInZhbHVlXCI6XCJ0ZXN0RmxlZXRJZFwiLFwibmFtZVwiOlwiQXV0b1NjYWxpbmdHcm91cE5hbWVcIn1dLFwiTWV0cmljTmFtZVwiOlwiR3JvdXBEZXNpcmVkQ2FwYWNpdHlcIixcIk5hbWVzcGFjZVwiOlwiQVdTL0F1dG9TY2FsaW5nXCJ9LFwiUGVyaW9kXCI6NjAsXCJTdGF0XCI6XCJBdmVyYWdlXCJ9LFwiUmV0dXJuRGF0YVwiOmZhbHNlfV19fSc7XG5cbiAgKGF3YWl0IGxhbWJkYUNvZGUuaGFuZGxlcihzdWNjZXNzRXZlbnRTaW5nbGUsIGNvbnRleHQpKTtcblxuICAvLyBUSEVOXG4gIHNpbm9uLmFzc2VydC5ub3RDYWxsZWQodXBkYXRlQXV0b1NjYWxpbmdHcm91cEZha2UpO1xufSk7Il19