"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const imported_acm_certificate_1 = require("../lib/imported-acm-certificate");
const x509_certificate_1 = require("../lib/x509-certificate");
test('Import cert', () => {
    const stack = new core_1.Stack(undefined, 'Stack', { env: { region: 'us-west-2' } });
    const secretCert = new x509_certificate_1.X509CertificatePem(stack, 'Pem', {
        subject: { cn: 'Server' },
    });
    const certPassphraseID = stack.getLogicalId(secretCert.passphrase.node.defaultChild);
    new imported_acm_certificate_1.ImportedAcmCertificate(stack, 'AcmCert', {
        cert: secretCert.cert,
        certChain: secretCert.certChain,
        key: secretCert.key,
        passphrase: secretCert.passphrase,
    });
    assert_1.expect(stack).to(assert_1.haveResourceLike('Custom::RFDK_AcmImportedCertificate', {
        X509CertificatePem: {
            Cert: {
                'Fn::GetAtt': [
                    'Pem',
                    'Cert',
                ],
            },
            Key: {
                'Fn::GetAtt': [
                    'Pem',
                    'Key',
                ],
            },
            Passphrase: {
                Ref: certPassphraseID,
            },
            CertChain: '',
        },
        Tags: [
            {
                Key: 'AcmCertImport-F4E2ABF9',
                Value: 'f4e2abf974443234fdb095fafcfa9ee2',
            },
            {
                Key: 'Name',
                Value: 'f4e2abf974443234fdb095fafcfa9ee2',
            },
        ],
    }));
    assert_1.expect(stack).to(assert_1.countResources('AWS::DynamoDB::Table', 2));
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::IAM::Policy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        'dynamodb:BatchGetItem',
                        'dynamodb:GetRecords',
                        'dynamodb:GetShardIterator',
                        'dynamodb:Query',
                        'dynamodb:GetItem',
                        'dynamodb:Scan',
                        'dynamodb:BatchWriteItem',
                        'dynamodb:PutItem',
                        'dynamodb:UpdateItem',
                        'dynamodb:DeleteItem',
                    ],
                },
                {
                    Action: 'dynamodb:DescribeTable',
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'Pem',
                            'Cert',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        'Fn::GetAtt': [
                            'Pem',
                            'Key',
                        ],
                    },
                },
                {
                    Action: [
                        'secretsmanager:GetSecretValue',
                        'secretsmanager:DescribeSecret',
                    ],
                    Resource: {
                        Ref: certPassphraseID,
                    },
                },
                {
                    Action: [
                        'acm:AddTagsToCertificate',
                        'acm:ImportCertificate',
                    ],
                    Condition: {
                        StringEquals: {
                            'aws:RequestTag/AcmCertImport-F4E2ABF9': 'f4e2abf974443234fdb095fafcfa9ee2',
                        },
                    },
                    Resource: '*',
                },
                {
                    Action: [
                        'acm:DeleteCertificate',
                        'acm:GetCertificate',
                    ],
                    Resource: '*',
                },
            ],
        },
    }));
    // Expect Lambda for doing the cert importation to use the importCert() handler, openssl layer, and set DATABASE
    assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Function', (props, error) => {
        var _a, _b;
        if (!props.Handler || props.Handler !== 'x509-certificate.importCert') {
            error.failureReason = 'x509-certificate.importCert handler not found';
            error.resource = props.Handler;
            return false;
        }
        // Our test for the correct openssl lambda layer does not include the version, so we use a filter
        // function to do a partial match
        const filterOpensslArn = (value) => {
            return value.toString().includes('arn:aws:lambda:us-west-2:224375009292:layer:openssl-al2:');
        };
        if (!props.Layers
            || !Array.isArray(props.Layers)
            || Array.of(props.Layers).filter(filterOpensslArn).length === 0) {
            error.failureReason = 'openssl Lambda Layer missing';
            error.resource = props.Layers;
            return false;
        }
        if (!props.Environment
            || !props.Environment.Variables
            || !props.Environment.Variables.DATABASE) {
            error.failureReason = 'DATABASE environment variable not set';
            error.resource = (_b = (_a = props.Environment) === null || _a === void 0 ? void 0 : _a.Variables) === null || _b === void 0 ? void 0 : _b.DATABASE;
            return false;
        }
        return true;
    }));
});
//# sourceMappingURL=data:application/json;base64,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