"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
let app;
let infraStack;
let hmStack;
let wfStack;
let vpc;
let healthMonitor;
class TestMonitorableFleet extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const fleet = new aws_autoscaling_1.AutoScalingGroup(this, 'ASG', {
            instanceType: aws_ec2_1.InstanceType.of(aws_ec2_1.InstanceClass.T2, aws_ec2_1.InstanceSize.LARGE),
            machineImage: new aws_ec2_1.AmazonLinuxImage({
                generation: aws_ec2_1.AmazonLinuxGeneration.AMAZON_LINUX_2,
            }),
            vpc: props.vpc,
            minCapacity: props.minCapacity,
        });
        this.connections = new aws_ec2_1.Connections();
        this.targetCapacity = parseInt(fleet.node.defaultChild.maxSize, 10);
        this.targetScope = this;
        this.targetToMonitor = fleet;
        this.targetCapacityMetric = new aws_cloudwatch_1.Metric({
            namespace: 'AWS/AutoScaling',
            metricName: 'GroupDesiredCapacity',
            dimensions: {
                AutoScalingGroupName: fleet.autoScalingGroupName,
            },
            label: 'GroupDesiredCapacity',
        });
        this.targetUpdatePolicy = new aws_iam_1.Policy(this, 'ASGUpdatePolicy', {
            statements: [new aws_iam_1.PolicyStatement({
                    actions: ['autoscaling:UpdateAutoScalingGroup'],
                    resources: [fleet.autoScalingGroupArn],
                })],
        });
    }
}
beforeEach(() => {
    app = new core_1.App();
    infraStack = new core_1.Stack(app, 'infraStack', {
        env: {
            region: 'us-east-1',
        },
    });
    hmStack = new core_1.Stack(app, 'hmStack', {
        env: {
            region: 'us-east-1',
        },
    });
    wfStack = new core_1.Stack(app, 'wfStack', {
        env: {
            region: 'us-east-1',
        },
    });
    vpc = new aws_ec2_1.Vpc(infraStack, 'VPC');
});
test('validating default health monitor properties', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
        vpc,
    });
    // THEN
    assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
    assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::KMS::Key', {
        KeyPolicy: {
            Statement: [
                {
                    Action: 'kms:*',
                    Effect: 'Allow',
                    Principal: {
                        AWS: {
                            'Fn::Join': [
                                '',
                                [
                                    'arn:',
                                    {
                                        Ref: 'AWS::Partition',
                                    },
                                    ':iam::',
                                    {
                                        Ref: 'AWS::AccountId',
                                    },
                                    ':root',
                                ],
                            ],
                        },
                    },
                    Resource: '*',
                },
                {
                    Action: [
                        'kms:Decrypt',
                        'kms:GenerateDataKey',
                    ],
                    Effect: 'Allow',
                    Principal: {
                        Service: 'cloudwatch.amazonaws.com',
                    },
                    Resource: '*',
                },
            ],
        },
        Description: `This key is used to encrypt SNS messages for ${healthMonitor.node.uniqueId}.`,
        EnableKeyRotation: true,
    }));
    assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::TopicPolicy', {
        PolicyDocument: {
            Statement: [
                {
                    Action: 'sns:Publish',
                    Effect: 'Allow',
                    Principal: {
                        Service: 'cloudwatch.amazonaws.com',
                    },
                    Resource: {
                        Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
                    },
                    Sid: '0',
                },
            ],
        },
        Topics: [
            {
                Ref: hmStack.getLogicalId(healthMonitor.unhealthyFleetActionTopic.node.defaultChild),
            },
        ],
    }));
    assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
        KmsMasterKeyId: {
            Ref: `${hmStack.getLogicalId(healthMonitor.node.findChild('SNSEncryptionKey').node.defaultChild)}`,
        },
    }));
    assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
    assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
        Protocol: 'lambda',
        TopicArn: {
            Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
        },
        Endpoint: {
            'Fn::GetAtt': [
                'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                'Arn',
            ],
        },
    }));
});
test('validating health monitor properties while passing a key', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
        vpc,
        encryptionKey: aws_kms_1.Key.fromKeyArn(hmStack, 'importedKey', 'arn:aws:kms:us-west-2:123456789012:key/testarn'),
    });
    // THEN
    assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::ElasticLoadBalancingV2::LoadBalancer'));
    assert_1.expect(hmStack).notTo(assert_1.haveResource('AWS::KMS::Key'));
    assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Topic', {
        KmsMasterKeyId: 'testarn',
    }));
    assert_1.expect(hmStack).to(assert_1.haveResource('AWS::Lambda::Function'));
    assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::SNS::Subscription', {
        Protocol: 'lambda',
        TopicArn: {
            Ref: `${infraStack.getLogicalId(healthMonitor.node.findChild('UnhealthyFleetTopic').node.defaultChild)}`,
        },
        Endpoint: {
            'Fn::GetAtt': [
                'unhealthyFleetTermination28bccf6aaa76478c9239e2f5bcc0254c8C612A5E',
                'Arn',
            ],
        },
    }));
});
test('validating the target with default health config', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
        vpc,
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
    });
    healthMonitor.registerFleet(fleet, {});
    // THEN
    assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
    assert_1.expect(hmStack).notTo((assert_1.haveResourceLike('AWS::EC2::SecurityGroup', {
        SecurityGroupIngress: assert_1.arrayWith(assert_1.deepObjectLike({
            CidrIp: '0.0.0.0/0',
            FromPort: 8081,
            IpProtocol: 'tcp',
            ToPort: 8081,
        })),
    })));
    assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        HealthCheckIntervalSeconds: 300,
        HealthCheckPort: '8081',
        HealthCheckProtocol: 'HTTP',
        Port: 8081,
        Protocol: 'HTTP',
        TargetType: 'instance',
    }));
    assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
});
test('validating the target with custom health config', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
        vpc,
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
    });
    healthMonitor.registerFleet(fleet, {
        port: 7171,
    });
    // THEN
    assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
    assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        HealthCheckIntervalSeconds: 300,
        HealthCheckPort: '7171',
        HealthCheckProtocol: 'HTTP',
        Port: 8081,
        Protocol: 'HTTP',
        TargetType: 'instance',
    }));
    assert_1.expect(wfStack).to(assert_1.haveResource('AWS::CloudWatch::Alarm'));
});
test('2 ASG gets registered to same LB', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor', {
        vpc,
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
    });
    healthMonitor.registerFleet(fleet, { port: 7171 });
    const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
        vpc,
    });
    healthMonitor.registerFleet(fleet2, { port: 7171 });
    // THEN
    assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 1, {
        LoadBalancerAttributes: [
            {
                Key: 'deletion_protection.enabled',
                Value: 'true',
            },
        ],
        Scheme: 'internal',
    }));
    assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
    assert_1.expect(wfStack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener'));
    assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        HealthCheckIntervalSeconds: 300,
        HealthCheckPort: '7171',
        HealthCheckProtocol: 'HTTP',
        Port: 8081,
        Protocol: 'HTTP',
        TargetType: 'instance',
    }));
    assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
        ComparisonOperator: 'GreaterThanThreshold',
        EvaluationPeriods: 8,
        ActionsEnabled: true,
        DatapointsToAlarm: 8,
        Threshold: 0,
        TreatMissingData: 'notBreaching',
    }));
    assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::CloudWatch::Alarm', {
        ComparisonOperator: 'GreaterThanThreshold',
        EvaluationPeriods: 1,
        ActionsEnabled: true,
        DatapointsToAlarm: 1,
        Threshold: 35,
        TreatMissingData: 'notBreaching',
    }));
});
test('validating LB target limit', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
        vpc,
        elbAccountLimits: [{
                name: 'targets-per-application-load-balancer',
                max: 50,
            }],
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
        minCapacity: 50,
    });
    healthMonitor.registerFleet(fleet, {});
    const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
        vpc,
        minCapacity: 50,
    });
    healthMonitor.registerFleet(fleet2, {});
    // THEN
    assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
        LoadBalancerAttributes: [
            {
                Key: 'deletion_protection.enabled',
                Value: 'true',
            },
        ],
        Scheme: 'internal',
        Type: 'application',
    }));
    assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
    assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Port: 8081,
        Protocol: 'HTTP',
    }));
});
test('validating LB listener limit', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
        vpc,
        elbAccountLimits: [{
                name: 'listeners-per-application-load-balancer',
                max: 1,
            }, {
                name: 'target-groups-per-action-on-application-load-balancer',
                max: 1,
            }],
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
    });
    healthMonitor.registerFleet(fleet, {});
    const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
        vpc,
    });
    healthMonitor.registerFleet(fleet2, {});
    // THEN
    assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
        LoadBalancerAttributes: [
            {
                Key: 'deletion_protection.enabled',
                Value: 'true',
            },
        ],
        Scheme: 'internal',
        Type: 'application',
    }));
    assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
    assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Port: 8081,
        Protocol: 'HTTP',
    }));
});
test('validating target group limit per lb', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
        vpc,
        elbAccountLimits: [{
                name: 'target-groups-per-application-load-balancer',
                max: 1,
            }],
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
    });
    healthMonitor.registerFleet(fleet, {});
    const fleet2 = new TestMonitorableFleet(wfStack, 'workerFleet2', {
        vpc,
    });
    healthMonitor.registerFleet(fleet2, {});
    // THEN
    assert_1.expect(hmStack).to(assert_1.countResourcesLike('AWS::ElasticLoadBalancingV2::LoadBalancer', 2, {
        Scheme: 'internal',
        Type: 'application',
    }));
    assert_1.expect(wfStack).to(assert_1.countResources('AWS::ElasticLoadBalancingV2::Listener', 2));
    assert_1.expect(wfStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Port: 8081,
        Protocol: 'HTTP',
    }));
});
test('validating target limit exhaustion', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
        vpc,
        elbAccountLimits: [{
                name: 'targets-per-application-load-balancer',
                max: 1,
            }],
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
        minCapacity: 2,
    });
    expect(() => {
        healthMonitor.registerFleet(fleet, {});
    }).toThrowError(/AWS service limit \"targets-per-application-load-balancer\" reached. Limit: 1/);
});
test('validating deletion protection', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
        vpc,
        deletionProtection: false,
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
    });
    healthMonitor.registerFleet(fleet, {});
    // THEN
    assert_1.expect(hmStack).to(assert_1.not(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        LoadBalancerAttributes: assert_1.arrayWith({
            Key: 'deletion_protection.enabled',
            Value: 'true',
        }),
        Scheme: assert_1.ABSENT,
        Type: assert_1.ABSENT,
    })));
});
test('drop invalid http header fields enabled', () => {
    // WHEN
    healthMonitor = new lib_1.HealthMonitor(hmStack, 'healthMonitor2', {
        vpc,
    });
    const fleet = new TestMonitorableFleet(wfStack, 'workerFleet', {
        vpc,
    });
    healthMonitor.registerFleet(fleet, {});
    // THEN
    assert_1.expect(hmStack).to(assert_1.haveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        LoadBalancerAttributes: assert_1.arrayWith({
            Key: 'routing.http.drop_invalid_header_fields.enabled',
            Value: 'true',
        }),
    }));
});
//# sourceMappingURL=data:application/json;base64,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