"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_elasticloadbalancingv2_1 = require("@aws-cdk/aws-elasticloadbalancingv2");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
// A numeric CDK token (see: https://docs.aws.amazon.com/cdk/latest/guide/tokens.html#tokens_number)
const CDK_NUMERIC_TOKEN = -1.8881545897087626e+289;
function escapeTokenRegex(s) {
    // A CDK Token looks like: ${Token[TOKEN.12]}
    // This contains the regex special characters: $, {, }, [, and ]
    // Escape those for use in a regex.
    return s.replace(/[.${}[\]]/g, '\\$&');
}
describe('Endpoint', () => {
    test('accepts tokens for the port value', () => {
        // GIVEN
        const token = CDK_NUMERIC_TOKEN;
        // WHEN
        const endpoint = new lib_1.Endpoint({
            address: '127.0.0.1',
            port: token,
        });
        // THEN
        expect(endpoint.port.toRuleJson()).toEqual(expect.objectContaining({
            fromPort: token,
            toPort: token,
            ipProtocol: 'tcp',
        }));
        expect(endpoint.socketAddress).toEqual(expect.stringMatching(new RegExp(escapeTokenRegex('127.0.0.1:${Token[TOKEN.\\d+]}'))));
    });
    test('accepts valid port string numbers', () => {
        // GIVEN
        for (const port of [1, 50, 65535]) {
            // WHEN
            const endpoint = new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
            // THEN
            expect(endpoint.port.toRuleJson()).toEqual(expect.objectContaining({
                fromPort: port,
                toPort: port,
                ipProtocol: 'tcp',
            }));
            expect(endpoint.socketAddress).toBe(`127.0.0.1:${port}`);
        }
    });
    test('throws an exception for port numbers below the minimum', () => {
        // GIVEN
        const port = 0;
        // WHEN
        function createInvalidEnpoint() {
            new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
        }
        // THEN
        expect(createInvalidEnpoint).toThrow();
    });
    test('throws an exception for port numbers above the maximum', () => {
        // GIVEN
        const port = 65536;
        // WHEN
        function createInvalidEnpoint() {
            new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
        }
        // THEN
        expect(createInvalidEnpoint).toThrow();
    });
    test('throws an exception for floating-point port numbers', () => {
        // GIVEN
        const port = 1.5;
        // WHEN
        function createInvalidEnpoint() {
            new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
        }
        // THEN
        expect(createInvalidEnpoint).toThrow();
    });
    describe('.portAsString()', () => {
        test('converts port tokens to string tokens', () => {
            // GIVEN
            const port = CDK_NUMERIC_TOKEN;
            const endpoint = new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
            // WHEN
            const result = endpoint.portAsString();
            // THEN
            // Should return a string token
            expect(core_1.Token.isUnresolved(result)).toBeTruthy();
            // It should not just be the string representation of the numeric token
            expect(result).not.toBe(port.toString());
        });
        test('converts resolved port numbers to string representation', () => {
            // GIVEN
            const port = 1500;
            const endpoint = new lib_1.Endpoint({
                address: '127.0.0.1',
                port,
            });
            // WHEN
            const result = endpoint.portAsString();
            // THEN
            expect(result).toBe(port.toString());
        });
    });
});
describe('ApplicationEndpoint', () => {
    test('uses TCP transport', () => {
        // WHEN
        const endpoint = new lib_1.ApplicationEndpoint({
            address: '127.0.0.1',
            port: 80,
        });
        // THEN
        expect(endpoint.protocol).toBe(aws_ec2_1.Protocol.TCP);
    });
    test('defaults to https', () => {
        // WHEN
        const endpoint = new lib_1.ApplicationEndpoint({
            address: '127.0.0.1',
            port: 80,
        });
        // THEN
        expect(endpoint.applicationProtocol).toBe(aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS);
    });
    test.each([
        [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTP],
        [aws_elasticloadbalancingv2_1.ApplicationProtocol.HTTPS],
    ])('sets protocol to %p', (protocol) => {
        // WHEN
        const endpoint = new lib_1.ApplicationEndpoint({
            address: '127.0.0.1',
            port: 80,
            protocol,
        });
        // THEN
        expect(endpoint.applicationProtocol).toBe(protocol);
    });
});
describe('ConnectableApplicationEndpoint', () => {
    test('Is connectable', () => {
        // WHEN
        const stack = new core_1.Stack();
        const vpc = new aws_ec2_1.Vpc(stack, 'VPC');
        const sg1 = new aws_ec2_1.SecurityGroup(stack, 'SomeSecurityGroup', { vpc, allowAllOutbound: false });
        const somethingConnectable = new SomethingConnectable(new aws_ec2_1.Connections({ securityGroups: [sg1] }));
        const securityGroup = aws_ec2_1.SecurityGroup.fromSecurityGroupId(stack, 'ImportedSG', 'sg-12345');
        const endpoint = new lib_1.ConnectableApplicationEndpoint({
            address: '127.0.0.1',
            port: 80,
            connections: new aws_ec2_1.Connections({ securityGroups: [securityGroup] }),
        });
        // WHEN
        somethingConnectable.connections.allowTo(endpoint, aws_ec2_1.Port.tcp(80), 'Connecting to endpoint');
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress'));
    });
});
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
//# sourceMappingURL=data:application/json;base64,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