"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.X509CertificatePkcs12 = exports.X509CertificatePem = void 0;
const crypto = require("crypto");
const path_1 = require("path");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_secretsmanager_1 = require("@aws-cdk/aws-secretsmanager");
const core_1 = require("@aws-cdk/core");
const lambdaLayerVersionArns_1 = require("../lambdas/lambdaLayerVersionArns");
class X509CertificateBase extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.database = new aws_dynamodb_1.Table(this, 'Table', {
            partitionKey: { name: 'PhysicalId', type: aws_dynamodb_1.AttributeType.STRING },
            sortKey: { name: 'CustomResource', type: aws_dynamodb_1.AttributeType.STRING },
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            serverSideEncryption: true,
            billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
        });
        this.passphrase = new aws_secretsmanager_1.Secret(this, 'Passphrase', {
            description: `Passphrase for the private key of the X509Certificate ${this.node.uniqueId}`,
            encryptionKey: props.encryptionKey,
            generateSecretString: {
                excludeCharacters: '"()$\'',
                excludePunctuation: true,
                includeSpace: false,
                passwordLength: 24,
                requireEachIncludedType: true,
            },
        });
        const region = core_1.Stack.of(this).region;
        const openSslLayerName = 'openssl-al2';
        const openSslLayerArns = lambdaLayerVersionArns_1.ARNS[openSslLayerName];
        const openSslLayerArn = openSslLayerArns[region];
        const openSslLayer = aws_lambda_1.LayerVersion.fromLayerVersionArn(this, 'OpenSslLayer', openSslLayerArn);
        /*
         * We cannot make this a singleton function; doing so would create circular references in the lambda role (to sign
         * a cert we need a cert that this lambda generated).
         */
        this.lambdaFunc = new aws_lambda_1.Function(this, 'Generator', {
            description: `Used by a X509Certificate ${this.node.uniqueId} to generate certificates.`,
            code: props.lambdaCode,
            environment: {
                DATABASE: this.database.tableName,
                DEBUG: 'false',
            },
            runtime: aws_lambda_1.Runtime.NODEJS_12_X,
            layers: [openSslLayer],
            handler: props.lambdaHandler,
            timeout: core_1.Duration.seconds(30),
            logRetention: aws_logs_1.RetentionDays.ONE_WEEK,
        });
        this.database.grantReadWriteData(this.lambdaFunc);
        this.database.grant(this.lambdaFunc, 'dynamodb:DescribeTable');
        (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.grantEncrypt(this.lambdaFunc);
        this.passphrase.grantRead(this.lambdaFunc);
        const uniqueValue = crypto.createHash('md5').update(this.node.uniqueId).digest('hex');
        this.uniqueTag = new core_1.Tag(`X509SecretGrant-${uniqueValue.slice(0, 8).toUpperCase()}`, uniqueValue);
        const tagCondition = {};
        tagCondition[`secretsmanager:ResourceTag/${this.uniqueTag.key}`] = this.uniqueTag.value;
        this.lambdaFunc.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: [
                'secretsmanager:CreateSecret',
                'secretsmanager:DeleteSecret',
                'secretsmanager:TagResource',
                'secretsmanager:PutSecretValue',
            ],
            resources: ['*'],
            conditions: {
                StringEquals: tagCondition,
            },
        }));
    }
}
/**
 * A Construct that uses a Lambda to generate an X.509 certificate and then saves the certificate's components into
 * Secrets. On an update, if any properties of the construct are changed, then a new certificate will be generated.
 * When the Stack is destroyed or the Construct is removed, the Secrets will all be deleted. An X.509 certificate is
 * comprised of the certificate, a certificate chain with the chain of signing certificates (if any), and a private key
 * that is password protected by a randomly generated passphrase.
 *
 * Cost:
 * The cost of four AWS SecretsManager Secrets in the deployed region.
 * The other resources created by this construct have negligible ongoing costs.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the Custom Resource.
 * 2) Secrets - 4 in total, for the certificate, it's private key, the passphrase to the key, and the cert chain.
 * 3) Lambda Function, with role - Used to create/update/delete the Custom Resource
 *
 * @ResourcesDeployed
 */
class X509CertificatePem extends X509CertificateBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.generate',
            encryptionKey: props.encryptionKey,
        });
        (_a = props.signingCertificate) === null || _a === void 0 ? void 0 : _a.cert.grantRead(this.lambdaFunc);
        (_b = props.signingCertificate) === null || _b === void 0 ? void 0 : _b.key.grantRead(this.lambdaFunc);
        (_c = props.signingCertificate) === null || _c === void 0 ? void 0 : _c.passphrase.grantRead(this.lambdaFunc);
        (_e = (_d = props.signingCertificate) === null || _d === void 0 ? void 0 : _d.certChain) === null || _e === void 0 ? void 0 : _e.grantRead(this.lambdaFunc);
        const signingCertificate = props.signingCertificate
            ? {
                Cert: props.signingCertificate.cert.secretArn,
                Key: props.signingCertificate.key.secretArn,
                Passphrase: props.signingCertificate.passphrase.secretArn,
                CertChain: props.signingCertificate.certChain ? props.signingCertificate.certChain.secretArn : '',
            }
            : undefined;
        const properties = {
            DistinguishedName: {
                CN: props.subject.cn,
                O: (_f = props.subject.o) !== null && _f !== void 0 ? _f : 'AWS',
                OU: (_g = props.subject.ou) !== null && _g !== void 0 ? _g : 'Thinkbox',
            },
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_h = props.encryptionKey) === null || _h === void 0 ? void 0 : _h.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            SigningCertificate: signingCertificate,
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509Generator',
        });
        if (this.lambdaFunc.role) {
            // There's a race on update where this resource could execute before the role has updated.
            resource.node.addDependency(this.lambdaFunc.role);
        }
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
        this.key = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Key', {
            secretArn: core_1.Token.asString(resource.getAtt('Key')),
            encryptionKey: props.encryptionKey,
        });
        // We'll only have a chain if we used a ca to sign this cert. We cannot check for certChainResource being an empty
        // string because it is an unresolved token at this point.
        if (signingCertificate) {
            const certChainResource = resource.getAtt('CertChain');
            this.certChain = certChainResource
                ? aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'CertChain', {
                    secretArn: core_1.Token.asString(certChainResource),
                    encryptionKey: props.encryptionKey,
                })
                : undefined;
        }
    }
    /**
     * Grant read permissions for the certificate
     */
    grantCertRead(grantee) {
        var _a;
        const result = this.cert.grantRead(grantee);
        (_a = this.certChain) === null || _a === void 0 ? void 0 : _a.grantRead(grantee);
        return result;
    }
    /**
     * Grant read permissions for the certificate, key, and passphrase
     */
    grantFullRead(grantee) {
        var _a;
        const result = this.cert.grantRead(grantee);
        (_a = this.certChain) === null || _a === void 0 ? void 0 : _a.grantRead(grantee);
        this.key.grantRead(grantee);
        this.passphrase.grantRead(grantee);
        return result;
    }
}
exports.X509CertificatePem = X509CertificatePem;
/**
 * This Construct will generate a PKCS #12 file from an X.509 certificate in PEM format. The PEM certificate must be
 * provided through an instance of the X509CertificatePem Construct. A Lambda Function is used to do the conversion and
 * the result is stored in a Secret. The PKCS #12 file is password protected with a passphrase that is randomly
 * generated and stored in a Secret.
 *
 * Resources Deployed
 * ------------------------
 * 1) DynamoDB Table - Used for tracking resources created by the CustomResource.
 * 2) Secrets - 2 in total, The binary of the PKCS #12 certificate and its passphrase.
 * 3) Lambda Function, with role - Used to create/update/delete the CustomResource.
 *
 * @ResourcesDeployed
 */
class X509CertificatePkcs12 extends X509CertificateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            lambdaCode: aws_lambda_1.Code.fromAsset(path_1.join(__dirname, '..', 'lambdas', 'nodejs')),
            lambdaHandler: 'x509-certificate.convert',
            encryptionKey: props.encryptionKey,
        });
        props.sourceCertificate.grantFullRead(this.lambdaFunc);
        const properties = {
            Passphrase: this.passphrase.secretArn,
            Secret: {
                NamePrefix: this.node.path,
                Description: this.node.path,
                EncryptionKey: (_a = props.encryptionKey) === null || _a === void 0 ? void 0 : _a.keyArn,
                Tags: [
                    {
                        Key: this.uniqueTag.key,
                        Value: this.uniqueTag.value,
                    },
                ],
            },
            Certificate: {
                Cert: props.sourceCertificate.cert.secretArn,
                CertChain: props.sourceCertificate.certChain ? props.sourceCertificate.certChain.secretArn : '',
                Key: props.sourceCertificate.key.secretArn,
                Passphrase: props.sourceCertificate.passphrase.secretArn,
            },
        };
        const resource = new core_1.CustomResource(this, 'Default', {
            serviceToken: this.lambdaFunc.functionArn,
            properties,
            resourceType: 'Custom::RFDK_X509_PKCS12',
        });
        this.cert = aws_secretsmanager_1.Secret.fromSecretAttributes(this, 'Cert', {
            secretArn: core_1.Token.asString(resource.getAtt('Cert')),
            encryptionKey: props.encryptionKey,
        });
    }
}
exports.X509CertificatePkcs12 = X509CertificatePkcs12;
//# sourceMappingURL=data:application/json;base64,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