"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.StaticPrivateIpServer = void 0;
const path = require("path");
const aws_autoscaling_1 = require("@aws-cdk/aws-autoscaling");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_kms_1 = require("@aws-cdk/aws-kms");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_sns_1 = require("@aws-cdk/aws-sns");
const aws_sns_subscriptions_1 = require("@aws-cdk/aws-sns-subscriptions");
const core_1 = require("@aws-cdk/core");
/**
 * This construct provides a single instance, provided by an Auto Scaling Group (ASG), that
 * has an attached Elastic Network Interface (ENI) that is providing a private ip address.
 * This ENI is automatically re-attached to the instance if the instance is replaced
 * by the ASG.
 *
 * The ENI provides an unchanging private IP address that can always be used to connect
 * to the instance regardless of how many times the instance has been replaced. Furthermore,
 * the ENI has a MAC address that remains unchanged unless the ENI is destroyed.
 *
 * Essentially, this provides an instance with an unchanging private IP address that will
 * automatically recover from termination. This instance is suitable for use as an application server,
 * such as a license server, that must always be reachable by the same IP address.
 *
 * Resources Deployed
 * ------------------------
 * 1) Auto Scaling Group (ASG) with min & max capacity of 1 instance;
 * 2) Elastic Network Interface (ENI);
 * 3) Security Group for the ASG;
 * 4) Instance Role and corresponding IAM Policy
 * 5) SNS Topic & Role for instance-launch lifecycle events -- max one of each per stack; and
 * 6) Lambda function, with role, to attach the ENI in response to instance-launch lifecycle events -- max one per stack.
 *
 * @ResourcesDeployed
 */
class StaticPrivateIpServer extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        if (subnets.length === 0) {
            throw new Error(`Did not find any subnets matching ${JSON.stringify(props.vpcSubnets)}. Please use a different selection.`);
        }
        const subnet = subnets[0];
        if (props.resourceSignalTimeout && props.resourceSignalTimeout.toSeconds() > (12 * 60 * 60)) {
            throw new Error('Resource signal timeout cannot exceed 12 hours.');
        }
        this.autoscalingGroup = new aws_autoscaling_1.AutoScalingGroup(this, 'Asg', {
            minCapacity: 1,
            maxCapacity: 1,
            vpc: props.vpc,
            instanceType: props.instanceType,
            machineImage: props.machineImage,
            vpcSubnets: { subnets: [subnet] },
            blockDevices: props.blockDevices,
            keyName: props.keyName,
            resourceSignalCount: props.resourceSignalTimeout ? 1 : undefined,
            resourceSignalTimeout: props.resourceSignalTimeout,
            role: props.role,
            securityGroup: props.securityGroup,
            userData: props.userData,
        });
        this.connections = this.autoscalingGroup.connections;
        this.grantPrincipal = this.autoscalingGroup.grantPrincipal;
        this.osType = this.autoscalingGroup.osType;
        this.role = this.autoscalingGroup.role;
        this.userData = this.autoscalingGroup.userData;
        const scopePath = this.node.scopes.map(construct => construct.node.id).slice(1); // Slice to remove the unnamed <root> scope.
        const eni = new aws_ec2_1.CfnNetworkInterface(this, 'Eni', {
            subnetId: subnet.subnetId,
            description: `Static ENI for ${scopePath.join('/')}`,
            groupSet: core_1.Lazy.listValue({ produce: () => this.connections.securityGroups.map(sg => sg.securityGroupId) }),
            privateIpAddress: props.privateIpAddress,
        });
        this.privateIpAddress = eni.attrPrimaryPrivateIpAddress;
        // We need to be sure that the ENI is created before the instance would be brought up; otherwise, we cannot attach it.
        this.autoscalingGroup.node.defaultChild.addDependsOn(eni);
        this.attachEniLifecyleTarget(eni);
        this.node.defaultChild = this.autoscalingGroup.node.defaultChild;
    }
    /**
     * Set up an instance launch lifecycle action that will attach the eni to the single instance
     * in this construct's AutoScalingGroup when a new instance is launched.
     */
    attachEniLifecyleTarget(eni) {
        // Note: The design of AutoScalingGroup life cycle notifications in CDK v1.49.1 is such that
        // using the provided AutoScalingGroup.addLifecycleHook() will result in a setup that misses
        // launch notifications for instances created when the ASG is created. This is because
        // it uses the separate CfnLifecycleHook resource to do it, and that resource references the
        // ASG ARN; i.e. it must be created after the ASG has an ARN... thus it can miss instance launches
        // when the ASG is first created.
        //
        // We work around this by using an escape-hatch to the L1 ASG to create our own notification from scratch.
        const eventHandler = this.setupLifecycleEventHandlerFunction();
        const { topic, role } = this.setupLifecycleNotificationTopic(eventHandler);
        // Ensure no race conditions that might prevent the lambda from being able to perform its required functions by making
        // the ASG depend on the creation of the SNS Subscription.
        // Note: The topic subscriptions are children of the lambda, and are given an id equal to the Topic's id.
        this.autoscalingGroup.node.defaultChild.node.addDependency(eventHandler.node.findChild(topic.node.id));
        this.autoscalingGroup.node.defaultChild.lifecycleHookSpecificationList = [
            {
                defaultResult: aws_autoscaling_1.DefaultResult.ABANDON,
                heartbeatTimeout: 120,
                lifecycleHookName: 'NewStaticPrivateIpServer',
                lifecycleTransition: aws_autoscaling_1.LifecycleTransition.INSTANCE_LAUNCHING,
                notificationTargetArn: topic.topicArn,
                roleArn: role.roleArn,
                notificationMetadata: JSON.stringify({ eniId: eni.ref }),
            },
        ];
    }
    /**
     * Create, or fetch, the lambda function that will process instance-start lifecycle events from this construct.
     */
    setupLifecycleEventHandlerFunction() {
        const stack = core_1.Stack.of(this);
        // The SingletonFunction does not tell us when it's newly created vs. finding a pre-existing
        // one. So, we do our own singleton Function so that we know when it's the first creation, and, thus,
        // we must attach one-time permissions.
        const functionUniqueId = 'AttachEniToInstance' + this.removeHyphens('83a5dca5-db54-4aa4-85d2-8d419cdf85ce');
        let singletonPreExists = true;
        let eventHandler = stack.node.tryFindChild(functionUniqueId);
        if (!eventHandler) {
            const handlerCode = aws_lambda_1.Code.fromAsset(path.join(__dirname, '..', 'lambdas', 'nodejs', 'asg-attach-eni'), {
                exclude: ['**/*', '!index*'],
            });
            eventHandler = new aws_lambda_1.Function(stack, functionUniqueId, {
                code: handlerCode,
                handler: 'index.handler',
                runtime: aws_lambda_1.Runtime.NODEJS_12_X,
                description: `Created by AWS-RFDK StaticPrivateIpServer to process instance launch lifecycle events in stack '${stack.stackName}'. This lambda attaches an ENI to newly launched instances.`,
                logRetention: aws_logs_1.RetentionDays.THREE_DAYS,
            });
            singletonPreExists = false;
        }
        // Note: We **cannot** reference the ASG's ARN in the lambda's policy. It would create a deadlock at deployment:
        //  Lambda policy waiting on ASG completion to get ARN
        //  -> lambda waiting on policy to be created
        //  -> ASG waiting on lambda to signal lifecycle continue for instance start
        //  -> back to the start of the cycle.
        // Instead we use resourcetags condition to limit the scope of the lambda.
        const tagKey = 'RfdkStaticPrivateIpServerGrantConditionKey';
        const tagValue = eventHandler.node.uniqueId;
        const grantCondition = {};
        grantCondition[`autoscaling:ResourceTag/${tagKey}`] = tagValue;
        core_1.Tag.add(this.autoscalingGroup, tagKey, tagValue);
        // Allow the lambda to complete the lifecycle action for only tagged ASGs.
        const iamCompleteLifecycle = new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'autoscaling:CompleteLifecycleAction',
            ],
            resources: [
                `arn:${stack.partition}:autoscaling:${stack.region}:${stack.account}:autoScalingGroup:*:autoScalingGroupName/*`,
            ],
            conditions: {
                'ForAnyValue:StringEquals': grantCondition,
            },
        });
        eventHandler.role.addToPolicy(iamCompleteLifecycle);
        if (!singletonPreExists) {
            // Allow the lambda to attach the ENI to the instance that was created.
            // Referencing: https://docs.aws.amazon.com/IAM/latest/UserGuide/list_amazonec2.html
            // Last-Accessed: July 2020
            // The ec2:DescribeNetworkInterfaces, and ec2:AttachNetworkInterface operations
            // do not support conditions, and do not support resource restriction.
            // So, we only attach the policy to the lambda function once; when we first create it.
            const iamEniAttach = new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: [
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:AttachNetworkInterface',
                ],
                resources: ['*'],
            });
            eventHandler.role.addToPolicy(iamEniAttach);
        }
        return eventHandler;
    }
    /**
     * Create, or fetch, an SNS Topic to which we'll direct the ASG's instance-start lifecycle hook events. Also creates, or fetches,
     * the accompanying role that allows the lifecycle events to be published to the SNS Topic.
     * @param lambdaHandler The lambda singleton that will be processing the lifecycle events.
     * @returns { topic: Topic, role: Role }
     */
    setupLifecycleNotificationTopic(lambdaHandler) {
        const stack = core_1.Stack.of(this);
        // We only need to have a single SNS topic & subscription set up to handle lifecycle events for *all* instances of this class.
        // We have to be careful, however, to ensure that our initial setup only happens once when we first add the topic and such
        // to this stack; otherwise, we will not be able to deploy more than one of these constructs in a stack.
        const notificationRoleUniqueId = 'AttachEniNotificationRole' + this.removeHyphens('a0376ff8-248e-4534-bf42-58c6ffa4d5b4');
        const notificationTopicUniqueId = 'AttachEniNotificationTopic' + this.removeHyphens('c8b1e9a6-783c-4954-b191-204dd5e3b9e0');
        let notificationTopic = stack.node.tryFindChild(notificationTopicUniqueId);
        let notificationRole;
        if (!notificationTopic) {
            // First time creating the singleton Topic in this stack. Set it all up...
            notificationRole = new aws_iam_1.Role(stack, notificationRoleUniqueId, {
                assumedBy: new aws_iam_1.ServicePrincipal('autoscaling.amazonaws.com'),
            });
            const notificationTopicEncryptKeyUniqueId = 'SNSEncryptionKey' + this.removeHyphens('255e9e52-ad03-4ddf-8ff8-274bc10d63d1');
            const notificationTopicEncryptKey = new aws_kms_1.Key(stack, notificationTopicEncryptKeyUniqueId, {
                description: `This key is used to encrypt SNS messages for ${notificationTopicUniqueId}.`,
                enableKeyRotation: true,
                removalPolicy: core_1.RemovalPolicy.DESTROY,
                trustAccountIdentities: true,
            });
            notificationTopic = new aws_sns_1.Topic(stack, notificationTopicUniqueId, {
                displayName: `For AWS-RFDK instance-launch notifications for stack '${stack.stackName}'`,
                masterKey: notificationTopicEncryptKey,
            });
            notificationTopicEncryptKey.grant(notificationRole, 'kms:Decrypt', 'kms:GenerateDataKey');
            notificationTopic.addSubscription(new aws_sns_subscriptions_1.LambdaSubscription(lambdaHandler));
            notificationTopic.grantPublish(notificationRole);
        }
        else {
            notificationRole = stack.node.findChild(notificationRoleUniqueId);
        }
        return {
            topic: notificationTopic,
            role: notificationRole,
        };
    }
    /**
     * Convert a UUID into a string that's usable in a construct id.
     */
    removeHyphens(x) {
        return x.replace(/[-]/g, '');
    }
}
exports.StaticPrivateIpServer = StaticPrivateIpServer;
//# sourceMappingURL=data:application/json;base64,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