"use strict";
/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountableEfs = void 0;
const path = require("path");
const aws_ec2_1 = require("@aws-cdk/aws-ec2");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const mount_permissions_helper_1 = require("./mount-permissions-helper");
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto
 * an instance.
 *
 * @remark The default access point is writeable only by the root user.
 * @todo Add support for specifying an AccessPoint for the EFS filesystem to  enforce user and group information for all file system requests.
 */
class MountableEfs {
    constructor(scope, props) {
        this.scope = scope;
        this.props = props;
    }
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target, mount) {
        if (target.osType !== aws_ec2_1.OperatingSystemType.LINUX) {
            throw new Error('Target instance must be Linux.');
        }
        target.connections.allowTo(this.props.filesystem, this.props.filesystem.connections.defaultPort);
        const mountScriptAsset = this.mountAssetSingleton();
        mountScriptAsset.grantRead(target.grantPrincipal);
        const mountScript = target.userData.addS3DownloadCommand({
            bucket: mountScriptAsset.bucket,
            bucketKey: mountScriptAsset.s3ObjectKey,
        });
        const mountDir = path.posix.normalize(mount.location);
        const mountOptions = [mount_permissions_helper_1.MountPermissionsHelper.toLinuxMountOption(mount.permissions)];
        if (this.props.extraMountOptions) {
            mountOptions.push(...this.props.extraMountOptions);
        }
        const mountOptionsStr = mountOptions.join(',');
        target.userData.addCommands('TMPDIR=$(mktemp -d)', 'pushd "$TMPDIR"', `unzip ${mountScript}`, `bash ./mountEfs.sh ${this.props.filesystem.fileSystemId} ${mountDir} ${mountOptionsStr}`, 'popd', `rm -f ${mountScript}`);
    }
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     */
    mountAssetSingleton() {
        var _a;
        const stack = core_1.Stack.of(this.scope);
        const uuid = '2b31c419-5b0b-4bb8-99ad-5b2575b2c06b';
        const uniqueId = 'MountableEfsAsset' + uuid.replace(/[-]/g, '');
        return (_a = stack.node.tryFindChild(uniqueId)) !== null && _a !== void 0 ? _a : new aws_s3_assets_1.Asset(stack, uniqueId, {
            path: path.join(__dirname, '..', 'scripts', 'bash'),
            exclude: ['**/*', '!mountEfs.sh', '!metadataUtilities.sh'],
        });
    }
}
exports.MountableEfs = MountableEfs;
//# sourceMappingURL=data:application/json;base64,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