/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0
 */
import * as efs from '@aws-cdk/aws-efs';
import { Asset } from '@aws-cdk/aws-s3-assets';
import { Construct } from '@aws-cdk/core';
import { IMountableLinuxFilesystem, IMountingInstance, LinuxMountPointProps } from './mountable-filesystem';
/**
 * Properties that are required to create a {@link MountableEfs}.
 */
export interface MountableEfsProps {
    /**
     * The {@link https://docs.aws.amazon.com/cdk/api/latest/docs/@aws-cdk_aws-efs.FileSystem.html|EFS}
     * filesystem that will be mounted by the object.
     */
    readonly filesystem: efs.IFileSystem;
    /**
     * Extra NFSv4 mount options that will be added to /etc/fstab for the file system.
     * See: {@link https://www.man7.org/linux/man-pages//man5/nfs.5.html}
     *
     * The given values will be joined together into a single string by commas.
     * ex: ['soft', 'rsize=4096'] will become 'soft,rsize=4096'
     *
     * @default No extra options.
     */
    readonly extraMountOptions?: string[];
}
/**
 * This class encapsulates scripting that can be used to mount an Amazon Elastic File System onto
 * an instance.
 *
 * @remark The default access point is writeable only by the root user.
 * @todo Add support for specifying an AccessPoint for the EFS filesystem to  enforce user and group information for all file system requests.
 */
export declare class MountableEfs implements IMountableLinuxFilesystem {
    protected readonly scope: Construct;
    protected readonly props: MountableEfsProps;
    constructor(scope: Construct, props: MountableEfsProps);
    /**
     * @inheritdoc
     */
    mountToLinuxInstance(target: IMountingInstance, mount: LinuxMountPointProps): void;
    /**
     * Fetch the Asset singleton for the EFS mounting scripts, or generate it if needed.
     */
    protected mountAssetSingleton(): Asset;
}
